import 'dart:math';

import 'package:cached_network_image/cached_network_image.dart';
import 'package:flutter/material.dart';
import 'package:flutter_clx_base/flutter_clx_base.dart';
import 'package:octo_image/octo_image.dart';

class ImageWidget {
  static String domain = "";

  /// 加载网络图片widget
  static Widget loadNetImage(
    String? url, {
    String? domain,
    ImageTransformerType? type,
    double? width,
    double? height,
    BoxFit fit = BoxFit.cover,
    Widget? placeholder,
    Widget? errorWidget,
    bool signature = false,
    int? maxHeight,
    int? maxWidth,
  }) {
    var domainUrl = domain ?? ImageWidget.domain;
    if (url != null && !url.startsWith("http")) {
      url = domainUrl + url;
    }
    if (signature) {
      url =
          url != null && url.isNotEmpty ? "$url?${Random().nextDouble()}" : "";
    }
    if (TextUtil.isEmpty(url)) {
      return const SizedBox();
    }
    return OctoImage(
      // 拼接Random，修复图片地址固定，无法展示最新图片问题
      image: CachedNetworkImageProvider(url!, maxWidth: maxWidth, maxHeight: maxHeight),
      imageBuilder: MyOctoImageTransformer.getOctoImageBuilder(type),
      placeholderBuilder: ImageWidget.placeholderBuilder(placeholder: placeholder),
      errorBuilder: ImageWidget.errorBuilder(errorWidget: placeholder),
      fit: fit,
      width: width,
      height: height,
    );
  }

  /// 加载本地图片widget
  static Widget loadAssetImage(
    String? image, {
    double? width,
    double? height,
    int? cacheWidth,
    int? cacheHeight,
    BoxFit? fit,
    String format = 'png',
    Color? color,
    String? package,
  }) {
    return Image.asset(
      ImageUtils.getImgPath(image, format: format),
      height: height,
      width: width,
      cacheWidth: cacheWidth,
      cacheHeight: cacheHeight,
      fit: fit,
      color: color,
      package: package,

      /// 忽略图片语义
      excludeFromSemantics: true,
    );
  }

  //加载头像默认图片
  static Widget loadHeadPlaceHolder() {
    return ImageWidget.loadAssetImage(
      "head",);
  }
  
    
  //加载头像默认图片-方形头像
  static Widget loadHeadPlaceHolderSquare() {
    return ImageWidget.loadAssetImage(
      "head_square",);
  }

  //缓存 本地图片
  static void cacheAssetImage(
    BuildContext context,
    String image, {
    String format = 'png',
  }) {
    precacheImage(
        AssetImage(ImageUtils.getImgPath(image, format: format)), context);
  }

  //缓存 网络图片
  static void cacheNetImage(
    BuildContext context,
    String url, {
    String? domain,
  }) {
    var domainUrl = domain ?? ImageWidget.domain;
    if (url.isNotEmpty && !url.startsWith("http")) {
      url = domainUrl + url;
    }
    precacheImage(CachedNetworkImageProvider(url), context);
  }

  //返回 network imageprovider
  static ImageProvider getNetImageProvider(
    String url, {
    String? domain,
  }) {
    var domainUrl = domain ?? ImageWidget.domain;
    if (url.isNotEmpty && !url.startsWith("http")) {
      url = domainUrl + url;
    }
    return CachedNetworkImageProvider(url);
  }

  static OctoErrorBuilder errorBuilder({
    Widget? errorWidget,
  }) {
    return (context, error, stacktrace) => errorWidget != null
        ? SizedBox(
            width: double.infinity,
            height: double.infinity,
            child: errorWidget,
          )
        : const Icon(
            Icons.error_outline_outlined,
            color: Colors.grey,
          );
  }

  static OctoPlaceholderBuilder? placeholderBuilder({
    Widget? placeholder,
  }) {
    return placeholder != null
        ? (context) => SizedBox(
              width: double.infinity,
              height: double.infinity,
              child: placeholder,
            )
        : null;
  }
}

class MyOctoImageTransformer {
  static OctoImageBuilder radiusAvatar() {
    return (context, child) => Center(
          child: AspectRatio(
            aspectRatio: 1.0,
            child: ClipRRect(
              borderRadius: BorderRadius.circular(5.0),
              child: child,
            ),
          ),
        );
  }

  static OctoImageBuilder circleAvatar() {
    return (context, child) => Center(
          child: AspectRatio(
            aspectRatio: 1.0,
            child: ClipOval(
              child: child,
            ),
          ),
        );
  }

  // 根据type返回 OctoImageBuilder

  static OctoImageBuilder getOctoImageBuilder(ImageTransformerType? type) {
    switch (type) {
      case ImageTransformerType.rect:
        return radiusAvatar();
      case ImageTransformerType.circle:
        return circleAvatar();
      default:
        return (context, child) => child;
    }
  }
}

enum ImageTransformerType {
  rect, // 矩形
  circle, // 圆形
}
