import 'package:flutter/widgets.dart';
import 'package:flutter_clx_base/utils/log_utils.dart';
import 'package:intl/intl.dart';

/// 扩展函数 Iterable
extension IterableExt<E> on Iterable<E>? {
  // 获取元素
  E? getElement(int? index) {
    if (this == null || index == null || index >= this!.length || index < 0) {
      return null;
    }
    int elementIndex = 0;
    for (E element in this!) {
      if (index == elementIndex) return element;
      elementIndex++;
    }
    return null;
  }

  Iterable<T>? mapIndexed<T>(T Function(E e, int i) f) {
    var i = 0;
    return this?.map((e) => f(e, i++));
  }

  bool isNullOrEmpty() {
    return this == null || this!.isEmpty;
  }

  bool containsKey(Map<String, dynamic>? element, {String key = "id"}) {
    if (this == null) return false;
    for (E e in this!) {
      if ((e as Map?)?[key] == element?[key]) return true;
    }
    return false;
  }

}

/// 扩展函数 GlobalKey
extension GlobalKeyExt on GlobalKey {
  // 获取 widget size
  Size get getSize {
    final RenderBox? widget = currentContext?.findRenderObject() as RenderBox?;
    return widget?.size ?? Size.zero;
  }

  // 获取 widget offset
  Offset get getOffset {
    final RenderBox? widget = currentContext?.findRenderObject() as RenderBox?;
    return widget?.localToGlobal(Offset.zero) ?? Offset.zero;
  }
}

/// DateTime 扩展函数
extension DateTimeExt on DateTime {}

/// String 扩展函数
extension StringExt on String? {
  bool isNullOrEmpty() {
    return this == null || this!.isEmpty;
  }

  int toInt() {
    if (isNullOrEmpty()) {
      return 0;
    }
    try {
      return int.parse(this!);
    } catch (e) {
      return 0;
    }
  }

  int? toIntOrNull() {
    if (isNullOrEmpty()) {
      return null;
    }
    try {
      return int.parse(this!);
    } catch (e) {
      return null;
    }
  }

  double toDouble() {
    if (isNullOrEmpty()) {
      return 0.0;
    }
    try {
      return double.parse(this!);
    } catch (e) {
      return 0.0;
    }
  }

  double? toDoubleOrNull() {
    if (isNullOrEmpty()) {
      return null;
    }
    try {
      return double.parse(this!);
    } catch (e) {
      return null;
    }
  }

}

/// num 扩展函数
extension DoubleExt on num? {
  // 格式化小数两位
  String formatDouble({nullTxt = "-"}) {
    if (this == null) {
      return nullTxt;
    }
    double num;
    try {
      num = double.parse(toString());
    } catch (e) {
      loggerNoStack.i("转换异常:$e");
      return nullTxt;
    }
    if ((num.toString().length - num.toString().lastIndexOf(".") - 1) < 2) {
      //小数点后有几位小数
      return num.toStringAsFixed(2)
          .substring(0, num.toString().lastIndexOf(".") + 2 + 1)
          .toString();
    } else {
      return num.toString()
          .substring(0, num.toString().lastIndexOf(".") + 2 + 1)
          .toString();
    }
  }

  // double格式化 默认 0.00
  String formatDoubleZero() {
    return formatDouble(nullTxt: "0.00");
  }

  // int 转换 默认0
  String formatInt({nullTxt = "-"}) {
    if (this == null) {
      return nullTxt;
    }
    int num;
    try {
      num = int.parse(toString());
    } catch (e) {
      loggerNoStack.d("转换异常:$e");
      return nullTxt;
    }
    return num.toString();
  }

  // int格式化 默认 0
  String formatIntZero() {
    return formatInt(nullTxt: "0");
  }

  /// 千位分隔
  String thousandsFormatter() {
    NumberFormat format = NumberFormat.decimalPattern('en_us');
    return format.format(this ?? 0);
  }
}

/// TextEditingController 扩展
extension TextEditingControllerExt on TextEditingController? {
  setContent(String text) {
    // 设置内容 光标展示最后
    this
      ?..text = text
      ..selection = TextSelection.fromPosition(
          TextPosition(offset: this?.value.text.toString().trim().length ?? 0));
  }
}
