import 'package:easy_refresh/easy_refresh.dart';
import 'package:flutter/material.dart';
import 'package:flutter_spinkit/flutter_spinkit.dart';
export 'package:easy_refresh/easy_refresh.dart';

class BaseListWidget extends StatefulWidget {
  final RequestDataCallback requestData;
  final int pageSize;
  final ItemBuilder itemBuilder;
  final Header? header;
  final Footer? footer;
  final Widget? startLoadWidget;
  final Widget? emptyWidget;
  final BaseListController? controller;

  const BaseListWidget({
    Key? key,
    required this.requestData,
    required this.pageSize,
    required this.itemBuilder,
    this.header,
    this.footer,
    this.startLoadWidget,
    this.controller,
    this.emptyWidget,
  }) : super(key: key);

  @override
  State<BaseListWidget> createState() => _BaseListWidgetState();
}

class _BaseListWidgetState extends State<BaseListWidget> {
  late BaseListController _controller;
  final List<dynamic> _dataList = [];
  int _page = 1;
  int _count = 0;
  @override
  void initState() {
    super.initState();
    _controller = widget.controller ?? BaseListController();
  }

  @override
  Widget build(BuildContext context) {
    return EasyRefresh(
      controller: _controller.c,
      refreshOnStart: true,
      header: widget.header,
      footer: widget.footer,
      refreshOnStartHeader: BuilderHeader(
        triggerOffset: 70,
        clamping: true,
        position: IndicatorPosition.above,
        processedDuration: Duration.zero,
        builder: (ctx, state) {
          if (state.mode == IndicatorMode.inactive ||
              state.mode == IndicatorMode.done) {
            return const SizedBox();
          }
          return Container(
              padding: const EdgeInsets.only(bottom: 100),
              width: double.infinity,
              height: state.viewportDimension,
              alignment: Alignment.center,
              child: widget.startLoadWidget ??
                  SpinKitFadingCircle(
                    color: Theme.of(context).primaryColor,
                    size: 30,
                  ));
        },
      ),
      onRefresh: _onRefresh,
      onLoad: _onLoad,
      child: buildScrollView(),
    );
  }

  Widget buildScrollView() {
    return CustomScrollView(
      slivers: buildSlivers(),
    );
  }

  Future<void> _onRefresh() async {
    _page = 1;
    await widget.requestData(
        _page, widget.pageSize, _requestSuccess, _requestError);
  }

  Future<void> _onLoad() async {
    _page++;
    await widget.requestData(
        _page, widget.pageSize, _requestSuccess, _requestError);
  }

  void _requestSuccess(List? result) {
    if (!mounted) {
      return;
    }
    result ??= [];
    if (_page == 1) {
      setState(() {
        _dataList.clear();
        _dataList.addAll(result!);
        _count = _dataList.length;
      });
      _controller.finishRefresh();
      _controller.resetFooter();
    } else {
      setState(() {
        _dataList.addAll(result!);
        _count = _dataList.length;
      });
      _controller.c.finishLoad(result.length <= widget.pageSize
          ? IndicatorResult.noMore
          : IndicatorResult.success);
    }
  }

  void _requestError(r, s) {
    if (!mounted) {
      return;
    }
  }

  /// Build header.
  Header buildHeader() => EasyRefresh.defaultHeaderBuilder();

  /// Build footer.
  Footer buildFooter() => EasyRefresh.defaultFooterBuilder();

  /// Check if there is no data.
  bool get isEmpty => _count == 0;

  dynamic getItem(int index) {
    return _dataList[index];
  }

  List<Widget> buildSlivers() {
    final header = buildHeader();
    final footer = buildFooter();
    Widget? emptyWidget;
    if (isEmpty) {
      emptyWidget = widget.emptyWidget;
    }
    return [
      if (header.position == IndicatorPosition.locator)
        const HeaderLocator.sliver(),
      if (emptyWidget != null)
        SliverFillViewport(
          delegate: SliverChildBuilderDelegate(
            (context, index) {
              return emptyWidget;
            },
            childCount: 1,
          ),
        ),
      buildSliver(),
      if (footer.position == IndicatorPosition.locator)
        const FooterLocator.sliver(),
    ];
  }

  /// Build sliver.
  Widget buildSliver() {
    return SliverList(
      delegate: SliverChildBuilderDelegate(
        (context, index) {
          return buildItem(context, index, getItem(index));
        },
        childCount: _count,
      ),
    );
  }

  /// Build item widget.
  Widget buildItem(BuildContext context, int index, dynamic item) {
    return widget.itemBuilder(context, index, item);
  }
}

// 列表控制器
class BaseListController {
  late EasyRefreshController _controller;

  BaseListController() {
    _controller = EasyRefreshController(
      controlFinishRefresh: true,
      controlFinishLoad: true,
    );
  }

  void finishRefresh() {
    _controller.finishRefresh();
  }

  void finishLoad() {
    _controller.finishLoad();
  }

  void resetFooter() {
    _controller.resetFooter();
  }

  // 重新加载数据
  void reload() {
    _controller.callRefresh();
  }

  get c => _controller;
}

typedef RefreshCallback = Future<void> Function();
typedef LoadMoreCallback = Future<void> Function();
typedef SuccessCallback = Function(List list);
typedef ErrorCallback = Function(int code, String msg);
typedef RequestDataCallback = Function(
    int page, int pageSize, SuccessCallback success, ErrorCallback error);
typedef ItemBuilder = Function(
    BuildContext context, int index, dynamic item);
