import 'package:flutter/material.dart';
import 'package:flutter_clx_base/common/gaps.dart';
import 'package:flutter_clx_base/utils/keyboard_utils.dart';
import 'package:flutter_clx_base/utils/string_util.dart';
import 'package:flutter_clx_base/utils/toast_util.dart';
import 'package:keyboard_actions/keyboard_actions.dart';

class SearchAppBar extends StatefulWidget implements PreferredSizeWidget {
  final Function? onCallback;
  final String? hintText;
  final Widget? leading;
  final Color? backgroundColor;
  final Color? foregroundColor;
  final bool primary;
  final Color? searchBackgroundColor;
  final Color? inputTextColor;
  final Color? hintTextColor;
  final Color? searchTextColor;
  final Color? searchIconColor;
  final double? searchIconSize;
  final Function? onFocusListener;
  final String? rightText;
  final Function? onRightBtnCallback;

  const SearchAppBar({
    Key? key,
    this.onCallback,
    this.hintText = "请输入搜索内容",
    this.leading,
    this.backgroundColor,
    this.foregroundColor,
    this.primary = true,
    this.searchBackgroundColor,
    this.inputTextColor,
    this.hintTextColor,
    this.searchTextColor,
    this.onFocusListener,
    this.rightText,
    this.onRightBtnCallback,
    this.searchIconColor,
    this.searchIconSize,
  }) : super(key: key);

  @override
  State<SearchAppBar> createState() => _SearchAppBarState();

  @override
  Size get preferredSize => const Size(double.infinity, 55.0);
}

class _SearchAppBarState extends State<SearchAppBar> {
  final _searchInfoController = TextEditingController();
  final _searchFocusNode = FocusNode();

  @override
  void initState() {
    _searchFocusNode.addListener(() {
      if (widget.onFocusListener != null) {
        widget.onFocusListener!(_searchFocusNode.hasFocus);
      }
    });
    super.initState();
  }

  @override
  Widget build(BuildContext context) {
    return KeyboardActions(
      config:
          KeyBoardUtils.getKeyboardActionsConfig(context, [_searchFocusNode]),
      tapOutsideBehavior: TapOutsideBehavior.opaqueDismiss,
      child: AppBar(
        backgroundColor: widget.backgroundColor,
        foregroundColor: widget.foregroundColor,
        titleSpacing: 0.0,
        primary: widget.primary,
        leading: widget.leading,
        title: Row(
          children: <Widget>[
            Expanded(
              child: Container(
                decoration: BoxDecoration(
                  borderRadius: const BorderRadius.all(Radius.circular(20.0)),
                  color:
                      widget.searchBackgroundColor ?? const Color(0xFFEEEEEE),
                ),
                child: Row(
                  children: <Widget>[
                    hGap10,
                    Icon(
                      Icons.search,
                      color: widget.searchIconColor ?? Colors.grey.shade300,
                      size: widget.searchIconSize,
                    ),
                    hGap5,
                    Expanded(
                      child: TextField(
                        maxLines: 1,
                        focusNode: _searchFocusNode,
                        style: TextStyle(
                            fontSize: 14,
                            color: widget.inputTextColor ??
                                const Color(0xFF000000)),
                        textInputAction: TextInputAction.search,
                        controller: _searchInfoController,
                        onSubmitted: (value) => _search(),
                        decoration: InputDecoration(
                          contentPadding: const EdgeInsets.symmetric(
                              horizontal: 0.0, vertical: 8.0),
                          isDense: true,
                          counterText: "",
                          hintText: widget.hintText,
                          hintStyle: TextStyle(
                              fontSize: 14.0,
                              color: widget.hintTextColor ??
                                  const Color(0xFF999999)),
                          border: InputBorder.none, //去掉下划线
                          //hintStyle: TextStyles.textGrayC14
                        ),
                      ),
                    ),
                  ],
                ),
              ),
            ),
            InkWell(
              onTap: () {
                widget.onRightBtnCallback != null
                    ? widget.onRightBtnCallback?.call()
                    : _search();
              },
              child: Container(
                width: 60.0,
                height: 50.0,
                alignment: Alignment.center,
                child: Text(
                  widget.rightText ?? "搜索",
                  style: TextStyle(
                      color: widget.searchTextColor ?? const Color(0xFFFFFFFF),
                      fontSize: 14.0),
                ),
              ),
            )
          ],
        ),
      ),
    );
  }

  /// 校验
  bool _checkInfo() {
    if (isNullOrEmpty(getControllerString(_searchInfoController))) {
      ToastUtil.showToast("请输入搜索内容");
      return false;
    }
    return true;
  }

  /// 搜索
  void _search() {
    if (_checkInfo()) {
      KeyBoardUtils.hideKeyboard();
      if (widget.onCallback != null) {
        widget.onCallback!(getControllerString(_searchInfoController));
      }
    }
  }
}
