import 'package:dio/dio.dart';
import 'package:dio/io.dart';
import 'package:flutter_clx_base/api/proxy.dart';
import 'package:flutter_clx_base/api/transformer.dart';
import 'package:flutter_clx_base/utils/log_utils.dart';
import 'package:flutter_clx_base/utils/toast_util.dart';
import 'package:pretty_dio_logger/pretty_dio_logger.dart';

export 'package:dio/dio.dart';
export 'package:dio/io.dart';

export 'proxy.dart';

abstract class BaseDio {
  final dio = Dio(
    BaseOptions(
        connectTimeout: const Duration(seconds: 10),
        receiveTimeout: const Duration(seconds: 10),
        contentType: Headers.jsonContentType,
        responseType: ResponseType.plain
        // Transform the response data to a String encoded with UTF8.
        // The default value is [ResponseType.JSON].
        ),
  );

  //初始化配置dio实例
  void initConfig() {
    dio.options.baseUrl = setBaseUrl();
    dio.options.headers = setHeaders();
    // 配置JSON转换器
    dio.interceptors.add(getHeadersInterceptor());
    dio.interceptors.add(getLogInterceptor());
    //配置代理
    setProxyUri(dio, isProdEnv());
  }

  final jsonTransformer = MyTransformer();

  String setBaseUrl();

  // 设置请求头拦截器
  Interceptor getHeadersInterceptor();

  // 是否是生产环境，默认为true，
  // 生产环境不设置代理，默认不设置
  bool isProdEnv() => true;

  // 设置log拦截器
  Interceptor getLogInterceptor() => _setLogInterceptor();

  Map<String, dynamic>? setHeaders() => _setHeaders();

  //设置代理
  HttpClientAdapter _setProxy() {
    return IOHttpClientAdapter();
  }

  HttpClientAdapter setProxy() => _setProxy();

  void onLoginExpire() => _onLoginExpire();

  void onNetSuccess(Response response, NetSuccessCallback successCallback,
          NetErrorCallback? errorCallback) =>
      _onNetSuccess(response, successCallback, errorCallback);

  // 登录过期的处理
  void _onLoginExpire() {
    ToastUtil.showToast("登录过期，请重新登录");
  }

  Interceptor _setLogInterceptor() {
    return PrettyDioLogger(
        requestHeader: true,
        requestBody: true,
        responseBody: true,
        responseHeader: false,
        error: true,
        compact: true,
        maxWidth: 120);
  }

  //设置请求头
  Map<String, dynamic>? _setHeaders() {
    return {
      'Content-Type': 'application/json;charset=UTF-8',
    };
  }

  // 构建网络请求
  void request({
    required String requestUrl,
    required Method method,
    Options? options,
    Map<String, dynamic>? queryParameters,
    Map<String, dynamic>? data,
    required NetSuccessCallback successCallback,
    NetErrorCallback? errorCallback,
  }) async {
    try {
      final Response response = await dio.request(
        requestUrl,
        queryParameters: queryParameters,
        data: data,
        options: _checkOptions(method.value, options),
      );
      onNetSuccess(response, successCallback, errorCallback);
    } on DioException catch (e) {
      var result = await _handleHttpException(e.response);
      // result 为null时，已调用重新登录方法 onLoginExpire，onLoginExpire需重写
      if (result != null) {
        await errorCallback?.call(result["code"], result["message"]);
      }
    }
  }

  // 构建网络请求同步请求
  Future<dynamic> requestSync({
    required String requestUrl,
    required Method method,
    Options? options,
    Map<String, dynamic>? queryParameters,
    Object? data,
  }) async {
    try {
      final Response response = await dio.request(
        requestUrl,
        queryParameters: queryParameters,
        data: data,
        options: _checkOptions(method.value, options),
      );
      try {
        final String data = response.data.toString();
        final bool isCompute = data.length > 10 * 1024;
        // loggerNoStack.i('isCompute:$isCompute');
        final Map<String, dynamic> map = isCompute
            ? await jsonTransformer.parseJson(data)
            : jsonTransformer.parseAndDecode(data);
        var responseCode = map['code']?.toString();
        if (responseCode == "100" || responseCode == "-100") {
          logger.e("登录过期：${map['msg']}");
          onLoginExpire();
        }
        return map;
      } catch (e) {
        logger.e("数据解析错误：${e.toString()}");
        return {"code": 400, "message": "数据解析错误"};
      }
    } on DioException catch (e) {
      var result = await _handleHttpException(e.response);
      // result 为null时，已调用重新登录方法 onLoginExpire，onLoginExpire需重写
      if (result != null) {
        return result;
      }
    }
  }

  // 处理网络请求成功逻辑
  void _onNetSuccess(Response response, NetSuccessCallback successCallback,
      NetErrorCallback? errorCallback) async {
    try {
      final String data = response.data.toString();
      final bool isCompute = data.length > 10 * 1024;
      // loggerNoStack.i('isCompute:$isCompute');
      final Map<String, dynamic> map = isCompute
          ? await jsonTransformer.parseJson(data)
          : jsonTransformer.parseAndDecode(data);
      var responseCode = map['code']?.toString();
      if (responseCode == "100" || responseCode == "-100") {
        logger.e("登录过期：${map['msg']}");
        onLoginExpire();
        return;
      }

      if (map['code'] == 0) {
        await successCallback.call(map['data']);
      } else {
        logger.e("请求错误：${map['code']} ${map['msg']}");
        await errorCallback?.call(map['code'], map['msg']);
      }
    } catch (e) {
      logger.e("数据解析错误：${e.toString()}");
      await errorCallback?.call(400, "数据解析错误");
    }
  }

  /// 处理http异常
  Future<Map<String, dynamic>?> _handleHttpException(Response? response) async {
    if (response == null) {
      return {"code": -500, "message": "网络请求错误，请稍后重试"};
    }
    var httpCode = response.statusCode; // httpCode
    var httpData = response.data; // httpData
    try {
      var httpResult = await jsonTransformer.parseJson(httpData);
      if (httpCode == 401 && httpResult["code"]?.toString() == "-100") {
        logger.e("登录过期：${httpResult['msg']}");
        onLoginExpire();
        return null;
      }
      return {"code": httpCode, "message": httpResult['msg'] ?? "网络请求错误，请稍后重试"};
    } on Exception catch (e) {
      logger.e("请求错误：${e.toString()}");
    }
    return {"code": httpCode ?? -500, "message": "网络请求错误，请稍后重试"};
  }
}

Options _checkOptions(String method, Options? options) {
  options ??= Options();
  options.method = method;
  return options;
}

typedef NetSuccessCallback = Function(dynamic data);
typedef NetSuccessListCallback = Function(List? list);
typedef NetErrorCallback = Function(dynamic code, String? msg);

enum Method {
  get,
  post,
  put,
  patch,
  delete,
}

extension MethodExtension on Method {
  String get value => ['GET', 'POST', 'PUT', 'PATCH', 'DELETE'][index];
}
