// ignore: depend_on_referenced_packages
import 'package:shared_preferences/shared_preferences.dart';

/// 推荐：使用 Enum 来管理所有的 Key，避免字符串拼写错误
enum StorageKey {
  token,
  themeMode,
  userName,
  isFirstLaunch,
  testKey, // 示例键
}

class StorageService {
  // 1. 私有构造函数
  StorageService._internal();

  // 2. 静态单例对象
  static final StorageService _instance = StorageService._internal();

  // 3. 工厂构造函数，返回单例
  factory StorageService() => _instance;

  // 4. 初始化新版 API 实例 (无缓存，全异步，跨 Isolate 安全)
  final SharedPreferencesAsync _prefs = SharedPreferencesAsync();

  // =========================================================
  // 通用泛型存储方法 (支持 String 和 Enum 作为 Key)
  // =========================================================

  /// 保存字符串
  Future<void> setString(dynamic key, String value) async {
    await _prefs.setString(_getKey(key), value);
  }

  /// 获取字符串
  Future<String?> getString(dynamic key) async {
    return await _prefs.getString(_getKey(key));
  }

  /// 保存布尔值
  Future<void> setBool(dynamic key, bool value) async {
    await _prefs.setBool(_getKey(key), value);
  }

  /// 获取布尔值
  Future<bool?> getBool(dynamic key) async {
    return await _prefs.getBool(_getKey(key));
  }

  /// 保存整数
  Future<void> setInt(dynamic key, int value) async {
    await _prefs.setInt(_getKey(key), value);
  }

  /// 获取整数
  Future<int?> getInt(dynamic key) async {
    return await _prefs.getInt(_getKey(key));
  }

  /// 保存浮点数
  Future<void> setDouble(dynamic key, double value) async {
    await _prefs.setDouble(_getKey(key), value);
  }

  /// 获取浮点数
  Future<double?> getDouble(dynamic key) async {
    return await _prefs.getDouble(_getKey(key));
  }

  /// 保存字符串列表
  Future<void> setStringList(dynamic key, List<String> value) async {
    await _prefs.setStringList(_getKey(key), value);
  }

  /// 获取字符串列表
  Future<List<String>?> getStringList(dynamic key) async {
    return await _prefs.getStringList(_getKey(key));
  }

  // =========================================================
  // 删除与清空
  // =========================================================

  /// 删除单个 Key
  Future<void> remove(dynamic key) async {
    await _prefs.remove(_getKey(key));
  }

  /// 清空所有数据 (慎用)
  /// allowList: 如果提供了列表，则只清除列表中的 Key
  Future<void> clear({Set<String>? allowList}) async {
    await _prefs.clear(allowList: allowList);
  }

  // =========================================================
  // 内部辅助方法
  // =========================================================

  /// 将 Enum 转换为 String，或者直接使用 String
  String _getKey(dynamic key) {
    if (key is Enum) {
      return key.name; // 或者使用 key.toString() 取决于你的偏好
    } else if (key is String) {
      return key;
    } else {
      throw Exception('Storage Key must be of type String or Enum');
    }
  }
}