import 'dart:io';

import 'package:feedback_complaint_plug/utils/toast_util.dart';
import 'package:flutter/material.dart';
import 'package:image_picker/image_picker.dart';
import 'package:permission_handler/permission_handler.dart';

/// 选择图片（相机、相册）
class SelectImageDialog extends StatelessWidget {
  final String? title;
  final Function onCallback;

  const SelectImageDialog({super.key, this.title, required this.onCallback});

  @override
  Widget build(BuildContext context) {
    return Column(
      mainAxisSize: MainAxisSize.min,
      children: [
        const SizedBox(height: 10.0),
        _cardContainer(children: [
          _buttonItem(
              title: "拍照",
              onTap: () async {
                await _selectPic(ImageSource.camera, onCallback);
              }),
          const Divider(height: 1.0),
          _buttonItem(
              title: "从相册选择",
              onTap: () async {
                await _selectPic(ImageSource.gallery, onCallback);
              }),
        ]),
        const SizedBox(height: 10.0),
        _cardContainer(children: [
          _buttonItem(
            title: "取消",
            onTap: () => Navigator.pop(context),
          )
        ]),
        const SizedBox(height: 10.0),
      ],
    );
  }

  /// 圆角卡片
  _cardContainer({required List<Widget> children}) {
    return Container(
      width: double.infinity,
      margin: const EdgeInsets.symmetric(horizontal: 20.0),
      decoration: BoxDecoration(
        color: Colors.white,
        borderRadius: BorderRadius.circular(10.0),
      ),
      child: Column(
        children: children,
      ),
    );
  }

  /// 按钮
  _buttonItem({title, onTap}) {
    return InkWell(
      onTap: onTap,
      child: Container(
        constraints: const BoxConstraints(minHeight: 45.0),
        alignment: Alignment.center,
        child: Text(
          title ?? "",
          style: const TextStyle(
              fontSize: 14.0,
              color: Color(0xFF212121),
              fontWeight: FontWeight.bold),
        ),
      ),
    );
  }

  /// 选择和加载图片
  Future<void> _selectPic(ImageSource source, onSuccess) async {
    if (Platform.isIOS) {
      switch (source) {
        case ImageSource.gallery:
          await _galleryIOS(source, onSuccess);
          break;
        case ImageSource.camera:
          await _cameraIOS(source, onSuccess);
          break;
      }
    } else {
      if (!await _requestPermissionCam()) {
        return;
      }
      await _callbackImage(source, onSuccess);
    }
  }

  _galleryIOS(ImageSource source, onSuccess) async {
    if (await _requestPermissionPhotos()) {
      _callbackImage(source, onSuccess);
    } else {
      if (await _requestPermissionPhotosAddOnly()) {
        _callbackImage(source, onSuccess);
      } else {
        ToastUtil.showToast("无法访问相册中的照片，请前往系统设置！");
        return;
      }
    }
  }

  _cameraIOS(ImageSource source, onSuccess) async {
    if (await _requestPermissionCam()) {
      _callbackImage(source, onSuccess);
    } else {
      ToastUtil.showToast("无法访问相机，请前往系统设置！");
      return;
    }
  }

  Future<void> _callbackImage(ImageSource source, onSuccess) async {
    onCallback.call(source);
  }

  Future<bool> _requestPermissionPhotos() async {
    final permissions = await Permission.photos.request();
    if (permissions.isGranted) {
      return true;
    } else {
      ToastUtil.showToast('需要相册权限!');
      return false;
    }
  }

  Future<bool> _requestPermissionPhotosAddOnly() async {
    final permissions = await Permission.photosAddOnly.request();
    if (permissions.isGranted) {
      return true;
    } else {
      ToastUtil.showToast('需要相册权限!');
      return false;
    }
  }

  Future<bool> _requestPermissionCam() async {
    final permissions = await Permission.camera.request();
    if (permissions.isGranted) {
      return true;
    } else {
      ToastUtil.showToast('需要相机权限!');
      return false;
    }
  }
}
