import 'dart:io';

import 'package:feedback_complaint_plug/widget/select_image_dialog.dart';
import 'package:file_picker/file_picker.dart';
import 'package:flutter/material.dart';
import 'package:image/image.dart' as image;
import 'package:image_picker/image_picker.dart';
import 'package:path_provider/path_provider.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:webview_flutter/webview_flutter.dart';
// Import for Android features.
import 'package:webview_flutter_android/webview_flutter_android.dart';
// Import for iOS features.
import 'package:webview_flutter_wkwebview/webview_flutter_wkwebview.dart';

import '../utils/toast_util.dart';

/// 反馈投诉
class ComplaintWebPage extends StatefulWidget {
  final String? url; // 网页地址
  final VoidCallback? onFinishPage; // 关闭页面回调方法

  const ComplaintWebPage({super.key, required this.url, this.onFinishPage});

  @override
  State<ComplaintWebPage> createState() => _ComplaintWebPageState();
}

class _ComplaintWebPageState extends State<ComplaintWebPage> {
  String? _title; // 题目
  late WebViewController _webViewController; // webController

  @override
  void initState() {
    _initWeb();
    super.initState();
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        leading: InkWell(
          onTap: () {
            Navigator.pop(context);
          },
          child: const Icon(Icons.arrow_back_ios),
        ),
        title: const Text("反馈投诉"),
        centerTitle: true,
      ),
      body: WebViewWidget(controller: _webViewController),
    );
  }

  /// 初始化web
  void _initWeb() {
    late final PlatformWebViewControllerCreationParams params;
    if (WebViewPlatform.instance is WebKitWebViewPlatform) {
      params = WebKitWebViewControllerCreationParams(
        allowsInlineMediaPlayback: true,
        mediaTypesRequiringUserAction: const <PlaybackMediaTypes>{
          PlaybackMediaTypes.audio,
          PlaybackMediaTypes.video,
        },
      );
    } else {
      params = AndroidWebViewControllerCreationParams();
    }

    final WebViewController controller =
        WebViewController.fromPlatformCreationParams(params);
    // #enddocregion platform_features
    controller
      ..setJavaScriptMode(JavaScriptMode.unrestricted)
      ..addJavaScriptChannel("finishPage",
          onMessageReceived: (JavaScriptMessage message) async {
        debugPrint(
            "JavascriptChannel:method = finishPage message = ${message.message}");
        _finishPage(message.message);
      })
      ..addJavaScriptChannel("showToast",
          onMessageReceived: (JavaScriptMessage message) async {
        debugPrint(
            "JavascriptChannel:method = showToast message = ${message.message}");
        _onShowToast(message.message);
      })
      ..setNavigationDelegate(
          NavigationDelegate(onPageFinished: _onPageFinished))
      ..loadRequest(Uri.parse(widget.url ?? ""));

    if (controller.platform is AndroidWebViewController) {
      // 选择文件回调
      (controller.platform as AndroidWebViewController)
        ..setOnPlatformPermissionRequest(
            (PlatformWebViewPermissionRequest request) {
          request.grant();
        })
        ..setOnShowFileSelector(_androidFilePicker);
    } else if (controller.platform is WebKitWebViewController) {
      (controller.platform as WebKitWebViewController)
          .setOnPlatformPermissionRequest(
              (PlatformWebViewPermissionRequest request) {
        debugPrint("request = $request");
        request.grant();
      });
    }
    _webViewController = controller;
  }

  /// Android选择图片
  Future<List<String>> _androidFilePicker(FileSelectorParams params) async {
    debugPrint("params-->${params.acceptTypes}");
    if (params.acceptTypes.any((type) => type == 'image/*')) {
      var result = await _showImagePicker(context);
      debugPrint("result-->$result");
      return result;
    } else if (params.acceptTypes.any((type) => type == '' || type == "*/*")) {
      if (!await _requestPermissionStorage()) {
        ToastUtil.showToast("需要访问存储权限才能使用该功能，请到设置中开启");
        return [];
      }
      params.acceptTypes.any((type) => type == '*/*');
      // 选择文件 发送给web
      FilePickerResult? result =
          await FilePicker.platform.pickFiles(allowMultiple: true);

      if (result == null) return [];

      List<File> files = result.paths.map((path) => File(path!)).toList();
      debugPrint(
          "files = ${files.map((file) => file.uri.toString()).toList()}");
      return files.map((file) => file.uri.toString()).toList(growable: false);
    }

    return [];
  }

  /// 选择图片底部弹框
  Future<List<String>> _showImagePicker(context) async {
    if (!await _checkPermission()) {
      //对话框说明使用相机的目的
      showDialog(
          context: context,
          builder: (context) => AlertDialog(
                title: const Text("温馨提示"),
                content: const Text("需要使用相机权限，用于拍摄反馈问题图片"),
                actions: <Widget>[
                  TextButton(
                    child: const Text("取消"),
                    onPressed: () => Navigator.pop(context),
                  ),
                  TextButton(
                    child: const Text("确定"),
                    onPressed: () async {
                      final permissions = await Permission.camera.request();
                      if (permissions.isGranted) {
                        debugPrint("相机权限申请成功");
                      } else {
                        ToastUtil.showToast('需要相机权限!');
                      }
                      Navigator.pop(context);
                    },
                  ),
                ],
              ));

      return [];
    }

    ImageSource? source;
    await showModalBottomSheet(
      backgroundColor: Colors.transparent,
      isScrollControlled: true,
      context: context,
      builder: (BuildContext context) {
        return SelectImageDialog(onCallback: (imageSource) async {
          source = imageSource;
          debugPrint("imageSource-->$source");
          Navigator.pop(context);
        });
      },
    );
    debugPrint("source-->$source");
    if (source == null) return [];
    var list = await _convertImagePicker(source!);
    return list;
  }

  /// 图片转换
  Future<List<String>> _convertImagePicker(ImageSource source) async {
    var photo = await ImagePicker().pickImage(source: source, imageQuality: 20);
    // 未选择图片时 返回值为null，不需要增加提示信息
    if (photo == null) {
      return [];
    }
    final imageData = await photo.readAsBytes();
    final decodedImage = image.decodeImage(imageData)!;
    final scaledImage = image.copyResize(decodedImage, width: 500);
    final jpg = image.encodeJpg(scaledImage, quality: 90);

    final filePath = (await getTemporaryDirectory()).uri.resolve(
          './image_${DateTime.now().microsecondsSinceEpoch}.jpg',
        );
    final file = await File.fromUri(filePath).create(recursive: true);
    await file.writeAsBytes(jpg, flush: true);
    debugPrint("file = ${file.uri.toString()}");
    return [file.uri.toString()];
  }

  /// 获取存储权限
  Future<bool> _requestPermissionStorage() async {
    final permissions = await Permission.storage.request();
    if (permissions.isGranted) {
      return true;
    } else {
      ToastUtil.showToast('需要存储权限!');
      return false;
    }
  }

  /// 检测是否有权限
  Future<bool> _checkPermission() async {
    final permissions = await Permission.camera.status;
    if (permissions.isGranted) {
      return true;
    } else {
      return false;
    }
  }

  /// 网页加载完成
  void _onPageFinished(String url) async {
    _title = await _webViewController.getTitle();
    debugPrint("web onPageFinished：title = $_title url = ${widget.url}");
    setState(() {});
  }

  /// 关闭页面
  _finishPage(String? msg) {
    debugPrint("finishPage：");
    Navigator.pop(context);
    widget.onFinishPage?.call();
  }

  /// 展示toast
  _onShowToast(String? msg) {
    ToastUtil.showToast(msg);
  }
}
