import 'package:flutter/widgets.dart';
import '../core/verification_service_interface.dart';
import '../core/countdown_controller.dart';
import '../core/verification_state_machine.dart';
import '../utils/code_send_enum.dart';

/// 验证码 ViewModel
class VerificationCodeViewModel extends ChangeNotifier {
  VerificationCodeViewModel({
    required this.service,
    this.maxLength = 6,
    this.isFirstResponsed = false,
    this.isAccessOutbound = true,
    this.onShowToast,
  })  : _countdownController = CountdownController(),
        _stateMachine = VerificationStateMachine(isAccessOutbound: isAccessOutbound) {
    codeInputCtrl.addListener(_verifyCodeInput);
    _countdownController.addListener(_onCountdownChanged);

    if (isFirstResponsed) {
      codeInputFocus.requestFocus();
    }
  }

  /// 外部注入的服务接口
  final IVerificationService service;

  /// 倒计时控制器
  final CountdownController _countdownController;
  
  /// 状态机
  final VerificationStateMachine _stateMachine;

  /// 验证码长度
  final int maxLength;

  /// 是否是成为第一响应者
  final bool isFirstResponsed;

  /// 是否接入外呼功能
  final bool isAccessOutbound;

  /// Toast 回调
  final Function(String msg)? onShowToast;

  /// 验证码输入框焦点
  final FocusNode codeInputFocus = FocusNode();

  /// 验证码输入框控制器
  final TextEditingController codeInputCtrl = TextEditingController();

  /// 验证码校验错误信息
  String _verifyCodeError = '';
  String get verifyCodeError => _verifyCodeError;

  /// 获取验证码按钮文案
  String _codeSendBtn = '获取验证码';
  String get codeSendBtn => _codeSendBtn;

  /// 验证码发送状态
  CodeSendType _codeSendType = CodeSendType.none;
  CodeSendType get codeSendType => _codeSendType;

  /// 是否展示提示信息模块
  bool _isShowTip = false;
  bool get isShowTip => _isShowTip;

  /// 记录上一次输入，防止重复校验
  String _lastInput = '';

  /// 是否允许发送验证码
  bool _canSend = true;

  @override
  void dispose() {
    codeInputCtrl.removeListener(_verifyCodeInput);
    _countdownController.removeListener(_onCountdownChanged);
    codeInputCtrl.dispose();
    codeInputFocus.dispose();
    _countdownController.dispose();
    super.dispose();
  }

  /// 倒计时变化回调
  void _onCountdownChanged() {
    if (_countdownController.isRunning) {
      _codeSendBtn = '已发送${_countdownController.currentCount}s';
    } else {
      // 倒计时结束
      _codeSendBtn = '';
      _canSend = true;

      // 在 startTimer 时记录类型，或者直接根据当前状态判断
      if (_codeSendType.isSend) {
         _codeSendType = _stateMachine.nextState(_codeSendType, isSmsSend: _codeSendType.isSms);
         _handleTipInfo();
      }
    }
    notifyListeners();
  }

  /// 校验验证码
  void _verifyCodeInput() async {
    if (_lastInput == codeInputCtrl.text) return;
    _lastInput = codeInputCtrl.text;

    _handleTipInfo();
    notifyListeners();

    if (codeInputCtrl.text.length != maxLength) return;
    
    final result = await service.verifyCode(codeInputCtrl.text);
    _verifyCodeError = result ?? '';
    _handleTipInfo();
    notifyListeners();
  }

  /// 外部调用开启定时器
  void startTimerAndChangeType() {
    if (!_canSend) return;
    _canSend = false;
    
    _codeSendType = _stateMachine.nextState(_codeSendType);
    _countdownController.start();
    notifyListeners();
  }

  /// 发送短信验证码
  Future<void> sendSmsCode() async {
    if (!_canSend) return;
    _canSend = false;
    
    if (_countdownController.isRunning) return;

    final success = await service.sendSmsCode();

    if (success) {
      _codeSendType = _stateMachine.nextState(_codeSendType, isSmsSend: true);
      _countdownController.start();
    } else {
      _canSend = true;
    }
    notifyListeners();
  }

  /// 发送语音验证码
  Future<void> sendVoiceCode() async {
    if (!_canSend) return;
    _canSend = false;
    
    if (_countdownController.isRunning) return;

    final success = await service.sendVoiceCode();

    if (success) {
      _codeSendType = _stateMachine.nextState(_codeSendType, isSmsSend: false);
      _countdownController.start();
      onShowToast?.call('即将向您电话播报');
    } else {
      _canSend = true;
    }
    notifyListeners();
  }

  /// 联系客服
  Future<void> contactService() async {
    await service.contactCustomerService();
  }

  /// 处理提示信息
  void _handleTipInfo() {
    _isShowTip = _verifyCodeError.isNotEmpty ||
        (codeInputCtrl.text.isEmpty && !_countdownController.isRunning && _codeSendType != CodeSendType.none);
  }
}
