//
//  ClxMapNavigationView.m
//  clx_map_navigation
//
//  Created by yjc on 2022/11/30.
//

#import "ClxMapNavigationView.h"
#import "ClxMapNavigationConvertUtil.h"
#import "MoreMenuView.h"

@interface ClxMapNavigationView ()<AMapNaviDriveViewDelegate, AMapNaviDriveDataRepresentable, AMapNaviDriveManagerDelegate, MoreMenuViewDelegate>

@property (nonatomic, strong) AMapNaviDriveView *driveView;
@property (nonatomic, strong) FlutterMethodChannel *channel;
@property (nonatomic, assign) int64_t viewId;
@property (nonatomic, strong) NSObject<FlutterPluginRegistrar>* registrar;

@property (nonatomic, assign) MAMapRect initLimitMapRect;//初始化时，限制的地图范围；如果为{0,0,0,0},则没有限制
@property (nonatomic, strong) AMapNaviRouteGroup *driveGroup;
@property (nonatomic, strong) AMapNaviPoint *startPoint;
@property (nonatomic, strong) AMapNaviPoint *endPoint;
@property (nonatomic, strong) MoreMenuView *moreMenu;

@end

@implementation ClxMapNavigationView

#pragma mark - 周期函数

- (instancetype)initWithFrame:(CGRect)frame
               viewIdentifier:(int64_t)viewId
                    arguments:(id _Nullable)args
                    registrar:(NSObject<FlutterPluginRegistrar>*)registrar {
    if (self = [super init]) {
        NSAssert([args isKindOfClass:[NSDictionary class]], @"传参错误");
        //构建methedChannel
        //NSString *channelName = [NSString stringWithFormat:@"amap_flutter_map_navigation_lld", viewId];
        NSString *channelName = [NSString stringWithFormat:@"clx_map_navigation"];
        _channel = [FlutterMethodChannel methodChannelWithName:channelName
                                               binaryMessenger:registrar.messenger];
        _viewId = viewId;
        _registrar = registrar;
        
        // 参数解析
        [self parseParams:args];
        
        // 创建UI
        [self getCurrentView:frame];
        
        // 回调处理
        [self setMethodCallHandler];
    }
    return self;
}

- (UIView*)view {
    return _driveView;
}

- (void)dealloc {
    // 销毁单例
    [AMapNaviDriveManager destroyInstance];
    [_driveView removeFromSuperview];
}

// 参数解析
- (void)parseParams:(id _Nullable)args {
    NSLog(@"---parseParams---: args=%@", args);

    NSDictionary *dict = args;
    
    // 开启 HTTPS 功能
    [[AMapServices sharedServices] setEnableHTTPS:YES];
    NSDictionary *apiKey = dict[@"apiKey"];
    if (apiKey && [apiKey isKindOfClass:[NSDictionary class]]) {
        NSString *iosKey = apiKey[@"iosKey"];
        if (iosKey && iosKey.length > 0) {//通过flutter传入key，则再重新设置一次key
            [AMapServices sharedServices].apiKey = iosKey;
        }
    }
    // 这里统一检查key的设置是否生效
    NSAssert(([AMapServices sharedServices].apiKey != nil), @"没有设置APIKey，请先设置key");
    
    // 隐私合规
    [[AMapNaviManagerConfig sharedConfig] updatePrivacyShow:AMapPrivacyShowStatusDidShow privacyInfo:AMapPrivacyInfoStatusDidContain];
    [[AMapNaviManagerConfig sharedConfig] updatePrivacyAgree:AMapPrivacyAgreeStatusDidAgree];
    
    self.startPoint = nil;
    NSDictionary *startPoint = dict[@"startPoint"];
    if (startPoint && [startPoint isKindOfClass:[NSDictionary class]]) {
        if ([startPoint.allKeys containsObject:@"latitude"] && [startPoint.allKeys containsObject:@"longitude"]) {
            NSString *startPointLat = startPoint[@"latitude"];
            NSString *startPointLon = startPoint[@"longitude"];
            NSLog(@"---startPointLat: %@, startPointLon=%@", startPointLat, startPointLon);
            self.startPoint = [AMapNaviPoint locationWithLatitude:[startPointLat floatValue] longitude:[startPointLon floatValue]];
            NSLog(@"---self.startPoint: %@", self.startPoint);
        }
    }
    
    self.endPoint = nil;
    NSDictionary *endPoint = dict[@"endPoint"];
    if (endPoint && [endPoint isKindOfClass:[NSDictionary class]]) {
        if ([endPoint.allKeys containsObject:@"latitude"] && [endPoint.allKeys containsObject:@"longitude"]) {
            NSString *endPointLat = endPoint[@"latitude"];
            NSString *endPointLon = endPoint[@"longitude"];
            NSLog(@"---endPointLat: %@, endPointLon=%@", endPointLat, endPointLon);
            self.endPoint = [AMapNaviPoint locationWithLatitude:[endPointLat floatValue] longitude:[endPointLon floatValue]];
            NSLog(@"---self.endPoint: %@", self.endPoint);
        }
    }
}

- (void)getCurrentView:(CGRect)frame {
    NSLog(@"---getCurrentView---");
    
    _driveView = [[AMapNaviDriveView alloc] initWithFrame:frame];
    _driveView.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight;
    _driveView.delegate = self;
    _driveView.showUIElements = YES;
    _driveView.showGreyAfterPass = YES;
    _driveView.autoZoomMapLevel = YES;
    _driveView.mapViewModeType = AMapNaviViewMapModeTypeNight;
    _driveView.autoSwitchShowModeToCarPositionLocked = YES;
    _driveView.trackingMode = AMapNaviViewTrackingModeCarNorth;
    _driveView.logoCenter = CGPointMake(_driveView.logoCenter.x + 2, _driveView.logoCenter.y + 60);
    [_driveView setCarImage:[UIImage imageNamed:@"customCar"]];
    _driveView.showMoreButton = YES;
    
    // 请在 dealloc 函数中执行 [AMapNaviDriveManager destroyInstance] 来销毁单例
    [[AMapNaviDriveManager sharedInstance] setDelegate:self];
    [[AMapNaviDriveManager sharedInstance] setIsUseInternalTTS:YES];
    
    [[AMapNaviDriveManager sharedInstance] setAllowsBackgroundLocationUpdates:YES];
    [[AMapNaviDriveManager sharedInstance] setPausesLocationUpdatesAutomatically:NO];
    
    // 将self 、driveView、trafficBarView 添加为导航数据的Representative，使其可以接收到导航诱导数据
    [[AMapNaviDriveManager sharedInstance] addDataRepresentative:_driveView];
    [[AMapNaviDriveManager sharedInstance] addDataRepresentative:self];
    
    // 加载设置按钮
    self.moreMenu = [[MoreMenuView alloc] init];
    self.moreMenu.autoresizingMask = UIViewAutoresizingFlexibleWidth|UIViewAutoresizingFlexibleHeight;
    [self.moreMenu setDelegate:self];
    
    // 判断是否进行路径规划
    if (self.startPoint && self.endPoint) {
        NSLog(@"---driveRoutePlan :  self.startPoint=%@, self.endPoint=%@", self.startPoint, self.endPoint);
        [self driveRoutePlan];
    }
}

#pragma mark -
#pragma mark - drive 独立路径规划和导航

//开始导航
- (void)startNav {
    // 判断是否进行路径规划
    if (self.startPoint && self.endPoint) {
        NSLog(@"---driveRoutePlan :  self.startPoint=%@, self.endPoint=%@", self.startPoint, self.endPoint);
        [self driveRoutePlan];
    }
}

- (void)driveRoutePlan {
    AMapNaviPOIInfo *startPOIInfo = [[AMapNaviPOIInfo alloc] init];
    // 1.只传经纬度
    startPOIInfo.locPoint = self.startPoint;
    // 2.只传POIID
    //startPOIInfo.mid = @"";
    
    // 3.传起始角度和经纬度，POIID
    //startPOIInfo.mid = @"";
    //startPOIInfo.startAngle = 10;
    
    AMapNaviPOIInfo *endPOIInfo = [[AMapNaviPOIInfo alloc] init];
    endPOIInfo.locPoint = self.endPoint;
    __weak typeof(self)weakSelf = self;
    [[AMapNaviDriveManager sharedInstance] independentCalculateDriveRouteWithStartPOIInfo:startPOIInfo
                                                                               endPOIInfo:endPOIInfo
                                                                              wayPOIInfos:nil
                                                                          drivingStrategy:AMapNaviDrivingStrategyMultipleDefault
                                                                                 callback:^(AMapNaviRouteGroup * _Nullable routeGroup, NSError * _Nullable error) {
        NSLog(@"routeGroup = %@, error = %@",routeGroup,error);
        weakSelf.driveGroup = routeGroup;
        [weakSelf startDriveNavi];
    }];
}

- (void)startDriveNavi {
    if (self.driveGroup == nil) {
        NSLog(@"请先进行独立路径规划");
        return;
    }
    // 1.导航前行前选路
    // [self.driveGroup selectNaviRouteWithRouteID:];
    // 2.开始导航
    [[AMapNaviDriveManager sharedInstance] startGPSNavi:self.driveGroup];
}

#pragma mark -
#pragma mark - AMapNaviDriveManagerDelegate

- (void)driveManager:(AMapNaviDriveManager *)driveManager error:(NSError *)error {
    NSLog(@"error:{%ld - %@}", (long)error.code, error.localizedDescription);
}

- (void)driveManagerOnCalculateRouteSuccess:(AMapNaviDriveManager *)driveManager {
    NSLog(@"onCalculateRouteSuccess");
    // 算路成功后开始GPS导航
    //[[AMapNaviDriveManager sharedInstance] startGPSNavi];
}

- (void)driveManager:(AMapNaviDriveManager *)driveManager onCalculateRouteFailure:(NSError *)error {
    NSLog(@"onCalculateRouteFailure:{%ld - %@}", (long)error.code, error.localizedDescription);
}

- (void)driveManager:(AMapNaviDriveManager *)driveManager didStartNavi:(AMapNaviMode)naviMode {
    NSLog(@"didStartNavi");
}

- (void)driveManagerNeedRecalculateRouteForYaw:(AMapNaviDriveManager *)driveManager {
    NSLog(@"needRecalculateRouteForYaw");
}

- (void)driveManagerNeedRecalculateRouteForTrafficJam:(AMapNaviDriveManager *)driveManager {
    NSLog(@"needRecalculateRouteForTrafficJam");
}

- (void)driveManager:(AMapNaviDriveManager *)driveManager onArrivedWayPoint:(int)wayPointIndex {
    NSLog(@"onArrivedWayPoint:%d", wayPointIndex);
}

- (void)driveManager:(AMapNaviDriveManager *)driveManager playNaviSoundString:(NSString *)soundString soundStringType:(AMapNaviSoundType)soundStringType {
    NSLog(@"playNaviSoundString:{%ld:%@}", (long)soundStringType, soundString);
}

- (void)driveManagerDidEndEmulatorNavi:(AMapNaviDriveManager *)driveManager {
    NSLog(@"didEndEmulatorNavi");
}

- (void)driveManagerOnArrivedDestination:(AMapNaviDriveManager *)driveManager {
    NSLog(@"onArrivedDestination");
}

#pragma mark -
#pragma mark - AMapNaviDriveViewDelegate

- (void)driveViewCloseButtonClicked:(AMapNaviDriveView *)driveView; {
    [driveView removeFromSuperview];
    [[AMapNaviDriveManager sharedInstance] stopNavi];
    [[AMapNaviDriveManager sharedInstance] removeDataRepresentative:_driveView];
    
    NSMutableDictionary *arguments = [NSMutableDictionary dictionary];
    [arguments setValue:@"退出" forKey:@"msg"];
    if (_channel) {
        NSLog(@"driveViewCloseButtonClicked - 退出: %@", arguments);
        [_channel invokeMethod:@"navigationClosed" arguments:arguments];
    }
}

- (void)driveViewMoreButtonClicked:(AMapNaviDriveView *)driveView {
    // 配置MoreMenu状态
    [_moreMenu setTrackingMode:self.driveView.trackingMode];
    [_moreMenu setShowNightType:self.driveView.mapViewModeType];
    
    [_moreMenu setFrame:self.view.bounds];
    [self.view addSubview:_moreMenu];
}

- (void)driveViewTrunIndicatorViewTapped:(AMapNaviDriveView *)driveView {
    NSLog(@"TrunIndicatorViewTapped");
}

- (void)driveView:(AMapNaviDriveView *)driveView didChangeShowMode:(AMapNaviDriveViewShowMode)showMode {
    NSLog(@"didChangeShowMode:%ld", (long)showMode);
}

- (void)driveView:(AMapNaviDriveView *)driveView didChangeDayNightType:(BOOL)showStandardNightType {
    NSLog(@"didChangeDayNightType:%ld", (long)showStandardNightType);
}

#pragma mark -
#pragma mark - MoreMenu Delegate

- (void)moreMenuViewFinishButtonClicked {
    [_moreMenu removeFromSuperview];
}

- (void)moreMenuViewNightTypeChangeTo:(AMapNaviViewMapModeType)mapModeType {
    [self.driveView setMapViewModeType:mapModeType];
}

- (void)moreMenuViewTrackingModeChangeTo:(AMapNaviViewTrackingMode)trackingMode {
    [self.driveView setTrackingMode:trackingMode];
}

#pragma mark -
#pragma mark - Flutter and Navite Method CallHandler

// kChannelName 通道
- (void)setMethodCallHandler {
    __weak __typeof__(self) weakSelf = self;
    [_channel setMethodCallHandler:^(FlutterMethodCall * _Nonnull call, FlutterResult  _Nonnull result) {
        NSLog(@"---setMethodCallHandler---: method=%@, params=%@", call.method, call.arguments);
        //开始导航
        if ([call.method isEqualToString:@"navigation#startNav"]) {
            weakSelf.startPoint = nil;
            NSDictionary *startPoint = [call.arguments objectForKey:@"startPoint"];
            if (startPoint && [startPoint isKindOfClass:[NSDictionary class]]) {
                if ([startPoint.allKeys containsObject:@"latitude"] && [startPoint.allKeys containsObject:@"longitude"]) {
                    NSString *startPointLat = startPoint[@"latitude"];
                    NSString *startPointLon = startPoint[@"longitude"];
                    NSLog(@"---开始导航 startPointLat: %@, startPointLon=%@", startPointLat, startPointLon);
                    weakSelf.startPoint = [AMapNaviPoint locationWithLatitude:[startPointLat floatValue] longitude:[startPointLon floatValue]];
                    NSLog(@"---开始导航 weakSelf.startPoint: %@", weakSelf.startPoint);
                }
            }
            weakSelf.endPoint = nil;
            NSDictionary *endPoint = [call.arguments objectForKey:@"endPoint"];
            if (endPoint && [endPoint isKindOfClass:[NSDictionary class]]) {
                if ([endPoint.allKeys containsObject:@"latitude"] && [endPoint.allKeys containsObject:@"longitude"]) {
                    NSString *endPointLat = endPoint[@"latitude"];
                    NSString *endPointLon = endPoint[@"longitude"];
                    NSLog(@"---开始导航 endPointLat: %@, endPointLon=%@", endPointLat, endPointLon);
                    weakSelf.endPoint = [AMapNaviPoint locationWithLatitude:[endPointLat floatValue] longitude:[endPointLon floatValue]];
                    NSLog(@"---开始导航 weakSelf.endPoint: %@", weakSelf.endPoint);
                }
            }
            //开始导航
            [weakSelf startNav];
        }
    }];
}

@end
