import 'dart:async';
import 'package:clx_map_navigation/amap_navigation_widget.dart';
import 'package:clx_map_navigation/core/map_event.dart';
import 'package:flutter/foundation.dart';
import 'package:flutter/gestures.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'map_navigation_platform_interface.dart';
import 'package:stream_transform/stream_transform.dart';

const VIEW_TYPE = 'clx_map_navigation';

/// An implementation of [ClxMapNavigationPlatform] that uses method channels.
class MethodChannelMapNavigation extends MapNavigationPlatform {
  /// The method channel used to interact with the native platform.
  @visibleForTesting
  final methodChannel = const MethodChannel('clx_map_navigation');

  MethodChannel channel() {
    return methodChannel;
  }

  @override
  Widget buildMapNavigationView(
      Map<String, dynamic> creationParams,
      Set<Factory<OneSequenceGestureRecognizer>> gestureRecognizers,
      void Function(int id) onPlatformViewCreated,
      MapClosedCallback? onMapClosed,) {

    debugPrint("--- buildMapNavigationView --- creationParams: $creationParams");

    if (defaultTargetPlatform == TargetPlatform.android) {
      creationParams['debugMode'] = kDebugMode;
      return AndroidView(
        viewType: VIEW_TYPE,
        onPlatformViewCreated: onPlatformViewCreated,
        gestureRecognizers: gestureRecognizers,
        creationParams: creationParams,
        creationParamsCodec: const StandardMessageCodec(),
      );
    } else if (defaultTargetPlatform == TargetPlatform.iOS) {
      return UiKitView(
        viewType: VIEW_TYPE,
        onPlatformViewCreated: onPlatformViewCreated,
        gestureRecognizers: gestureRecognizers,
        creationParams: creationParams,
        creationParamsCodec: const StandardMessageCodec(),
      );
    }
    return Text('当前平台:$defaultTargetPlatform, 不支持使用高德地图插件');
  }

  // handleMethodCall的`broadcast`
  final StreamController<MapEvent> _mapEventStreamController = StreamController<MapEvent>.broadcast();

  StreamController<MapEvent> getMapEventStreamController() {
    return _mapEventStreamController;
  }

  // 根据mapid返回相应的event.
  Stream<MapEvent> _events() => _mapEventStreamController.stream;

  // 底部退出回调  - whereType：返回特定对象【筛选】
  Stream<MapClosedEvent> onMapClosed() {
    return _events().whereType<MapClosedEvent>();
  }

  @override
  Future<void> startNav(Map? startPoint, Map? endPoint) {
    return channel().invokeMethod<void>('navigation#startNav', <String, dynamic>{
      'startPoint': startPoint,
      'endPoint': endPoint,
    });
  }

}
