import 'dart:async';
import 'package:clx_map_navigation/core/amap_navigation_util.dart';
import 'package:clx_map_navigation/core/map_event.dart';
import 'package:flutter/foundation.dart';
import 'package:flutter/gestures.dart';
import 'package:flutter/material.dart';
import 'package:amap_flutter_base/amap_flutter_base.dart';
import 'package:flutter/services.dart';

import 'core/map_navigation_controller.dart';
import 'core/map_navigation_method_channel.dart';
import 'core/map_navigation_platform_interface.dart';

final MethodChannelMapNavigation _methodChannel = MapNavigationPlatform.instance as MethodChannelMapNavigation;
typedef MapClosedCallback = void Function();
typedef NavCreatedCallback = void Function(MapNavigationController controller);

class AMapNavigationWidget extends StatefulWidget {

  /// 高德开放平台的key
  final AMapApiKey? apiKey;
  /// 起点
  final Map? startPoint;
  /// 终点
  final Map? endPoint;
  /// 需要应用到地图上的手势集合
  final Set<Factory<OneSequenceGestureRecognizer>> gestureRecognizers;
  /// 导航地图底部退出的回调
  final MapClosedCallback? onMapClosed;

  /// 导航创建成功的回调, 收到此回调之后才可以操作地图
  final NavCreatedCallback? onMapCreated;

  const AMapNavigationWidget({
    Key? key,
    this.apiKey,
    required this.startPoint,
    required this.endPoint,
    this.gestureRecognizers = const <Factory<OneSequenceGestureRecognizer>>{},
    this.onMapClosed,
    this.onMapCreated,
  }) : super(key: key);

  @override
  State<StatefulWidget> createState() => _MapNavigationState();
}

class _MapNavigationState extends State<AMapNavigationWidget> {

  //消息订阅对象
  late StreamSubscription _streamSubscription;
  final Completer<MapNavigationController> _controller = Completer<MapNavigationController>();

  @override
  void initState() {
    super.initState();
    _setupChannelsConfig();
    _connectStreams();
  }

  _setupChannelsConfig() {
    _methodChannel.channel().setMethodCallHandler((call) => _handleMethodCall(call));
  }

  // 监听
  _connectStreams() {
    debugPrint("---_connectStreams---");
    if (widget.onMapClosed != null) {
      debugPrint("------ _connectStreams listen ----- ");
      _streamSubscription = _methodChannel.onMapClosed().listen((MapClosedEvent e) => widget.onMapClosed!());
    }
  }

  /// 原生与Flutter交互
  Future<dynamic> _handleMethodCall(MethodCall call) async {
    debugPrint("------_handleMethodCall ----- ");
    switch (call.method) {
      case 'navigationClosed':
        debugPrint("------ navigationClosed ----- ");
        _methodChannel.getMapEventStreamController().add(MapClosedEvent("退出"));
        break;
    }
  }

  @override
  Widget build(BuildContext context) {
    AMapNavigationUtil.init(context);
    final Map<String, dynamic> creationParams = <String, dynamic> {
      'apiKey': widget.apiKey?.toMap(),
      'startPoint': widget.startPoint,
      'endPoint': widget.endPoint,
    };
    debugPrint("--- build --- creationParams: $creationParams");
    Widget mapView = _methodChannel.buildMapNavigationView(
      creationParams,
      widget.gestureRecognizers,
      onPlatformViewCreated,
      widget.onMapClosed,
    );
    return mapView;
  }

  Future<void> onPlatformViewCreated(int id) async {
    debugPrint("********** 地图路径规划 ***********id=$id");
    final MapNavigationController controller =
        await MapNavigationController.init(id);
    _controller.complete(controller);
    final NavCreatedCallback? onMapCreated = widget.onMapCreated;
    if (onMapCreated != null) {
      onMapCreated(controller);
    }
  }

  @override
  void dispose() {
    super.dispose();
    //取消监听
    _streamSubscription.cancel();
  }

}
