import 'package:flutter/material.dart';

import 'button_public_type.dart';

/// 按钮样式
class ButtonPublicRect extends StatefulWidget {
  final String? text; // 按钮文本
  final VoidCallback? onPressed; // 点击事件
  final double? minWidth; // 最小宽度
  final double minHeight; // 最小高度
  final EdgeInsetsGeometry? margin; // 外边距
  final Color? borderColor; // 边框颜色
  final double borderWidth; // 边框宽度
  final Color? textColor; // 按钮文本颜色
  final double textSize; // 按钮文本字体大小
  final Color? backgroundColor; // 按钮可点击颜色
  final FontWeight fontWeight; // 按钮文本字体粗细
  final double borderRadius; // 按钮圆角大小
  final ButtonPublicType type; // 按钮类型
  final Color buttonColor; //按钮颜色

  const ButtonPublicRect({
    super.key,
    required this.text,
    required this.onPressed,
    required this.buttonColor,
    this.minWidth = 0.0,
    this.minHeight = 48.0,
    this.margin,
    this.textSize = 17.0,
    this.textColor,
    this.borderColor,
    this.borderWidth = 1.0,
    this.backgroundColor ,
    this.fontWeight = FontWeight.bold,
    this.borderRadius = 8.0,
    this.type = ButtonPublicType.solid, 
  });

  @override
  State<ButtonPublicRect> createState() => _ButtonPublicRectState();
}

class _ButtonPublicRectState extends State<ButtonPublicRect> {
  @override
  Widget build(BuildContext context) {
    return Container(
      margin: widget.margin,
      constraints: BoxConstraints(
          minWidth: widget.minWidth ?? 0.0, minHeight: widget.minHeight),
      child: TextButton(
        onPressed: () {
          widget.onPressed?.call();
        },
        style: ButtonStyle(
          padding: MaterialStateProperty.all(
              const EdgeInsets.symmetric(horizontal: 5.0)),
          textStyle: MaterialStateProperty.all(TextStyle(
              fontSize: widget.textSize, fontWeight: widget.fontWeight)),
          // 设置文字颜色
          foregroundColor: MaterialStateProperty.all(_textColor()),
          // 设置背景颜色
          backgroundColor: MaterialStateProperty.resolveWith((states) {
            return _backgroundColor();
          }),
          //设置水波纹颜色
          shape: MaterialStateProperty.all(
            RoundedRectangleBorder(
                borderRadius: BorderRadius.circular(widget.borderRadius)),
          ),
          side: MaterialStateProperty.all(
              BorderSide(color: _borderColor(), width: widget.borderWidth)),
        ),
        child: Text(widget.text ?? ""),
      ),
    );
  }

  /// 根据按钮类型设置按钮文本颜色
  /// 按钮类型为outline、dash 文本颜色设置边框颜色
  Color _textColor() {
    switch (widget.type) {
      case ButtonPublicType.solid:
        return widget.textColor ?? Colors.white;
      case ButtonPublicType.outline:
      case ButtonPublicType.dash:
        return widget.textColor ?? widget.borderColor ?? widget.buttonColor;
      default:
        return widget.textColor ?? Colors.white;
    }
  }

  /// 根据按钮类型设置按钮可点击颜色
  /// 按钮类型为outline、dash 按钮可点击颜色为透明
  Color _backgroundColor() {
    switch (widget.type) {
      case ButtonPublicType.solid:
        return widget.backgroundColor ?? widget.buttonColor;
      case ButtonPublicType.outline:
      case ButtonPublicType.dash:
        return Colors.transparent;
      default:
        return widget.backgroundColor?? widget.buttonColor;
    }
  }

  /// 根据按钮类型设置按钮边框颜色
  /// 按钮类型为outline 设置边框颜色
  Color _borderColor() {
    switch (widget.type) {
      case ButtonPublicType.outline:
        return widget.borderColor ?? widget.buttonColor;
      default:
        return Colors.transparent;
    }
  }
}
