import 'dart:ui';

import 'package:flutter/material.dart';
import 'package:get/get.dart';

import '../../../clx_flutter_message.dart';
import '../../model/message_node.dart';
import 'widget/list_item.dart';

class NotificationLayoutWidget extends GetView<NotificationLayoutController> {
  const NotificationLayoutWidget({super.key});

  @override
  Widget build(BuildContext context) {
    return GetBuilder<NotificationLayoutController>(
      init: messageConfig.messageManagement?.notificationLayoutController,
      id: "NotificationLayoutWidget",
      builder: (_) {
        if (controller.headNode == null) {
          return const SizedBox();
        }
        return Container(
          margin: const EdgeInsets.only(top: 35),
          child: buildStackedList(context),
        );
      },
    );
  }

  Widget buildStackedList(BuildContext context) {
    return SizedBox(
      height: 140,
      child: Stack(
        fit: StackFit.expand,
        children: [
          if (controller.headNode?.next != null)
            Positioned(
              bottom: 15,
              left: 25,
              right: 25,
              child: _message(controller.headNode?.next?.data),
            ),
          Positioned(
            bottom: 0,
            left: 10,
            right: 10,
            child: _buildNotification(controller.headNode?.data),
          )
        ],
      ),
    );
  }

  Widget _buildNotification(data) {
    if (data == null) {
      return const SizedBox();
    }
    return Dismissible(
      key: Key(data.messageNo?.toString() ?? ''),
      direction: DismissDirection.endToStart,
      onDismissed: (direction) {
        controller.dismiss();
      },
      child: Stack(
        children: [
          Positioned.fill(
            child: ClipRRect(
              borderRadius: BorderRadius.circular(8),
              child: BackdropFilter(
                filter: ImageFilter.blur(sigmaX: 1.5, sigmaY: 1.5),
                child: Container(
                  decoration: BoxDecoration(
                      borderRadius: BorderRadius.circular(8),
                      color: Colors.black26),
                  width: null,
                  height: null,
                ),
              ),
            ),
          ),
          _message(data),
        ],
      ),
    );
  }

  MessageWidget _message(data) {
    return MessageWidget(
      data: data,
      width: double.infinity,
      onCancel: () {
        controller.ignore();
      },
      onHand: () {
        controller.ignore();
        controller.confirm(data);
      },
    );
  }
}

class NotificationLayoutController extends GetxController {
  NotificationLayoutController();

  MessageNode? headNode;

  _initData() {
    update(["NotificationLayoutWidget"]);
  }

  void setCashMessage(List<MessageData> messages) {
    headNode = null;

    /// 上一个节点
    MessageNode? preNode;
    for (var data in messages) {
      var theNode = MessageNode(data);
      if (preNode == null) {
        preNode = theNode;
        headNode = preNode;
      } else {
        preNode.next = theNode;
        theNode.prev = preNode;
        preNode = theNode;
      }
    }
  }

  void next() {
    messageConfig.messageManagement?.setMessageShowed(headNode?.data);
    if (headNode?.next == null) {
      headNode = null;
      messageConfig.messageManagement?.hideNotification();
      return;
    }
    headNode = headNode?.next;
    headNode?.prev?.next = null;
    headNode?.prev = null;
    _initData();
  }

  void ignore() {
    messageConfig.messageManagement?.markReadAction(headNode?.data);
    messageConfig.messageManagement?.setMessageShowed(headNode?.data);
    next();
  }

  void dismiss() {
    messageConfig.messageManagement?.setMessageShowed(headNode?.data);
    next();
  }

  void confirm(MessageData data) {
    messageConfig.messageManagement?.gotoDealMessage(data);
    messageConfig.messageManagement?.markReadAction(headNode?.data);
    messageConfig.messageManagement?.setMessageShowed(headNode?.data);
    next();
  }

  /// 插入消息
  void insertNotification(MessageData data) {
    if (headNode == null) {
      headNode = MessageNode(data);
    } else {
      headNode?.insertData(data);
    }
  }

  void clear() {
    headNode = null;
    messageConfig.messageManagement?.hideNotification();
    update();
  }

  @override
  void onReady() {
    super.onReady();
    _initData();
  }
}
