import 'package:flutter/material.dart';
import 'package:get/get.dart';

import '../../clx_flutter_message.dart';
import '../../util/string_util.dart';
import '../model/message_node.dart';
import '../widget/button_public/button_public_rect.dart';
import '../widget/button_public/button_public_type.dart';

class NoticeDialogWidget extends GetView<NoticeDialogWidgetController> {
  const NoticeDialogWidget({
    super.key,
    this.decoration,
    this.cancelText,
    this.confirmText,
    this.cancelType,
    this.confirmType,
    this.cancelCallback,
    this.confirmCallback,
    required this.buttonColor,
  });

  // 对话框背景 decoration
  final Decoration? decoration;

  // 取消按钮文字
  final String? cancelText;
  // 确认按钮文字
  final String? confirmText;

  // 取消按钮类型
  final ButtonPublicType? cancelType;
  // 确认按钮类型
  final ButtonPublicType? confirmType;

  // 取消按钮回调
  final VoidCallback? cancelCallback;
  // 确认按钮回调
  final VoidCallback? confirmCallback;

  // 按钮颜色
  final Color buttonColor;

  // 主视图
  Widget _buildView() {
    var message = controller.headNode?.data;
    var extraShowInfoList = message?.textVo?.extraShowInfoList;

    /// 消息能否处理，如果能处理展示去处理按钮，如果不能处理，不展示去处理按钮
    bool canHand = message?.textVo?.data['jumpPageAppUrl'] != null;
    return !controller.showDialog
        ? const SizedBox()
        : Container(
            height: double.infinity,
            width: double.infinity,
            color: Colors.black.withOpacity(0.5),
            padding: const EdgeInsets.symmetric(horizontal: 20),
            child: Center(
              child: Stack(
                children: [
                  Container(
                    decoration: decoration,
                    padding: const EdgeInsets.symmetric(horizontal: 24),
                    child: SingleChildScrollView(
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.center,
                        mainAxisSize: MainAxisSize.min,
                        children: [
                          const SizedBox(height: 32),
                          Text(
                            message?.textVo?.title ?? '',
                            style: const TextStyle(
                                fontSize: 18, fontWeight: FontWeight.bold),
                          ),
                          const SizedBox(height: 8),
                          Text(
                            message?.textVo?.subtitle ?? '',
                            style: const TextStyle(
                                color: Color(0x99000000), fontSize: 16),
                          ),
                          if (extraShowInfoList?.isNotEmpty == true)
                            _valueContent(extraShowInfoList),
                          const SizedBox(height: 24),
                          Row(
                            children: [
                              Expanded(
                                child: ButtonPublicRect(
                                  buttonColor: buttonColor,
                                  type: cancelType ?? ButtonPublicType.outline,
                                  text: cancelText ?? '忽略',
                                  onPressed: () {
                                    if (cancelCallback != null) {
                                      cancelCallback?.call();
                                      controller.next();
                                    } else {
                                      controller.ignore;
                                    }
                                  },
                                  borderRadius: 8,
                                ),
                              ),
                              if (canHand) ...[
                                const SizedBox(width: 10),
                                Expanded(
                                  child: ButtonPublicRect(
                                    buttonColor: buttonColor,
                                    type: confirmType ?? ButtonPublicType.solid,
                                    text: confirmText ?? '去处理',
                                    onPressed: () {
                                      if (confirmCallback != null) {
                                        confirmCallback?.call();
                                        controller.next();
                                      } else {
                                        controller.gotoDealMessage(message);
                                        controller.ignore();
                                      }
                                    },
                                    borderRadius: 8,
                                  ),
                                )
                              ],
                            ],
                          ),
                          const SizedBox(height: 24),
                        ],
                      ),
                    ),
                  ),
                  Positioned(
                    top: 0,
                    right: 0,
                    child: IconButton(
                      onPressed: controller.dismiss,
                      icon: const Icon(
                        Icons.close,
                        size: 16,
                        color: Colors.grey,
                      ),
                    ),
                  )
                ],
              ),
            ),
          );
  }

  Container _valueContent(List<dynamic>? extraShowInfoList) {
    return Container(
      padding: const EdgeInsets.only(left: 10, right: 10, bottom: 10),
      margin: const EdgeInsets.only(top: 8),
      decoration: BoxDecoration(
          color: const Color(0xFFF3F5F8),
          borderRadius: BorderRadius.circular(8)),
      child: Column(
        children: extraShowInfoList!.map<Widget>((e) {
          return Container(
            margin: const EdgeInsets.only(top: 10),
            child: Row(
              children: [
                SizedBox(
                  width: 75,
                  child: Text(
                    (e['name'] ?? '') + "：",
                    style: const TextStyle(
                      color: Color(0xFF4E5969),
                      fontSize: 15,
                    ),
                  ),
                ),
                const SizedBox(width: 10),
                Expanded(
                  child: Row(
                    children: [
                      Flexible(
                        child: Text(
                          StringUtil.formatValue(e['value']),
                          style: const TextStyle(
                            color: Color(0xFF333C4C),
                            fontSize: 15,
                          ),
                          overflow: TextOverflow.ellipsis,
                        ),
                      ),
                      const SizedBox(
                        width: 10,
                      ),
                      Text(
                        StringUtil.formatValue(e['unit'], defaultValue: ''),
                        style: const TextStyle(
                          color: Color(0xFF4E5969),
                          fontSize: 14,
                        ),
                      ),
                    ],
                  ),
                ),
              ],
            ),
          );
        }).toList(),
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    return GetBuilder<NoticeDialogWidgetController>(
      init: messageConfig.messageManagement?.noticeDialogWidgetController,
      id: "NoticeDialogWidget",
      builder: (_) {
        return _buildView();
      },
    );
  }
}

class NoticeDialogWidgetController extends GetxController {
  NoticeDialogWidgetController();
  _initData() {
    update(["NoticeDialogWidget"]);
  }

  /// 所属页面Context
  BuildContext? context;

  MessageNode? headNode;

  bool showDialog = false;

  /// 插入公告
  void insertNotice(MessageData data) async {
    final result = await messageConfig.messageManagement
        ?.onShowAnnouncementMessage(data.toJson());
    if (result == null) {
      _insertNode(data);
      _initData();
    }
  }

  /// 处理公告
  gotoDealMessage(MessageData? data) {
    messageConfig.messageManagement?.gotoDealMessage(data);
  }

  void _insertNode(MessageData data) {
    if (headNode == null) {
      headNode = MessageNode(data);
      showDialog = true;
    } else {
      headNode?.insertData(data);
    }
  }

  void next() {
    headNode = headNode?.next;

    /// 隐藏当前的
    showDialog = false;
    _initData();

    /// 显示下一个
    if (headNode != null) {
      Future.delayed(const Duration(milliseconds: 300)).then((value) {
        showDialog = true;
        _initData();
      });
    }
  }

  void setNotice(List<MessageData> data) async {
    for (var element in data) {
      final result = await messageConfig.messageManagement
          ?.onShowAnnouncementMessage(element.toJson());
      if (result == null) {
        _insertNode(element);
      }
    }

    _initData();
  }

  void clear() {
    headNode = null;
    showDialog = false;
    _initData();
  }

  void ignore() {
    /// 标记已展示
    messageConfig.messageManagement?.setMessageShowed(headNode?.data);
    messageConfig.messageManagement?.markReadAction(headNode?.data);
    next();
  }

  void dismiss() {
    messageConfig.messageManagement?.setMessageShowed(headNode?.data);
    next();
  }

  @override
  void onReady() {
    super.onReady();
    _initData();
  }
}
