import 'dart:async';
import 'dart:io';

import 'package:intl/intl.dart';
import 'package:path_provider/path_provider.dart';

/// 日志级别
enum LogLevel { debug, info, warning, error }

class LogUtil {
  static LogUtil? _instance;
  late String _logDir; // 日志文件存储目录
  final int _maxRetentionDays = 7; // 日志保留天数（超过自动删除）

  LogUtil._();

  static LogUtil get instance {
    _instance ??= LogUtil._();
    return _instance!;
  }

  /// 初始化日志工具（必须在APP启动时调用）
  Future<void> init() async {
    // 获取应用文档目录（iOS/Android 私有目录，无需额外权限）
    Directory appDocDir = await getApplicationDocumentsDirectory();
    _logDir = "${appDocDir.path}/logs";
    // 创建日志目录（若不存在）
    await Directory(_logDir).create(recursive: true);
    // 清理过期日志
    _cleanExpiredLogs();
  }

  /// 写入日志（公共方法，支持不同级别）
  Future<void> log(
      String message, {
        LogLevel level = LogLevel.debug,
        String? tag, // 日志标签（如页面/功能名）
      }) async {
    if (_logDir.isEmpty) {
      throw Exception("LogUtil 未初始化，请先调用 init()");
    }

    // 1. 格式化日志内容（时间 + 级别 + 标签 + 消息）
    String time = DateFormat("yyyy-MM-dd HH:mm:ss.SSS").format(DateTime.now());
    String levelStr = _levelToString(level);
    String logContent = "[$time] [$levelStr] ${tag != null ? "[$tag] " : ""}$message\n";

    // 2. 获取当前日志文件（按天划分，如 2023-10-01.log）
    String fileName = "${DateFormat("yyyy-MM-dd").format(DateTime.now())}.log";
    File logFile = File("$_logDir/$fileName");

    // 3. 追加写入日志（异步操作，避免阻塞UI）
    await logFile.writeAsString(logContent, mode: FileMode.append, flush: true);
  }

  /// 辅助方法：日志级别转字符串
  String _levelToString(LogLevel level) {
    switch (level) {
      case LogLevel.debug:
        return "DEBUG";
      case LogLevel.info:
        return "INFO";
      case LogLevel.warning:
        return "WARNING";
      case LogLevel.error:
        return "ERROR";
    }
  }

  /// 清理过期日志（保留最近 N 天）
  Future<void> _cleanExpiredLogs() async {
    final now = DateTime.now();
    final dir = Directory(_logDir);
    if (!await dir.exists()) return;

    // 遍历目录下所有日志文件
    await for (var entity in dir.list()) {
      if (entity is File && entity.path.endsWith(".log")) {
        // 解析文件名中的日期（如 2023-10-01.log → 2023-10-01）
        String fileName = entity.path.split("/").last.replaceAll(".log", "");
        try {
          DateTime logDate = DateFormat("yyyy-MM-dd").parse(fileName);
          // 计算与当前日期的差值
          int daysDiff = now.difference(logDate).inDays;
          if (daysDiff > _maxRetentionDays) {
            await entity.delete(); // 删除过期文件
          }
        } catch (e) {
          // 文件名格式错误，跳过
        }
      }
    }
  }

  /// 获取所有日志文件路径（用于上传等场景）
  /// Android 路径示例：/data/data/包名/app_flutter/logs/2023-10-01.log
  /// iOS 路径示例：/Users/用户名/Library/Developer/CoreSimulator/Devices/.../data/Containers/Data/Application/.../Documents/logs/2023-10-01.log
  Future<List<String>> getLogFilePaths() async {
    final dir = Directory(_logDir);
    if (!await dir.exists()) return [];

    List<String> paths = [];
    await for (var entity in dir.list()) {
      if (entity is File && entity.path.endsWith(".log")) {
        paths.add(entity.path);
      }
    }
    return paths;
  }
}