import 'dart:async';
import 'dart:io';

import 'package:apk_update/core/services/download_service/download_service.dart';
import 'package:apk_update/utils/dio_update_util.dart';
import 'package:flutter/cupertino.dart';

import '../../../../apk_update_platform_interface.dart';
import '../../../../utils/utils.dart';
import '../../../common/constants.dart';
import 'package:path/path.dart' as p;

import '../../../common/servier_api.dart';

/// OSS下载
class OSSDownloadImpl implements DownloadService {
  final String baseUrl;
  final String token;
  final String productCode;
  final bool needSTS;
  StreamSubscription<Map<String, Object>>? _subscription;

  OSSDownloadImpl({
    required this.baseUrl,
    required this.token,
    required this.productCode,
  }) : needSTS = true;

  OSSDownloadImpl.withoutToken({
    required this.baseUrl,
    required this.productCode,
  })  : token = "",
        needSTS = false;

  @override
  Future<void> download({
    required String url,
    DownloadProgressCallback? onProgress,
    DownloadSuccessCallback? onSuccess,
    DownloadErrorCallback? onError,
  }) async {
    try {
      String? path = await createDir();
      if (path == null) {
        onError?.call(Exception("下载路径错误，请稍后重新"));
        return;
      }
      File apkFile = File("$path/${getFileName(url)}");
      File apkFileTmp = File("$path/${getFileName(url)}.tmp");
      // 判断localPath 下有没有下载完成的文件
      if (apkFile.existsSync() && !apkFileTmp.existsSync()) {
        onSuccess?.call(apkFile.path);
        return;
      }
      // 删除历史下载文件
      await deleteFiles(Directory(path));

      _subscription =
          ApkUpdatePlatform.instance.addDownloadListener().listen((event) {
        var count = event['currentSize'] as num;
        var total = event['totalSize'] as num;
        var isDownloadSuccess = event['isDownloadSuccess'] == "1";
        var isDownloadFailed = event['isDownloadSuccess'] == "0";

        if (isDownloadFailed) {
          // 下载失败
          onError?.call(Exception("oss下载失败"));
          return;
        }
        if (isDownloadSuccess) {
          // 下载成功
          onSuccess?.call(apkFile.path);
        }
        final progress = count / total;
        onProgress?.call(progress); // 下载进度
      });

      final objectKey = getObjectKey(url);
      if (needSTS) {
        var ossInfo = await getOssInfo();
        if (ossInfo == null) {
          onError?.call(Exception("oss信息获取失败，请稍后重新"));
          return;
        }
        // 使用OSS下载
        var ak = ossInfo?['onceAccessKeyId'];
        var sk = ossInfo?["onceAccessKeySecret"];
        var token = ossInfo?["onceSecurityToken"];
        ApkUpdatePlatform.instance.downloadApk(
            ak,
            sk,
            token,
            objectKey,
            Constants.ossEndpoint,
            getOssBucketName(),
            path,
            getFileName(objectKey));
      } else {
        ApkUpdatePlatform.instance.downloadApkWithoutSTS(
            'LTAI5t9tqPPGM2g8YE8g899t',
            'ENGYeR4r5TQdeYz7C51KBsrwE89g7d',
            objectKey,
            Constants.ossEndpoint,
            getOssBucketName(),
            path,
            getFileName(objectKey));
      }
    } catch (e) {
      debugPrint(e.toString());
      onError?.call(Exception(e.toString()));
    }
  }

  @override
  void cancel() {
    _subscription?.cancel();
  }

  /// 获取oss信息
  Future<dynamic> getOssInfo() async {
    var params = {"extension": "apk", "bucketName": getOssBucketName()};
    var headers = {"token": token, "product-code": productCode};
    var result = await DioUpdateUtil.get("$baseUrl${ServerApi.generateByExt}",
        queryParameters: params, headers: headers);
    return result;
  }

  /// 处理apk下载url 获取 objectKey
  String getObjectKey(String url) {
    int startIndex = url.indexOf("aliyuncs.com/") + "aliyuncs.com/".length;
    return url.substring(startIndex);
  }

  /// 获取下载url文件名
  String getFileName(String url) {
    return p.basename(url);
  }

  /// 获取objectKey中的文件夹字符 “msl/common/prod/video/2025-06-20/app-release11.apk” 中获取 “msl/common/prod/video/2025-06-20”
  String getPathByObjectKey(String url) {
    return p.dirname(url);
  }

  /// 获取oss bucketName
  String getOssBucketName() {
    if (baseUrl.contains("gateway.devclx.cn") ||
        baseUrl.contains("gateway.testclx.cn") ||
        baseUrl.contains("gateway.preclx.cn")) {
      return "clx-dev";
    } else {
      return "clx-prod";
    }
  }
}
