import 'dart:async';
import 'dart:io';
import 'package:apk_update/core/common/version_info.dart';
import 'package:apk_update/utils/toast_util.dart';
import 'package:apk_update/utils/utils.dart';
import 'package:get/get.dart';
import 'apk_update_platform_interface.dart';
import 'core/common/app_upgrade_config.dart';
import 'core/services/download_service/download_service.dart';
import 'core/services/version_service/version_service.dart';
import 'widget/update_dialog.dart';

/// 升级逻辑
class AppUpgradeControl {
  /// 版本
  final VersionService versionService;

  /// 下载
  final DownloadService downloadService;

  /// 配置
  final AppUpgradeConfig config;

  AppUpgradeControl({
    required this.versionService,
    required this.downloadService,
    required this.config,
  });

  /// 检查更新
  Future<void> checkUpdate() async {
    // 检查更新，获取新版本详情
    final versionInfo = await versionService.checkVersion();
    if (versionInfo == null) return;
    // 升级弹框提示
    _showUpgradeDialog(versionInfo, config.appleId);
  }

  /// 检查下载
  void _showUpgradeDialog(VersionInfo versionInfo, String? appleId) async {
    Get.dialog(
      AppUpgradeDialog(
        title: versionInfo.name,
        content: versionInfo.content,
        versionForce: versionInfo.versionForce == true,
        topImageProvider: config.topImageProvider,
        onAppMarket: (String? url) {
          if (Platform.isIOS) {
            jumpAppStore(appleId ?? "");
          } else {
            openBrowser(url ?? "");
          }
        },
        onDownLoad: () {
          if (versionInfo.versionPath == null) {
            ToastUtil.showToast("下载地址错误，请稍后重新");
            return;
          }
          _downloadApk(versionInfo.versionPath ?? "");
        },
      ),
    );
  }

  /// 下载apk
  void _downloadApk(String downloadUrl) {
    findOtherLogic<AppUpgradeDialogLogic>()?.updateDownload(true);
    downloadService.download(
      url: downloadUrl,
      onProgress: (progress) {
        findOtherLogic<AppUpgradeDialogLogic>()?.updateProgress(progress);
      },
      onSuccess: (path) {
        // 下载完成，去安装
        findOtherLogic<AppUpgradeDialogLogic>()?.updateDownload(false);
        ApkUpdatePlatform.instance.installApk(path);
      },
      onError: (Exception error) {
        findOtherLogic<AppUpgradeDialogLogic>()?.updateDownload(false);
        ToastUtil.showToast(error.toString());
      },
    );
  }
}
