package com.clx.apk_update

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.util.Log
import androidx.annotation.NonNull
import com.clx.apk_update.DownloadAppUtil.aliDownload

import io.flutter.embedding.engine.plugins.FlutterPlugin
import io.flutter.plugin.common.MethodCall
import io.flutter.plugin.common.MethodChannel
import io.flutter.plugin.common.MethodChannel.MethodCallHandler
import io.flutter.embedding.engine.plugins.activity.ActivityAware
import io.flutter.embedding.engine.plugins.activity.ActivityPluginBinding
import io.flutter.plugin.common.EventChannel
import java.io.File

/** ApkUpdatePlugin */
class ApkUpdatePlugin : FlutterPlugin, MethodCallHandler, EventChannel.StreamHandler,
    ProgressCallback, ActivityAware {
    /// The MethodChannel that will the communication between Flutter and native Android
    ///
    /// This local reference serves to register the plugin with the Flutter Engine and unregister it
    /// when the Flutter Engine is detached from the Activity
    private lateinit var channel: MethodChannel
    private lateinit var eventChannel: EventChannel
    private val tag = "ApkUpdatePlugin"
    private lateinit var mContext: Context
    private var lastTime = 0L
    private var mEventSink: EventChannel.EventSink? = null
    private var mActivity: Activity? = null

    override fun onAttachedToEngine(@NonNull flutterPluginBinding: FlutterPlugin.FlutterPluginBinding) {
        mContext = flutterPluginBinding.applicationContext
        channel = MethodChannel(flutterPluginBinding.binaryMessenger, "apk_update")
        eventChannel = EventChannel(flutterPluginBinding.binaryMessenger, "download_install_event")
        channel.setMethodCallHandler(this)
        eventChannel.setStreamHandler(this)
    }

    override fun onMethodCall(@NonNull call: MethodCall, @NonNull result: MethodChannel.Result) {
        Log.d(tag, "onMethodCall: method = ${call.method} arguments = ${call.arguments}")
        when (call.method) {
            "installApk" -> {
                val path = call.argument<String?>("path")
                if (path == null || path == "") {
                    Log.d(tag, "onMethodCall: path is null")
                    result.error("error", "path is null", null)
                    return
                }
                openFile(path)
            }

            "downloadInstall" -> {
                val data = call.arguments<Map<String, String>>()
                val ak = data?.get("ak")
                val sk = data?.get("sk")
                val token = data?.get("token")
                val endpoint = data?.get("endpoint")
                val bucketName = data?.get("bucketName")
                val objectKey = data?.get("objectKey")
                val localPath = data?.get("localPath")
                val localFileName = data?.get("localFileName")
                if (ak == null || sk == null || token == null || endpoint == null || bucketName == null || objectKey == null || localPath == null || localFileName == null) {
                    result.error("error", "data is null", null)
                    return
                }
                Log.d(
                    "InstallAPKPlugin",
                    "setupMethodChannel: $ak $sk $token $endpoint $bucketName $objectKey $localPath"
                )
                aliDownload(
                    mContext,
                    ak,
                    sk,
                    token,
                    endpoint,
                    bucketName,
                    objectKey,
                    localPath,
                    localFileName,
                    this
                )

            }

            else -> {
                result.notImplemented()
            }
        }
    }

    override fun onListen(arguments: Any?, events: EventChannel.EventSink?) {
        mEventSink = events
    }

    override fun onCancel(arguments: Any?) {
        Log.d("InstallAPKPlugin", "onCancel: ")
    }

    /**
     * 安装 文件（APK）
     */
    private fun openFile(path: String) {
        val intents = Intent()
        intents.action = Intent.ACTION_VIEW
        intents.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        FileProvider7.setIntentDataAndType(
            mContext, intents, "application/vnd.android.package-archive", File(path), false
        )
        mContext.startActivity(intents)
    }

    override fun onDetachedFromEngine(@NonNull binding: FlutterPlugin.FlutterPluginBinding) {
        channel.setMethodCallHandler(null)
    }

    override fun onProgress(currentSize: Long, totalSize: Long) {

        // 主线程发送 event 每1s 发一次
        val time = System.currentTimeMillis()
        if (time - lastTime < 1000) {
            return
        }
        lastTime = time
        mActivity?.runOnUiThread {
            mEventSink?.success(
                mapOf(
                    "currentSize" to currentSize,
                    "totalSize" to totalSize
                )
            )

        }

    }

    override fun success(path: String) {
        mActivity?.runOnUiThread {
            mEventSink?.success(
                mapOf(
                    "currentSize" to 1,
                    "totalSize" to 1,
                    "isDownloadSuccess" to "1"
                )
            )

        }
        openFile(path)
    }

    override fun onFailed() {
        mActivity?.runOnUiThread {
            mEventSink?.success(
                mapOf(
                    "currentSize" to 1,
                    "totalSize" to 1,
                    "isDownloadSuccess" to "0"
                )
            )

        }
    }

    override fun onAttachedToActivity(binding: ActivityPluginBinding) {
        mActivity = binding.activity
    }

    override fun onDetachedFromActivityForConfigChanges() {
    }

    override fun onReattachedToActivityForConfigChanges(binding: ActivityPluginBinding) {
    }

    override fun onDetachedFromActivity() {
    }
}
