import 'dart:convert';
import 'package:apk_update/utils/toast_util.dart';
import 'package:dio/dio.dart';
import 'package:flutter/material.dart';

import 'connectivity_util.dart';

class DioUpdateUtil {
  static final Dio _dio = _initDio();

  DioUpdateUtil._internal();

  static Dio get dio => _dio;

  static Dio _initDio() {
    final dio = Dio();
    // 基础配置
    dio.options = BaseOptions(
        connectTimeout: const Duration(seconds: 15),
        receiveTimeout: const Duration(seconds: 15),
        contentType: Headers.jsonContentType,
        responseType: ResponseType.plain);
    dio.interceptors.add(LogInterceptor(
      requestBody: true,
      responseBody: true,
    ));
    return dio;
  }

  // GET请求
  static Future<dynamic> get(
    String path, {
    Map<String, dynamic>? queryParameters,
    Map<String, dynamic>? headers,
  }) async {
    if (!await checkNetWorker()) {
      return null;
    }
    try {
      final Response response = await _dio.get(
        path,
        queryParameters: queryParameters,
        options: Options(headers: headers),
      );
      return successCallbackHandler(response);
    } on DioException catch (e) {
      debugPrint(e.message);
      ToastUtil.showToast(e.message);
      return null;
    }
  }

  // POST请求
  static Future<dynamic> post(
    String path, {
    dynamic data,
    Map<String, dynamic>? queryParameters,
    Map<String, dynamic>? headers,
  }) async {
    if (!await checkNetWorker()) {
      return null;
    }
    try {
      final Response response = await _dio.post(
        path,
        data: data,
        queryParameters: queryParameters,
        options: Options(headers: headers),
      );
      return successCallbackHandler(response);
    } on DioException catch (e) {
      debugPrint(e.message);
      ToastUtil.showToast(e.message);
      return null;
    }
  }

  /// 请求成功处理
  static dynamic successCallbackHandler(Response response) {
    var responseData = response.data;
    if (responseData.runtimeType == String) {
      // 兼容后端数据返回string
      responseData = json.decode(responseData);
    }
    if (responseData?["code"] != 0) {
      ToastUtil.showToast(responseData?["msg"]);
      return null;
    }
    return responseData?["data"];
  }
}

/// 网络检测
Future<bool> checkNetWorker() async {
  // 判断网络是否连接
  var resList = await connectivity.checkConnectivity();

  if (resList.isNotEmpty && resList[0] == ConnectivityResult.none) {
    ToastUtil.showToast("网络异常,请检查网络");
    return false;
  }
  return true;
}
