import 'dart:io';

import 'package:apk_update/utils/image_utils.dart';
import 'package:flustars_flutter3/flustars_flutter3.dart';
import 'package:flutter/material.dart';
import 'package:flutter_widget_from_html_core/flutter_widget_from_html_core.dart';
import 'package:get/get.dart';

import '../core/common/constants.dart';

/// 升级弹框（①强制升级不关闭弹框）
class AppUpgradeDialog extends StatelessWidget {
  final logic = Get.put(AppUpgradeDialogLogic());

  /// 升级版本
  final String? title;

  /// 升级内容
  final String? content;

  /// 是否强升
  final bool versionForce;

  /// 下载apk
  final VoidCallback? onDownLoad;

  /// 跳转应用市场
  final ValueChanged<String?>? onAppMarket;

  /// 顶部图片
  final ImageProvider? topImageProvider;

  AppUpgradeDialog({
    Key? key,
    this.title,
    this.content,
    this.versionForce = false,
    this.onDownLoad,
    this.onAppMarket,
    this.topImageProvider,
  }) : super(key: key);

  @override
  Widget build(BuildContext context) {
    return GetBuilder<AppUpgradeDialogLogic>(builder: (logic) {
      return PopScope(
        canPop: false,
        child: Scaffold(
          resizeToAvoidBottomInset: false,
          backgroundColor: Colors.transparent,
          body: Center(
            child: SizedBox(
              width: 280.0,
              child: Column(
                mainAxisSize: MainAxisSize.min,
                children: <Widget>[
                  // 顶部图片
                  _topImage(),
                  Container(
                    padding: const EdgeInsets.symmetric(horizontal: 15.0),
                    decoration: const BoxDecoration(
                      color: Colors.white,
                      borderRadius: BorderRadius.only(
                        bottomLeft: Radius.circular(8.0),
                        bottomRight: Radius.circular(8.0),
                      ),
                    ),
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        // 标题
                        _content(title: title, content: content),
                        logic.isDownload
                            // 下载进度
                            ? LinearProgressIndicator(
                                backgroundColor: const Color(0xFFF2F3F3),
                                value: logic.downloadProgress,
                              )
                            // 按钮
                            : Column(
                                children: <Widget>[
                                  _btn(
                                    context: context,
                                    title: '立即更新',
                                    onTap: () {
                                      if (Platform.isIOS) {
                                        onAppMarket?.call(null);
                                      } else {
                                        onDownLoad?.call();
                                      }
                                    },
                                  ),
                                  const SizedBox(height: 5.0),
                                  Offstage(
                                    offstage: versionForce,
                                    child: _btn(
                                      context: context,
                                      title: '忽略此版本',
                                      textColor: const Color(0xFF666666),
                                      bgColor: Colors.transparent,
                                      onTap: () {
                                        //缓存当前时间
                                        String current = DateUtil.formatDate(
                                            DateTime.now(),
                                            format: Constants.dateFormat);
                                        SpUtil.putString(
                                            Constants.spCurrentDay, current);
                                        Navigator.pop(context);
                                      },
                                    ),
                                  ),
                                ],
                              ),
                        const SizedBox(height: 10.0),
                      ],
                    ),
                  ),
                ],
              ),
            ),
          ),
        ),
      );
    });
  }

  /// 顶部图片
  Widget _topImage() => Container(
        height: 140.0,
        width: 280.0,
        decoration: BoxDecoration(
          borderRadius: const BorderRadius.only(
            topLeft: Radius.circular(8.0),
            topRight: Radius.circular(8.0),
          ),
          image: DecorationImage(
            image: topImageProvider ??
                ImageUtils.getAssetImage('update_head', package: "apk_update"),
            fit: BoxFit.cover,
          ),
        ),
      );

  /// 标题、内容
  Widget _content({String? title, String? content}) => Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          const SizedBox(height: 15.0),
          // 标题
          Text(
            title ?? "",
            style: const TextStyle(
              fontSize: 15.0,
              color: Color(0xFF333C4C),
            ),
          ),
          const SizedBox(height: 5.0),
          // 内容
          htmlWidget(content),
          const SizedBox(height: 15.0),
        ],
      );

  /// 显示html内容
  Widget htmlWidget(String? content) {
    return HtmlWidget(
      content ?? "",
      onTapUrl: (url) {
        onAppMarket?.call(url);
        return true;
      },
    );
  }

  /// 按钮
  Widget _btn({
    String? title,
    GestureTapCallback? onTap,
    Color? textColor,
    Color? bgColor,
    required BuildContext context,
  }) =>
      InkWell(
        onTap: onTap,
        child: Container(
          width: double.infinity,
          constraints: const BoxConstraints(minHeight: 40.0),
          alignment: Alignment.center,
          decoration: BoxDecoration(
            color: bgColor ?? Theme.of(context).primaryColor,
            borderRadius: BorderRadius.circular(4.0),
          ),
          child: Text(
            title ?? "",
            style: TextStyle(fontSize: 15.0, color: textColor ?? Colors.white),
          ),
        ),
      );
}

class AppUpgradeDialogLogic extends GetxController {
  /// 是否正在下载
  bool isDownload = false;

  /// 下载进度
  double downloadProgress = 0.0;

  void updateProgress(double progress) {
    downloadProgress = progress;
    update();
  }

  void updateDownload(bool isDownload) {
    this.isDownload = isDownload;
    update();
  }
}
