import 'package:account_center/page/account_company/company_auth/widget/common/gesture_tap.dart';
import 'package:account_center/style/style_params.dart';
import 'package:account_center/widget/tree_node/tree_node.dart';
import 'package:flutter/material.dart';
import 'package:flutter_clx_base/flutter_clx_base.dart';

class TreeListWidget extends StatefulWidget {
  final List<TreeNode> positions;
  final String keyWork;

  /// 分组是否可以选择
  final TreeSelectType selectType;

  const TreeListWidget(this.positions,
      {Key? key, this.selectType = TreeSelectType.none, this.keyWork = ''})
      : super(key: key);

  @override
  State<TreeListWidget> createState() => _TreeListWidgetState();
}

class _TreeListWidgetState extends State<TreeListWidget> {
  @override
  Widget build(BuildContext context) {
    return ListView.builder(
        padding: EdgeInsets.only(top: 10),
        itemCount: widget.positions.length,
        itemBuilder: (context, index) {
          var positionData = widget.positions[index];
          return Visibility(
            visible: positionData.hasKey(widget.keyWork),
            child: Container(
              color: Colors.white,
              margin: EdgeInsets.only(bottom: 10),
              padding: EdgeInsets.symmetric(vertical: 12, horizontal: 10),
              child: getPositionItem(positionData, 0, index),
            ),
          );
        });
  }

  Widget getPositionItem(TreeNode node, int level, int baseIndex) {
    bool? isChecked = node.isChecked;
    bool expand = node.expand == true;
    List<dynamic> children = node.children;
    bool hasChildren = children.isNotEmpty;
    return Visibility(
      visible: node.hasKey(widget.keyWork),
      child: SizedBox(
        width: double.infinity,
        child: Column(
            children: [
          Row(
            children: [
              SizedBox(
                width: level == 0 ? 5 : level * 10 + 10,
              ),
              Visibility(
                visible: node.canCheck,
                child: Checkbox(
                    visualDensity: VisualDensity(vertical: -4, horizontal: -4),
                    fillColor: node.isForbid
                        ? MaterialStatePropertyAll(Colors.grey)
                        : MaterialStatePropertyAll(StyleParams.themeColor),
                    tristate:
                        widget.selectType == TreeSelectType.normalTristate,
                    onChanged: (value) {
                      if (node.isForbid) {
                        return;
                      }
                      checkItem(node, isChecked, baseIndex);
                    },
                    value: isChecked),
              ),
              SizedBox(
                width: 3,
              ),
              Expanded(
                child: GestureTapWidget(
                  behavior: HitTestBehavior.opaque,
                  onTap: () {
                    if (node.canExpand) {
                      expandItem(expand, node);
                    }
                  },
                  child: Container(
                    padding: EdgeInsets.symmetric(vertical: 10),
                    child: Row(
                      mainAxisAlignment: MainAxisAlignment.spaceBetween,
                      children: [
                        Expanded(
                          child: Text(
                            node.title,
                            maxLines: 1,
                            overflow: TextOverflow.ellipsis,
                          ),
                        ),
                        Visibility(
                          visible: node.canExpand,
                          child: Transform.rotate(
                            angle: expand ? 90.0 * 0.0174533 : 0,
                            child: Icon(
                              Icons.arrow_forward_ios_rounded,
                              size: 14,
                              color: Color(0xffDDE0E5),
                            ),
                          ),
                        )
                      ],
                    ),
                  ),
                ),
              ),
            ],
          ),
          Divider(
            color: Color(0xffE1E1E2),
            height: 0.5,
          ),
        ]..addIf(
                hasChildren,
                Visibility(
                  visible: expand,
                  child: Column(
                    children: getChildList(children, level + 1, baseIndex),
                  ),
                ))),
      ),
    );
  }

  List<Widget> getChildList(List<dynamic> positions, int level, int baseIndex) {
    var result = <Widget>[];
    for (var element in positions) {
      result.add(getPositionItem(element, level, baseIndex));
    }
    return result;
  }

  void checkItem(TreeNode node, bool? isChecked, int baseIndex) {
    switch (widget.selectType) {
      case TreeSelectType.normalTristate:
        checkGroupItem(node, isChecked, baseIndex);
        break;
      case TreeSelectType.productMenu:
        checkProductMenuItem(node, isChecked, baseIndex);
        break;
      case TreeSelectType.none:
        node.isChecked = isChecked != true;
        break;
      case TreeSelectType.single:
        for (var element in widget.positions) {
          element.isChecked = false;
          setChildrenChecked(element, false);
        }
        node.isChecked = true;
        break;
    }
    setState(() {});
  }

  void expandItem(bool expand, TreeNode node) {
    node.expand = !expand;
    setState(() {});
  }

  /// 产品菜单选择逻辑
  void checkProductMenuItem(TreeNode node, bool? isChecked, int baseIndex) {
    bool result = isChecked != true;
    node.isChecked = result;
    setChildrenChecked(node, result);
    if (result) {
      node.checkParent();
    }
    setState(() {});
  }

  /// 正常三态选择
  void checkGroupItem(TreeNode node, bool? isChecked, int baseIndex) {
    bool result = false;
    if (isChecked == null) {
      result = false;
    } else if (isChecked == true) {
      result = false;
    } else {
      result = true;
    }
    node.isChecked = result;
    setChildrenChecked(node, result);
    setParentChecked(baseIndex, node);
  }

  /// 设置字节点的选择状态
  void setChildrenChecked(TreeNode node, bool checked) {
    for (var element in node.children) {
      element.isChecked = checked;
      setChildrenChecked(element, checked);
    }
  }

  /// 更新父节点的选择状态
  void setParentChecked(int baseIndex, TreeNode node) {
    var basePosition = widget.positions.getElement(baseIndex);
    var children = basePosition?.children ?? [];
    if (children.isEmpty) {
      basePosition?.isChecked = node.isChecked == true;
    } else {
      basePosition?.isChecked = getChildCheck(children);
    }
  }

  bool? getChildCheck(List<TreeNode> children) {
    bool hasChecked = false;
    bool hasUnChecked = false;
    bool hasNull = false;
    for (var element in children) {
      bool? result;
      if (element.children.isNotEmpty) {
        result = getChildCheck(element.children);
        element.isChecked = result;
      } else {
        result = element.isChecked == true;
      }
      if (result == true) {
        hasChecked = true;
      } else if (result == false) {
        hasUnChecked = true;
      } else {
        hasNull = true;
      }
    }

    // 全部选中状态
    if (hasChecked && !hasUnChecked && !hasNull) {
      return true;
    }
    // 全部没有选中
    else if (hasUnChecked && !hasChecked && !hasNull) {
      return false;
    } else {
      return null;
    }
  }
}

enum TreeSelectType {
  /// 正常的三种状态选择
  normalTristate,

  /// 产品菜单独有逻辑
  productMenu,

  /// 无逻辑
  none,

  single
}
