import 'dart:io';

import 'package:device_info_plus/device_info_plus.dart';

class DeviceInfo {
  String deviceId;
  String deviceName;

  /// 系统版本
  String systemVersion;

  /// 手机品牌
  String brand;

  DeviceInfo(
      {required this.deviceId,
      required this.deviceName,
      required this.systemVersion,
      this.brand = ''});

  String getDeviceShowName() {
    if (Platform.isAndroid) {
      return '$brand-$deviceName';
    } else {
      return deviceName;
    }
  }

  // 根据 Android 设备信息创建 DeviceInfo 对象
  factory DeviceInfo.fromAndroidDeviceInfo(AndroidDeviceInfo androidInfo) {
    return DeviceInfo(
        deviceId: androidInfo.serialNumber,
        deviceName: androidInfo.model,
        systemVersion: androidInfo.version.release,
        brand: androidInfo.brand);
  }

  // 根据 iOS 设备信息创建 DeviceInfo 对象
  factory DeviceInfo.fromIosDeviceInfo(IosDeviceInfo iosInfo) {
    return DeviceInfo(
      deviceId: iosInfo.identifierForVendor ?? "",
      deviceName: AppleDeviceModelMapper.getModelName(iosInfo.utsname.machine),
      systemVersion: iosInfo.systemVersion,
    );
  }
}

// 获取设备信息的方法
Future<DeviceInfo> getDeviceInfo() async {
  DeviceInfoPlugin deviceInfoPlugin = DeviceInfoPlugin();

  if (Platform.isAndroid) {
    AndroidDeviceInfo androidInfo = await deviceInfoPlugin.androidInfo;
    return DeviceInfo.fromAndroidDeviceInfo(androidInfo);
  } else if (Platform.isIOS) {
    IosDeviceInfo iosInfo = await deviceInfoPlugin.iosInfo;
    return DeviceInfo.fromIosDeviceInfo(iosInfo);
  }

  // 如果不是 Android 或 iOS，返回一个默认的 DeviceInfo 对象
  return DeviceInfo(
    deviceId: 'Unknown',
    deviceName: 'Unknown',
    systemVersion: 'Unknown',
  );
}

class AppleDeviceModelMapper {
  static final Map<String, String> deviceModels = {
    "iPhone3,1": "iPhone 4",
    "iPhone3,2": "iPhone 4",
    "iPhone3,3": "iPhone 4",
    "iPhone4,1": "iPhone 4S",
    "iPhone5,1": "iPhone 5",
    "iPhone5,2": "iPhone 5",
    "iPhone5,3": "iPhone 5c",
    "iPhone5,4": "iPhone 5c",
    "iPhone6,1": "iPhone 5s",
    "iPhone6,2": "iPhone 5s",
    "iPhone7,1": "iPhone 6 Plus",
    "iPhone7,2": "iPhone 6",
    "iPhone8,1": "iPhone 6s",
    "iPhone8,2": "iPhone 6s Plus",
    "iPhone8,4": "iPhone SE (1st generation)",
    "iPhone9,1": "iPhone 7",
    "iPhone9,2": "iPhone 7 Plus",
    "iPhone9,3": "iPhone 7",
    "iPhone9,4": "iPhone 7 Plus",
    "iPhone10,1": "iPhone 8",
    "iPhone10,2": "iPhone 8 Plus",
    "iPhone10,3": "iPhone X",
    "iPhone10,4": "iPhone 8",
    "iPhone10,5": "iPhone 8 Plus",
    "iPhone10,6": "iPhone X",
    "iPhone11,2": "iPhone XS",
    "iPhone11,4": "iPhone XS Max",
    "iPhone11,6": "iPhone XS Max",
    "iPhone11,8": "iPhone XR",
    "iPhone12,1": "iPhone 11",
    "iPhone12,3": "iPhone 11 Pro",
    "iPhone12,5": "iPhone 11 Pro Max",
    "iPhone12,8": "iPhone SE (2nd generation)",
    "iPhone13,1": "iPhone 12 mini",
    "iPhone13,2": "iPhone 12",
    "iPhone13,3": "iPhone 12 Pro",
    "iPhone13,4": "iPhone 12 Pro Max",
    "iPhone14,4": "iPhone 13 mini",
    "iPhone14,5": "iPhone 13",
    "iPhone14,2": "iPhone 13 Pro",
    "iPhone14,3": "iPhone 13 Pro Max",
    "iPhone14,6": "iPhone SE 3",
    "iPhone14,7": "iPhone 14",
    "iPhone14,8": "iPhone 14 Plus",
    "iPhone15,2": "iPhone 14 Pro",
    "iPhone15,3": "iPhone 14 Pro Max",
    "iPhone15,4": "iPhone 14 Pro Max (China)"
  };

  static String getModelName(String identifier) {
    return deviceModels[identifier] ?? "Unknown Device";
  }
}
