import 'package:account_center/account_center.dart';
import 'package:account_center/common/utils/date_picker.dart' as datePicker;
import 'package:account_center/style/widget/button_public_radius.dart';
import 'package:account_center/widget/bottom_safe.dart';
import 'package:flutter/material.dart';

class TimeRangeSelectDialog extends StatefulWidget {
  final DateTime? startTime;
  final DateTime? endTime;
  final int beforeDays;
  final int afterDays;
  final datePicker.DateType dateType;

  /// 是否可为空
  final bool canEmpty;

  /// 最大区间 不能选空时生效
  final int maxRangDays;
  final Function(DateTime? startTime, DateTime? endTime)? onConfirm;

  const TimeRangeSelectDialog({
    Key? key,
    this.startTime,
    this.endTime,
    this.beforeDays = 30,
    this.afterDays = 30,
    this.onConfirm,
    this.dateType = datePicker.DateType.YMD_HM,
    this.maxRangDays = 100000,
    this.canEmpty = true,
  }) : super(key: key);

  @override
  State<TimeRangeSelectDialog> createState() => _TimeRangeSelectDialogState();
}

class _TimeRangeSelectDialogState extends State<TimeRangeSelectDialog> {
  String getShowTime(DateTime? time, {String format = 'yyyy-MM-dd HH:mm'}) {
    if (time == null) {
      return '';
    }
    return DateUtil.formatDate(time, format: format);
  }

  String get startTime => getShowTime(startDate, format: format);

  String get endTime => getShowTime(endDate, format: format);

  late datePicker.DateType dateType = widget.dateType;
  late DateTime? startDate = widget.startTime ?? DateTime.now();
  late DateTime? endDate = widget.endTime ?? DateTime.now();

  DateTime get minValue =>
      DateTime.now().subtract(Duration(days: widget.beforeDays));

  DateTime get maxValue => DateTime.now().add(Duration(days: widget.afterDays));

  @override
  Widget build(BuildContext context) {
    return Container(
      height: 450,
      width: double.infinity,
      decoration: const BoxDecoration(
        color: Colors.white,
        borderRadius: BorderRadius.vertical(
          top: Radius.circular(20),
        ),
      ),
      padding: const EdgeInsets.symmetric(horizontal: 20),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          const SizedBox(height: 30),
          _title('开始时间'),
          const SizedBox(height: 16),
          _selectLayout(
            content: startTime,
            hint: '开始时间',
            onTap: _selectStartTime,
          ),
          const SizedBox(height: 20),
          _title('结束时间'),
          const SizedBox(height: 16),
          _selectLayout(
            content: endTime,
            hint: '结束时间',
            onTap: _selectEndTime,
          ),
          Expanded(child: Container()),
          Row(
            children: [
              // Expanded(
              //     child: BorderButton(
              //   onPressed: _reset,
              //   title: '重置',
              //   height: 40,
              //   radius: 8,
              // )),
              // const SizedBox(width: 10),
              Expanded(
                  child: ButtonPublicRadius(
                height: 40,
                radius: 8,
                text: '确定',
                onPress: _confirm,
              )),
            ],
          ),
          const BottomSafeArea(),
        ],
      ),
    );
  }

  Widget _title(String title) {
    return Text(
      title,
      style: const TextStyle(
        fontSize: 16,
        fontWeight: FontWeight.w600,
        color: Color(0xFF344254),
      ),
    );
  }

  Widget _selectLayout({
    required String content,
    required String hint,
    required void Function() onTap,
  }) {
    return InkWell(
      onTap: onTap,
      child: Container(
        alignment: Alignment.centerLeft,
        height: 38,
        width: double.infinity,
        decoration: BoxDecoration(
          color: const Color(0xFFF3F5F8),
          borderRadius: BorderRadius.circular(8),
        ),
        padding: const EdgeInsets.only(left: 10),
        child: Text(
          content.isEmpty ? hint : content,
          style: TextStyle(
            color: content.isEmpty
                ? const Color(0xFFC9CDD4)
                : const Color(0xFF344254),
            fontSize: 14,
            fontWeight: content.isEmpty ? FontWeight.w400 : FontWeight.w500,
          ),
        ),
      ),
    );
  }

  String get format {
    switch (dateType) {
      case datePicker.DateType.YMD:
        return 'yyyy-MM-dd';
      case datePicker.DateType.YMD_HM:
      case datePicker.DateType.YM:
        return 'yyyy';
      case datePicker.DateType.YMD_H:
        return 'yyyy-MM-dd HH';
      case datePicker.DateType.YMD_HMS:
        return 'yyyy-MM-dd HH:mm:ss';
    }
  }

  void _selectStartTime() {
    Get.bottomSheet(datePicker.DatePicker(
      (date) {
        setState(() {
          startDate = date;
          if (endDate?.isBefore(startDate!) ?? false) {
            endDate = startDate;
          }
        });
      },
      selectData: startDate,
      minValue: minValue,
      maxValue: maxValue,
      type: dateType,
    ));
  }

  void _selectEndTime() {
    Get.bottomSheet(datePicker.DatePicker(
      (date) {
        setState(() {
          endDate = date;
          if (startDate?.isAfter(endDate!) ?? false) {
            startDate = endDate;
          }
        });
      },
      selectData: endDate,
      minValue: minValue,
      maxValue: maxValue,
      type: dateType,
    ));
  }

  void _reset() {
    if (widget.canEmpty) {
      setState(() {
        startDate = null;
        endDate = null;
      });
    } else {
      setState(() {
        startDate = DateTime.now();
        endDate = DateTime.now();
      });
    }
  }

  void _confirm() {
    Get.back();
    widget.onConfirm?.call(startDate, endDate);
  }
}
