import 'dart:io';

import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:flutter_clx_base/flutter_clx_base.dart';
import 'package:flutter_widget_from_html_core/flutter_widget_from_html_core.dart';
import 'package:open_filex/open_filex.dart';
import 'package:open_store/open_store.dart';

class UpdateDialog extends StatefulWidget {
  final String? title;
  final String? content;
  final bool? isUpdateMore;
  final String? versionPath;
  final String versionNumber;
  final Function()? onIgnore;
  final String? bgPath;
  final Color? mainColor;
  final String? iosAppId;
  const UpdateDialog(
      {Key? key,
      this.title,
      this.content,
      this.isUpdateMore,
      this.versionPath,
      this.onIgnore,
      this.bgPath,
      this.mainColor,
      this.iosAppId,
      required this.versionNumber})
      : super(key: key);

  @override
  State<UpdateDialog> createState() => _UpdateDialogState();
}

class _UpdateDialogState extends State<UpdateDialog> {
  final CancelToken _cancelToken = CancelToken();
  bool _isDownloading = false;
  final _value = 0.0.obs;
  late Color primaryColor;

  @override
  void initState() {
    super.initState();
  }

  @override
  void dispose() {
    if (!_cancelToken.isCancelled && _value.value != 1) {
      _cancelToken.cancel();
    }
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    primaryColor = widget.mainColor ?? Theme.of(context).primaryColor;
    return WillPopScope(
      onWillPop: () {
        /// 使用false禁止返回键返回，达到强制升级目的
        return Future.value(false);
      },
      child: Scaffold(
          resizeToAvoidBottomInset: false,
          backgroundColor: Colors.transparent,
          body: Center(
            child: Container(
                decoration: BoxDecoration(
                  color: Colors.white,
                  borderRadius: BorderRadius.circular(8.0),
                ),
                width: 280.0,
                child: Stack(
                  children: <Widget>[
                    Container(
                      height: 140.0,
                      width: 290.0,
                      decoration: BoxDecoration(
                        borderRadius: const BorderRadius.only(
                            topLeft: Radius.circular(8.0),
                            topRight: Radius.circular(8.0)),
                        image: DecorationImage(
                          image: AssetImage(widget.bgPath ?? ""),
                          fit: BoxFit.cover,
                        ),
                      ),
                    ),
                    Column(
                      mainAxisSize: MainAxisSize.min,
                      children: [
                        Padding(
                          padding: const EdgeInsets.only(
                              left: 15.0, right: 15.0, top: 16.0),
                          child: Text(
                            widget.title!,
                            style: const TextStyle(
                                fontSize: 18.0, fontWeight: FontWeight.w600),
                          ),
                        ),
                        Padding(
                          padding: const EdgeInsets.symmetric(
                              horizontal: 15.0, vertical: 10.0),
                          child: HtmlWidget(widget.content!),
                        ),
                        Padding(
                          padding: const EdgeInsets.only(
                              left: 15.0, right: 15.0, top: 5.0),
                          child: _isDownloading
                              ? _progress(context, primaryColor)
                              : _buildButton(context),
                        ),
                      ],
                    ),
                  ],
                )),
          )),
    );
  }

  Widget _installButton(BuildContext context) {
    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 10, horizontal: 15),
      child: GFButton(
        color: primaryColor,
        size: 40,
        textStyle: const TextStyle(fontSize: 16.0),
        fullWidthButton: true,
        onPressed: () {
          getAPkName().then((value) {
            openApk(value ?? '');
          });
        },
        text: '文件已下载，点击安装',
      ),
    );
  }

  Obx _progress(BuildContext context, Color primaryColor) {
    return Obx(() {
      return Padding(
        padding: const EdgeInsets.only(bottom: 15.0),
        child: Column(
          children: [
            LinearProgressIndicator(
              backgroundColor: Colors.grey,
              valueColor: AlwaysStoppedAnimation<Color>(primaryColor),
              value: _value.value,
            ),
            vGap8,
            Center(
              child: Text('${(_value.value * 100).toInt()}%'),
            )
          ],
        ),
      );
    });
  }

  ///下载apk
  Future<void> _download(String path) async {
    try {
      File file = File(path);

      /// 链接可能会失效
      await Dio().download(
        widget.versionPath!,
        file.path,
        cancelToken: _cancelToken,
        onReceiveProgress: (int count, int total) {
          if (total != -1) {
            _value.value = count / total;
            if (count == total) {
              Navigator.pop(context);
              openApk(path);
            }
          }
        },
      );
    } catch (e) {
      ToastUtil.showToast('下载失败');
      loggerNoStack.d(e);
      setState(() {
        _isDownloading = false;
      });
    }
  }

  Future<String?> getAPkName() async {
    setInitDir(initStorageDir: true);
    await DirectoryUtil.getInstance();
    DirectoryUtil.createStorageDirSync(category: 'Download');
    return DirectoryUtil.getStoragePath(
        fileName: 'report_${widget.versionNumber}',
        category: 'Download',
        format: 'apk');
  }

  void _toUpdate() {
    if (defaultTargetPlatform == TargetPlatform.iOS) {
      Navigator.pop(context);
      OpenStore.instance.open(appStoreId: widget.iosAppId ?? '');
    } else if (defaultTargetPlatform == TargetPlatform.android) {
      getAPkName().then((path) async {
        setState(() {
          _isDownloading = true;
        });
        await _download(path ?? '');
      });
    }
  }

  openApk(String path) async {
    OpenResult openResult = await OpenFilex.open(path);
    // final openResult = await OpenFile.open(path);
    if (openResult.type == ResultType.error) {
      ToastUtil.showToast(openResult.message);
    } else if (openResult.type == ResultType.permissionDenied) {
      ToastUtil.showToast("无权限");
    } else if (openResult.type == ResultType.fileNotFound) {
      ToastUtil.showToast("未发现文件");
    } else if (openResult.type == ResultType.noAppToOpen) {
      ToastUtil.showToast("无打开方式");
    } else {
      // ToastUtil.showToast(openResult.message);
    }
  }

  Widget _buildButton(BuildContext context) {
    return Column(
      children: <Widget>[
        GFButton(
          color: primaryColor,
          size: 40,
          textStyle: const TextStyle(fontSize: 16.0),
          fullWidthButton: true,
          onPressed: () {
            _toUpdate();
          },
          text: '立即更新',
        ),
        Offstage(offstage: !widget.isUpdateMore!, child: vGap10),
        Offstage(
          offstage: widget.isUpdateMore!,
          child: TextButton(
            onPressed: () {
              widget.onIgnore?.call();
              Navigator.pop(context);
            },
            child: const Text(
              '忽略此版本',
              style: TextStyle(color: Color(0xFF666666), fontSize: 14.0),
            ),
          ),
        ),
      ],
    );
  }
}
