package com.clx.performance.service.impl.vehiclewarn;

import com.clx.performance.dao.vehiclewarn.VehicleWarnConfigDao;
import com.clx.performance.dao.vehiclewarn.VehicleWarnInfoDao;
import com.clx.performance.enums.vehiclewarn.VehicleWarnConfigEnum;
import com.clx.performance.enums.vehiclewarn.VehicleWarnInfoEnum;
import com.clx.performance.model.OrderChild;
import com.clx.performance.model.vehiclewarn.VehicleWarnChild;
import com.clx.performance.model.vehiclewarn.VehicleWarnConfig;
import com.clx.performance.model.vehiclewarn.VehicleWarnInfo;
import com.clx.performance.service.vehiclewarn.VehicleWarnCommonService;
import com.clx.performance.service.vehiclewarn.VehicleWarnLoadAndUnloadWarnService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;

@Slf4j
@Service
public class VehicleWarnLoadAndUnloadWarnServiceImpl implements VehicleWarnLoadAndUnloadWarnService {

    private static final Integer MIN_FER_HOUR = 60;
    @Autowired
    private VehicleWarnConfigDao vehicleWarnConfigDao;
    @Autowired
    private VehicleWarnInfoDao vehicleWarnInfoDao;
    @Autowired
    private VehicleWarnCommonService vehicleWarnCommonService;

    /**
     * 装车超时
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void loadTimeout(OrderChild orderChild, VehicleWarnChild vehicleWarnChild){

        List<VehicleWarnConfig> vehicleWarnConfigList = vehicleWarnConfigDao
                .listByWarnType(VehicleWarnConfigEnum.WarnType.LOAD_TIMEOUT.getCode());
        if (vehicleWarnConfigList.isEmpty()) {return;}

        for (VehicleWarnConfig item : vehicleWarnConfigList) {
            doLoadTimeout(orderChild, vehicleWarnChild, item);
        }

    }

    private void doLoadTimeout(OrderChild orderChild, VehicleWarnChild vehicleWarnChild
            , VehicleWarnConfig vehicleWarnConfig){
        VehicleWarnInfo vehicleWarnInfo = vehicleWarnInfoDao
                .findByChildNoAndWarnConfigId(vehicleWarnChild.getChildNo(), vehicleWarnConfig.getId()).orNull();
        if (vehicleWarnInfo != null
                && Objects.equals(vehicleWarnInfo.getStatus(), VehicleWarnInfoEnum.Status.RESOLVE.getCode())){
            return;
        }

        // 暂停检测
        boolean suspend = vehicleWarnCommonService.suspendCheck(vehicleWarnChild, vehicleWarnConfig.getId());
        if (suspend) {return;}

        // 超时判断
        int time = vehicleWarnConfig.getTimeoutRatio().multiply(new BigDecimal(MIN_FER_HOUR)).intValue();
        if (orderChild.getArriveSendTime().plusMinutes(time).isAfter(LocalDateTime.now())){return;}

        // 更新
        vehicleWarnCommonService.vehicleWarnInfoUpdate(orderChild, vehicleWarnInfo, vehicleWarnConfig);

    }

    /**
     * 到达货源地超时
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void unloadTimeout(OrderChild orderChild, VehicleWarnChild vehicleWarnChild){

        List<VehicleWarnConfig> vehicleWarnConfigList = vehicleWarnConfigDao
                .listByWarnType(VehicleWarnConfigEnum.WarnType.UNLOAD_TIMEOUT.getCode());
        if (vehicleWarnConfigList.isEmpty()) {return;}

        for (VehicleWarnConfig item : vehicleWarnConfigList) {
            doUnloadTimeout(orderChild, vehicleWarnChild, item);
        }

    }

    private void doUnloadTimeout(OrderChild orderChild, VehicleWarnChild vehicleWarnChild,
                                 VehicleWarnConfig vehicleWarnConfig){
        VehicleWarnInfo vehicleWarnInfo = vehicleWarnInfoDao
                .findByChildNoAndWarnConfigId(vehicleWarnChild.getChildNo(), vehicleWarnConfig.getId()).orNull();
        if (vehicleWarnInfo != null
                && Objects.equals(vehicleWarnInfo.getStatus(), VehicleWarnInfoEnum.Status.RESOLVE.getCode())){
            return;
        }

        // 暂停检测
        boolean suspend = vehicleWarnCommonService.suspendCheck(vehicleWarnChild, vehicleWarnConfig.getId());
        if (suspend) {return;}

        // 超时判断
        int time = vehicleWarnConfig.getTimeoutRatio().multiply(new BigDecimal(MIN_FER_HOUR)).intValue();
        if (orderChild.getArriveReceiveTime().plusMinutes(time).isAfter(LocalDateTime.now())){return;}

        // 更新
        vehicleWarnCommonService.vehicleWarnInfoUpdate(orderChild, vehicleWarnInfo, vehicleWarnConfig);
    }

}
