package com.clx.performance.strategy.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.clx.order.feign.OrderFeign;
import com.clx.order.param.feign.UpdateOrderInfoResidueWeightParam;
import com.clx.order.params.OrderGoodsChildParams;
import com.clx.order.params.OrderGoodsParams;
import com.clx.order.vo.feign.FeignOrderVO;
import com.clx.order.vo.feign.OrderOwnTruckVo;
import com.clx.performance.component.GoodsOrderStrategyContext;
import com.clx.performance.component.GoodsOrderTruckRecordComponent;
import com.clx.performance.component.OrderGoodsIdGenerate;
import com.clx.performance.component.OrderGoodsStatusLazyComponent;
import com.clx.performance.constant.RabbitKeyConstants;
import com.clx.performance.dao.OrderGoodsDao;
import com.clx.performance.dao.OrderGoodsTruckBindDao;
import com.clx.performance.enums.OrderGoodsStatusEnum;
import com.clx.performance.enums.PendingOrderWayStatusEnum;
import com.clx.performance.enums.PerformanceResultEnum;
import com.clx.performance.enums.TruckDemandEnum;
import com.clx.performance.model.OrderGoods;
import com.clx.performance.strategy.GoodsOrderStrategy;
import com.msl.common.exception.ServiceSystemException;
import com.msl.common.utils.DateUtils;
import com.msl.user.data.UserSessionData;
import com.msl.user.utils.TokenUtil;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageBuilder;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 自由车辆
 */
@Component("ThreeGoodsOrderStrategy")
@Slf4j
@AllArgsConstructor
public class ThreeGoodsOrderStrategy extends GoodsOrderStrategy implements InitializingBean {

    private final OrderGoodsIdGenerate orderGoodsIdGenerate;

    private final OrderGoodsTruckBindDao orderGoodsTruckBindDao;

    private final OrderGoodsDao orderGoodsDao;

    private final GoodsOrderStrategyContext goodsOrderStrategyContext;

    private final OrderFeign orderFeign;

    private final RabbitTemplate rabbitTemplate;

    private final OrderGoodsStatusLazyComponent orderGoodsStatusLazyComponent;

    private final GoodsOrderTruckRecordComponent goodsOrderTruckRecordComponent;


    @Override
    @Transactional(rollbackFor = Exception.class)
    public LocalDateTime goodsOrderProcess(OrderGoodsParams orderGoodsParams, FeignOrderVO orderInfo, LocalDateTime now) {
        String orderNo = orderGoodsParams.getOrderNo();
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();

        LocalDateTime sendLazyTime = null;
        List<OrderGoodsChildParams> childParamsList = orderGoodsParams.getOrderGoodsChildParams();
        BigDecimal childSum = childParamsList.stream().map(OrderGoodsChildParams::getExtractWeight).reduce(
                BigDecimal.ZERO, BigDecimal::add);
        BigDecimal residueWeight = new BigDecimal(orderInfo.getResidueWeight());
        if (residueWeight.compareTo(childSum) < 0) {
            throw new ServiceSystemException(PerformanceResultEnum.HTTP_ERROR, "当前货单总吨数已超订单总吨数");
        }
        if (childParamsList.size() > 1) {
            throw new ServiceSystemException(PerformanceResultEnum.HTTP_ERROR, "全部自有车辆只能全部提取");
        }
        Map<String, LocalDateTime> mqMap = new HashMap<>();

        long beginOrderGoodsId = orderGoodsIdGenerate.getOrderGoodsId(childParamsList.size());
        List<OrderGoods> orderGoodsList = new LinkedList<>();

        for (OrderGoodsChildParams child : childParamsList) {

            OrderGoods orderGoods = new OrderGoods();
            //提取吨数
            orderGoods.setExtractWeight(child.getExtractWeight());
            orderGoods.setResidueTransportWeight(child.getExtractWeight());
            orderGoods.setAlreadyTransportWeight(BigDecimal.ZERO);

            orderGoods.setLossPrice(orderInfo.getLossPrice());
            orderGoods.setTruckDemand(TruckDemandEnum.TruckType.OWNER_CAR.getCode());
            orderGoods.setVehicleUsage(Integer.valueOf(child.getVehicleUsage()));

            //提取方式 1提取全部 2提取部分
            Integer extractWay = child.getExtractWay();
            if (extractWay.equals(2)) {
                throw new ServiceSystemException(PerformanceResultEnum.HTTP_ERROR, "全部自有车辆只能全部提取");
            }
            orderGoods.setExtractWay(extractWay);
            //需要车辆
            orderGoods.setNeedTruckNum(child.getNeedTruckNum());
            //挂单方式
            orderGoods.setPendingOrderWay(child.getPendingOrderWay());
            //挂单时间
            LocalDateTime postedTime = DateUtils.parseDateTime(child.getPendingOrderTime()).get();

            if (sendLazyTime == null) {
                sendLazyTime = postedTime;
            } else if (sendLazyTime.isAfter(postedTime)) {
                sendLazyTime = postedTime;
            }
            orderGoods.setPendingOrderTime(postedTime);
            orderGoods.setPendingOrderFreight(child.getPendingOrderFreight());

            orderGoods.setLastArriveSendTime(DateUtils.parseDateTime(child.getLastArriveSendTime()).get());
            //定向运单失效时间
//                    orderGoods.setDirectionalExpireTime(DateUtils.parseDateTime(child.getDirectionalExpireTime()).get());
            orderGoods.setSeniorLogisticsManagerId(child.getSeniorLogisticsManagerId());
            orderGoods.setSeniorLogisticsManagerName(child.getSeniorLogisticsManagerName());

            orderGoods.setOrderNo(orderNo);
            Integer pendingOrderWay = orderGoods.getPendingOrderWay();
            String orderNoPrefix = PendingOrderWayStatusEnum.Status.GK.getCode().equals(pendingOrderWay) ? PendingOrderWayStatusEnum.Status.GK.getName() : PendingOrderWayStatusEnum.Status.DX.getName();
            String orderGoodsNo = orderNoPrefix + beginOrderGoodsId;
            orderGoods.setOrderGoodsNo(orderGoodsNo);

            orderGoods.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.CREATED.getCode());
            orderGoods.setSendAddressId(orderInfo.getSendAddressId());
            orderGoods.setSendAddressShorter(orderInfo.getSendAddressShorter());
            //收发货经纬度
            orderGoods.setSendLongitude(orderInfo.getSendLongitude());
            orderGoods.setSendLatitude(orderInfo.getSendLatitude());
            orderGoods.setReceiveLatitude(orderInfo.getReveiveLatitude());
            orderGoods.setReceiveLongitude(orderInfo.getReveiveLongitude());
            orderGoods.setReceiveAddressId(orderInfo.getReveiveAddressId());
            orderGoods.setReceiveAddressShorter(orderInfo.getReveiveAddressShorter());
            orderGoods.setGoodsName(orderInfo.getGoodsName());
            orderGoods.setGoodsId(orderInfo.getGoodsNameId());

            orderGoods.setCreateTime(now);
            orderGoods.setModifiedTime(now);
            orderGoods.setUserName(loginUserInfo.getUserName());
            orderGoods.setUserNo(loginUserInfo.getUserNo());

            //设置新加的三个最晚时间
            orderGoods.setLastOrderTime(DateUtils.parseDateTime(child.getLastOrderTime()).get());
            orderGoods.setLastLoadTime(DateUtils.parseDateTime(child.getLastLoadTime()).get());
            orderGoods.setLastArriveReceiveTime(DateUtils.parseDateTime(child.getLastArriveReceiveTime()).get());

            List<OrderOwnTruckVo> orderOwnTruckVos = orderFeign.selectTruckListFeign(orderNo);
            if (orderOwnTruckVos == null || orderOwnTruckVos.isEmpty()) {
                throw new ServiceSystemException(PerformanceResultEnum.HTTP_ERROR, "全部自由车辆列表数据为空");
            }
            List<String> truckList = orderOwnTruckVos.stream().map(OrderOwnTruckVo::getTruckNo).collect(Collectors.toList());
            if (truckList.isEmpty()) {
                throw new ServiceSystemException(PerformanceResultEnum.HTTP_ERROR, "全部自由车辆列表数据为空");

            }
            //车牌号集合
            if (CollectionUtil.isNotEmpty(truckList)) {
                orderGoodsTruckBindDao.saveBatchEntity(orderNo, orderGoodsNo, truckList, now);
                goodsOrderTruckRecordComponent.saveTruckRecord(orderGoodsNo, truckList);
            }
            mqMap.put(orderNoPrefix + beginOrderGoodsId, postedTime);
            orderGoodsStatusLazyComponent.expireProduce(orderGoods.getLastOrderTime(), orderGoodsNo);

            beginOrderGoodsId = beginOrderGoodsId + 1;

            orderGoodsList.add(orderGoods);
        }
        orderGoodsDao.saveBatchEntity(orderGoodsList);

        UpdateOrderInfoResidueWeightParam param = new UpdateOrderInfoResidueWeightParam();
        param.setOrderId(orderInfo.getId());
        param.setUpdateType("1");
        param.setResidueWeight(residueWeight.subtract(childSum));
        orderFeign.updateOrderInfoResidueWeight(param);
        sendMq(mqMap, now);
        return sendLazyTime;
    }

    @Override
    public void afterPropertiesSet() {
        goodsOrderStrategyContext.strategyContext.put(TruckDemandEnum.TruckType.OWNER_CAR.getCode(), this);
    }

    public void sendMq(Map<String, LocalDateTime> mqMap, LocalDateTime now) {
        log.info("执行sendMq方法,mqMap大小为：{}", mqMap.size());
        for (Map.Entry<String, LocalDateTime> entry : mqMap.entrySet()) {
            log.info("货单更改挂单中,发送货单更改挂单中延迟消息,时间:{},orderGoodsNo:{}", entry.getValue(), entry.getKey());
            Message message = MessageBuilder.withBody(entry.getKey().getBytes()).build();
            long epochMilli = 0L;
            if (entry.getValue().isAfter(now)) {
                epochMilli = Duration.between(now,entry.getValue()).toMillis();

                log.info("货单更改挂单中,选择时间在当前时间之后，则设置延迟队列时间,时间为：{}", epochMilli);

            }
            if (epochMilli == 0L) {
                log.info("货单更改挂单中,epochMilli时间为0,时间为：{}", epochMilli);
                log.info("货单更改挂单中直接发延迟队列，因为时间早于当前时间");
                rabbitTemplate.send(
                        RabbitKeyConstants.ORDER_GOODS_ON_DEAD_EXCHANGE, RabbitKeyConstants.ORDER_GOODS_ON_DEAD_ROUTE_KEY, message
                );
            } else {
                log.info("货单更改挂单中,epochMilli时间不为0,时间为：{}", epochMilli);
                log.info("货单更改挂单中发送延迟消息，因为挂单时间大于当前时间{}", epochMilli);

                message.getMessageProperties().setExpiration(String.valueOf(epochMilli));
                rabbitTemplate.send(
                        RabbitKeyConstants.ORDER_GOODS_ON_EXCHANGE, RabbitKeyConstants.ORDER_GOODS_ON_ROUTE_KEY, message
                );
            }
        }
    }
}
