package com.clx.performance.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.clx.order.enums.OrderEnum;
import com.clx.order.feign.OrderFeign;
import com.clx.order.param.feign.UpdateOrderInfoResidueWeightParam;
import com.clx.order.vo.feign.FeignOrderVO;
import com.clx.performance.constant.RabbitKeyConstants;
import com.clx.performance.dao.OrderChildDao;
import com.clx.performance.dao.OrderGoodsDao;
import com.clx.performance.enums.OrderChildEnum;
import com.clx.performance.enums.OrderGoodsStatusEnum;
import com.clx.performance.enums.TruckDemandEnum;
import com.clx.performance.model.OrderChild;
import com.clx.performance.model.OrderGoods;
import com.clx.performance.service.OrderCancelService;
import com.msl.common.result.Result;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageBuilder;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Slf4j
@Service
@AllArgsConstructor
public class OrderCancelServiceImpl implements OrderCancelService {


    private final OrderFeign orderFeign;

    private final OrderChildDao orderChildDao;

    private final OrderGoodsDao orderGoodsDao;

    private final RabbitTemplate rabbitTemplate;


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void ownerCancelOrderPre(String orderNo) {
        FeignOrderVO orderInfoFeign = orderFeign.getOrderInfoFeign(orderNo);
        BigDecimal orderTransportWeight = new BigDecimal(orderInfoFeign.getTransportWeight());

        List<OrderGoods> orderGoodsList = orderGoodsDao.getOrderGoodsListByOrderNo(orderNo);
        if (CollectionUtil.isEmpty(orderGoodsList)) {
            //未产生货单
            Result<Object> result = orderFeign.updateOrderStatusByOrderNo(orderNo, OrderEnum.Status.CANCELED.getCode());
            if (!result.succeed()) {
                throw new RuntimeException("货主取消订单失败,原因：远程调用取消订单失败");
            }
            return;
        }

        List<OrderChild> orderChildrenList = orderChildDao.selectResidueWeight(orderNo);

        if (CollectionUtil.isEmpty(orderChildrenList)) {
            Result<Object> result = orderFeign.updateOrderStatusByOrderNo(orderNo, OrderEnum.Status.CANCELED.getCode());
            if (!result.succeed()) {
                throw new RuntimeException("货主取消订单失败,原因：远程调用取消订单失败");
            }
            //说明此刻未产生运单
            for (OrderGoods orderGoods : orderGoodsList) {
                OrderGoods update = new OrderGoods();
                update.setId(orderGoods.getId());
                update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.CANCEL.getCode());
                orderGoodsDao.updateEntityByKey(update);
            }

        } else {
            BigDecimal orderChildResidueWeight = orderChildrenList.stream()
                    .map(OrderChild::getWeight)
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
            if (orderChildResidueWeight.compareTo(orderTransportWeight) == 0) {
                throw new RuntimeException("当前订单已无余量可取消");
            }
            for (OrderGoods orderGoods : orderGoodsList) {
                OrderGoods update = new OrderGoods();
                update.setId(orderGoods.getId());
                update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.COMPLETED.getCode());
                orderGoodsDao.updateEntityByKey(update);
            }
            Result<Object> result = orderFeign.updateOrderStatusByOrderNo(orderNo, OrderEnum.Status.COMPLETED.getCode());
            if (!result.succeed()) {
                throw new RuntimeException("货主取消订单失败,原因：远程调用取消订单失败");
            }
        }

        log.info("执行订单取消前置：更改相关状态完毕，开始执行吨数回填，发送MQ消息,订单No {}", orderNo);
        Message message = MessageBuilder.withBody(orderNo.getBytes()).build();
        rabbitTemplate.send(
                RabbitKeyConstants.ORDER_CANCEL_EXCHANGE, RabbitKeyConstants.ORDER_CANCEL_ROUTE_KEY, message
        );

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void ownerCancelOrder(String orderNo) {
        log.info("开始执行订单取消吨数回填");
        FeignOrderVO orderInfoFeignVO = orderFeign.getOrderInfoFeign(orderNo);

        List<OrderGoods> orderGoodsList = orderGoodsDao.getOrderGoodsListByOrderNo(orderNo);
        Map<Integer, BigDecimal> truckDemandMap = new HashMap<>();
        truckDemandMap.put(TruckDemandEnum.TruckType.PLATFORM_CAR.getCode(), BigDecimal.ZERO);
        truckDemandMap.put(TruckDemandEnum.TruckType.OWNER_CAR.getCode(), BigDecimal.ZERO);
        if (!TruckDemandEnum.TruckType.PLATFORM_ANY_CAR.getCode().equals(orderInfoFeignVO.getTruckDemand())) {
            if (OrderEnum.Status.COMPLETED.getCode().equals(orderInfoFeignVO.getOrderStatus())) {
                //已完结
                List<OrderChild> orderChildrenList = orderChildDao.selectResidueWeight(orderNo);

                for (OrderChild orderChild : orderChildrenList) {
                    OrderChild update = new OrderChild();
                    update.setId(orderChild.getId());
                    update.setStatus(OrderChildEnum.Status.OWNER_CANCEL.getCode());
                    //TODO 补充批量更新接口  关闭运单状态
                    orderChildDao.updateEntityByKey(update);
                }
                Map<String, List<OrderGoods>> orderGoodsMap = orderGoodsList.stream()
                        .collect(Collectors.groupingBy(OrderGoods::getOrderGoodsNo));

                Map<String, List<OrderChild>> orderChildMap = orderChildrenList.stream()
                        .collect(Collectors.groupingBy(OrderChild::getOrderGoodsNo));

                for (Map.Entry<String, List<OrderGoods>> entry : orderGoodsMap.entrySet()) {
                    String orderGoodsNo = entry.getKey();
                    OrderGoods orderGoods = entry.getValue().get(0);

                    BigDecimal sum = orderChildMap.get(orderGoodsNo).stream()
                            .map(OrderChild::getWeight).reduce(BigDecimal.ZERO, BigDecimal::add);
                    truckDemandMap.put(orderGoods.getTruckDemand(),
                            truckDemandMap.get(orderGoods.getTruckDemand())
                                    .add(orderGoods.getResidueTransportWeight().add(sum)));

                    OrderGoods update = new OrderGoods();
                    update.setId(orderGoods.getId());
                    //TODO 补充批量更新接口  关闭货单状态
                    update.setResidueTransportWeight(BigDecimal.ZERO);
                    update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.COMPLETED.getCode());
                    orderGoodsDao.updateEntityByKey(update);
                }

            } else if (OrderEnum.Status.CANCELED.getCode().equals(orderInfoFeignVO.getOrderStatus())) {
                //已取消
                for (OrderGoods orderGoods : orderGoodsList) {
                    truckDemandMap.put(orderGoods.getTruckDemand(),
                            truckDemandMap.get(orderGoods.getTruckDemand())
                                    .add(orderGoods.getResidueTransportWeight()));
                    OrderGoods update = new OrderGoods();
                    update.setId(orderGoods.getId());
                    //TODO 补充批量更新接口  关闭货单状态
                    update.setResidueTransportWeight(BigDecimal.ZERO);
                    update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.CANCEL.getCode());
                    orderGoodsDao.updateEntityByKey(update);
                }
            }

            Integer truckDemand = orderInfoFeignVO.getTruckDemand();
            BigDecimal residueWeight = truckDemandMap.get(truckDemand);
            UpdateOrderInfoResidueWeightParam param = new UpdateOrderInfoResidueWeightParam();
            param.setOrderId(orderInfoFeignVO.getId());
            if (truckDemand.equals(TruckDemandEnum.TruckType.PLATFORM_CAR.getCode())) {
                param.setUpdateType("2");
                param.setResidueWeight(residueWeight.add(new BigDecimal(orderInfoFeignVO.getResidueWeight())));
                param.setPlatWeight(residueWeight);
            }
            if (truckDemand.equals(TruckDemandEnum.TruckType.OWNER_CAR.getCode())) {
                param.setUpdateType("3");
                param.setResidueWeight(residueWeight.add(new BigDecimal(orderInfoFeignVO.getResidueWeight())));
                param.setOwnWeight(residueWeight);
            }
            orderFeign.updateOrderInfoResidueWeight(param);

        } else {
            Map<Integer, BigDecimal> vehicleUsageMap = new HashMap<>();
            if (OrderEnum.Status.COMPLETED.getCode().equals(orderInfoFeignVO.getOrderStatus())) {
                //已完结
                List<OrderChild> orderChildrenList = orderChildDao.selectResidueWeight(orderNo);

                for (OrderChild orderChild : orderChildrenList) {
                    OrderChild update = new OrderChild();
                    update.setId(orderChild.getId());
                    update.setStatus(OrderChildEnum.Status.OWNER_CANCEL.getCode());
                    //TODO 补充批量更新接口  关闭运单状态
                    orderChildDao.updateEntityByKey(update);
                }
                //货单
                Map<String, List<OrderGoods>> orderGoodsMap = orderGoodsList.stream()
                        .collect(Collectors.groupingBy(OrderGoods::getOrderGoodsNo));
                //运单
                Map<String, List<OrderChild>> orderChildMap = orderChildrenList.stream()
                        .collect(Collectors.groupingBy(OrderChild::getOrderGoodsNo));


                for (Map.Entry<String, List<OrderGoods>> entry : orderGoodsMap.entrySet()) {
                    String orderGoodsNo = entry.getKey();
                    OrderGoods orderGoods = entry.getValue().get(0);
                    BigDecimal sum = orderChildMap.get(orderGoodsNo).stream()
                            .map(OrderChild::getWeight).reduce(BigDecimal.ZERO, BigDecimal::add);
                    vehicleUsageMap.put(orderGoods.getVehicleUsage(),
                            vehicleUsageMap.get(orderGoods.getVehicleUsage())
                                    .add(orderGoods.getResidueTransportWeight().add(sum)));
                    OrderGoods update = new OrderGoods();
                    update.setId(orderGoods.getId());
                    //TODO 补充批量更新接口  关闭货单状态
                    update.setResidueTransportWeight(BigDecimal.ZERO);
                    update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.COMPLETED.getCode());
                    orderGoodsDao.updateEntityByKey(update);

                }

            } else if (OrderEnum.Status.CANCELED.getCode().equals(orderInfoFeignVO.getOrderStatus())) {
                //已取消
                for (OrderGoods orderGoods : orderGoodsList) {
                    vehicleUsageMap.put(orderGoods.getVehicleUsage(),
                            vehicleUsageMap.get(orderGoods.getVehicleUsage())
                                    .add(orderGoods.getResidueTransportWeight()));
                    OrderGoods update = new OrderGoods();
                    update.setId(orderGoods.getId());
                    //TODO 补充批量更新接口  关闭货单状态
                    update.setResidueTransportWeight(BigDecimal.ZERO);
                    update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.COMPLETED.getCode());
                    orderGoodsDao.updateEntityByKey(update);
                }
            }

            String platformResidueCarryWeight = orderInfoFeignVO.getPlatformResidueCarryWeight();
            String ownResidueCarryWeight = orderInfoFeignVO.getOwnResidueCarryWeight();

            BigDecimal platformResidueSum = new BigDecimal(platformResidueCarryWeight).add(vehicleUsageMap.get(1));
            BigDecimal ownResidueSum = new BigDecimal(ownResidueCarryWeight).add(vehicleUsageMap.get(2));
            BigDecimal residueSum = new BigDecimal(orderInfoFeignVO.getResidueWeight()).add(platformResidueSum).add(ownResidueSum);
            UpdateOrderInfoResidueWeightParam param = new UpdateOrderInfoResidueWeightParam();
            param.setOrderId(orderInfoFeignVO.getId());
            param.setUpdateType("4");
            param.setResidueWeight(residueSum);
            param.setPlatWeight(platformResidueSum);
            param.setOwnWeight(ownResidueSum);
            orderFeign.updateOrderInfoResidueWeight(param);
        }


    }
}
