package com.clx.performance.service.impl.trace;

import com.clx.order.enums.ResultEnum;
import com.clx.performance.dao.OrderChildDao;
import com.clx.performance.model.OrderChild;
import com.clx.performance.param.pc.trace.QueryTraceParam;
import com.clx.performance.service.trace.TraceMonitorService;
import com.clx.performance.service.trace.TruckTraceService;
import com.clx.performance.vo.pc.trace.DriverTraceVO;
import com.clx.performance.vo.pc.trace.DriverTruckTraceVO;
import com.clx.performance.vo.pc.trace.OrderChildTraceVO;
import com.clx.user.feign.TruckFeign;
import com.clx.user.vo.feign.TruckOwnerInfoVO;
import com.msl.common.exception.ServiceSystemException;
import com.msl.common.result.Result;
import com.msl.common.utils.DateUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;

@Slf4j
@Service
@AllArgsConstructor
public class TraceMonitorServiceImpl implements TraceMonitorService {

    private final TruckTraceService truckTraceService;
    private final OrderChildDao orderChildDao;
    private final TruckFeign truckFeign;

    private final int TRACE_DAY_MAX = 7;
    @Override
    public OrderChildTraceVO getOrderChildTrace(QueryTraceParam param) {
        if(StringUtils.isBlank(param.getChildNo())){
            throw new ServiceSystemException(ResultEnum.PARAM_ERROR,"运单编号不能为空");
        }
        checkParam(param);
        OrderChild child = orderChildDao.getByChildNo(param.getChildNo()).orElseThrow(ResultEnum.DATA_NOT_FIND,
                "运单数据不存在");
        List<DriverTruckTraceVO> truckTrace = truckTraceService.listTruckTraceByTime(param.getTruckNo(),param.getBeginTime(),param.getEndTime());
        List<DriverTraceVO> driverTrace = truckTraceService.listDriverTraceByTime(child.getDriverUserNo(),param.getBeginTime(),param.getEndTime());

        OrderChildTraceVO result = new OrderChildTraceVO();
        result.setDriverTraceList(driverTrace);
        result.setTruckTraceList(truckTrace);
        return result;
    }

    @Override
    public OrderChildTraceVO getTruckTrace(QueryTraceParam param) {
        checkParam(param);
        //查询车辆的车主信息
        Result<TruckOwnerInfoVO> result = truckFeign.getTruckOwnerInfo(
                param.getTruckNo());
        log.info("查询车辆车主信息,车牌号:{},结果:{}",param.getTruckNo(),result);
        if(!result.succeed()){
            throw new ServiceSystemException(ResultEnum.DATA_NOT_FIND,"未查询到车辆车主信息");
        }
        long userNo = result.getData().getUserNo();

        return this.getTrace(userNo,param.getTruckNo(),param.getBeginTime(),param.getEndTime());
    }


    private OrderChildTraceVO getTrace(long userNo,String truckNo,String beginTime,String endTime){
        List<DriverTruckTraceVO> truckTrace = truckTraceService.listTruckTraceByTime(truckNo,beginTime,endTime);
        List<DriverTraceVO> driverTrace = truckTraceService.listDriverTraceByTime(userNo,beginTime,endTime);
        OrderChildTraceVO result = new OrderChildTraceVO();
        result.setDriverTraceList(driverTrace);
        result.setTruckTraceList(truckTrace);
        return result;
    }


    public void checkParam(QueryTraceParam param){
        LocalDateTime beginTime = DateUtils.parseDateTime(param.getBeginTime()).get();
        LocalDateTime endTime = DateUtils.parseDateTime(param.getEndTime()).get();
        if (endTime.isAfter(beginTime.plusDays(TRACE_DAY_MAX)) ) {      //最多查询7天
            throw new ServiceSystemException(ResultEnum.PARAM_ERROR,"选择的时间范围需小于7天");
        }
    }
}
