package com.clx.performance.service.impl.linewarn;

import com.clx.performance.dao.OrderChildDao;
import com.clx.performance.dao.OrderGoodsDao;
import com.clx.performance.dto.gd.GdPosDTO;
import com.clx.performance.enums.DriverTruckEnum;
import com.clx.performance.enums.OrderChildEnum;
import com.clx.performance.enums.ResultEnum;
import com.clx.performance.model.OrderChild;
import com.clx.performance.model.OrderGoods;
import com.clx.performance.service.linewarn.LineWarnTraceService;
import com.clx.performance.service.trace.TruckTraceService;
import com.clx.performance.struct.OrderChildStruct;
import com.clx.performance.struct.OrderGoodsStruct;
import com.clx.performance.struct.linewarn.LineWarnInfoStruct;
import com.clx.performance.utils.LocalDateTimeUtils;
import com.clx.performance.utils.gd.GdUtils;
import com.clx.performance.vo.pc.linewarn.LineWarnRoutePositionVO;
import com.clx.performance.vo.pc.linewarn.LineWarnRouteVO;
import com.clx.performance.vo.pc.linewarn.LineWarnTraceVO;
import com.clx.performance.vo.pc.linewarn.LineWarnTruckTraceInfoVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;


@Slf4j
@Service
public class LineWarnTraceServiceImpl implements LineWarnTraceService {
    // 最近位置时间
    private static final Integer LAST_POSITION_TIME = 5;

    @Autowired
    private OrderGoodsDao orderGoodsDao;
    @Autowired
    private OrderChildDao orderChildDao;

    @Autowired
    private OrderGoodsStruct orderGoodsStruct;
    @Autowired
    private OrderChildStruct orderChildStruct;
    @Autowired
    private TruckTraceService truckTraceService;

    @Autowired
    private GdUtils gdUtils;

    @Autowired
    private LineWarnInfoStruct lineWarnInfoStruct;
    @Override
    public LineWarnTraceVO getTraceInfo(String orderGoodsNo) {
        OrderGoods orderGoods = orderGoodsDao
                .getByOrderGoodsNo(orderGoodsNo).orElseThrow(ResultEnum.DATA_NOT_FIND);
        LineWarnTraceVO result = orderGoodsStruct.convertLineWarnTraceVO(orderGoods);

        // 装卸吨数
        BigDecimal loadWeight = orderChildDao.sumLoadWeightByOrderGoodsNo(orderGoodsNo);
        BigDecimal unloadWeight = orderChildDao.sumUnloadWeightByOrderGoodsNo(orderGoodsNo);
        result.setLoadWeight(loadWeight==null? BigDecimal.ZERO : loadWeight);
        result.setUnloadWeight(unloadWeight==null? BigDecimal.ZERO : unloadWeight);

        // 车辆信息
        List<OrderChild> orderChildList = orderChildDao.listByOrderGoodsNoAndStatus(orderGoods.getOrderGoodsNo(),
                Arrays.asList(
                        OrderChildEnum.Status.CREATED.getCode(),
                        OrderChildEnum.Status.PAY.getCode(),
                        OrderChildEnum.Status.GO_TO_SEND.getCode(),
                        OrderChildEnum.Status.ARRIVE_SEND.getCode(),
                        OrderChildEnum.Status.LOAD.getCode(),
                        OrderChildEnum.Status.GO_TO_RECEIVE.getCode(),
                        OrderChildEnum.Status.ARRIVE_RECEIVE.getCode()
                )
        );
        List<LineWarnTruckTraceInfoVO> truckList = orderChildStruct.convertLineWarnTruckTraceInfoVO(orderChildList);
        LocalDateTime now = LocalDateTime.now();
        // 查询最近位置
        for (LineWarnTruckTraceInfoVO item : truckList) {
            BigDecimal[] currentPosition = truckTraceService.getCurrentPosition(item.getTruckNo(),
                    item.getDriverUserNo(), LocalDateTimeUtils.formatTime(LocalDateTime.now().minusMinutes(LAST_POSITION_TIME)));
            if (currentPosition != null){
                item.setLongitude(currentPosition[0]);
                item.setLatitude(currentPosition[1]);
            }

            if (Objects.equals(item.getStatus(), OrderChildEnum.Status.CREATED.getCode())
                || Objects.equals(item.getStatus(), OrderChildEnum.Status.PAY.getCode())
                || Objects.equals(item.getStatus(), OrderChildEnum.Status.GO_TO_SEND.getCode())
                || Objects.equals(item.getStatus(), OrderChildEnum.Status.ARRIVE_SEND.getCode())
            ){
                item.setTruckStatus(DriverTruckEnum.TraceTruckStatus.EMPTY.getCode());
            }
            else{
                item.setTruckStatus(DriverTruckEnum.TraceTruckStatus.HEAVY.getCode());
            }
        }

        // 推荐线路
        List<LineWarnRoutePositionVO> positionList = lineWarnInfoStruct
                .convertLineWarnRoutePositionVO(gdUtils.getRoute(orderGoods.getSendLongitude(),
                        orderGoods.getSendLatitude(), orderGoods.getReceiveLongitude(),
                        orderGoods.getReceiveLatitude()));

        LineWarnRouteVO route = new LineWarnRouteVO();
        route.setPositonList(positionList);
        result.setRoute(route);

        result.setTruckList(truckList);
        return result;
    }

}
