package com.clx.performance.utils.gd;

import com.clx.performance.dto.gd.GdPosDTO;
import com.clx.performance.dto.gd.GdRouteDTO;
import com.msl.common.exception.ServiceSystemException;
import com.msl.common.utils.gps.GpsUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

@Slf4j
@Component
public class GdUtils {
    private static final Integer SEC_FER_MIN = 60;
    private static final String REDIS_WEATHER = "clx-performance:weather:";

    @Autowired
    private GdService gdService;

    @Autowired
    private RedisTemplate<String, String> redisTemplate;

    /**
     * 获取需要时间（min）
     */
    public int getNeedTime(BigDecimal originLongitude, BigDecimal originLatitude,
                           BigDecimal destinationLongitude, BigDecimal destinationLatitude){
        try {
            List<GdRouteDTO> gdRouteDTOS = gdService.getRoute(originLongitude, originLatitude, destinationLongitude, destinationLatitude);

            if (!gdRouteDTOS.isEmpty()){
                return gdRouteDTOS.get(0).getDuration() /SEC_FER_MIN;
            }

        } catch (ServiceSystemException e) {
            log.info("获取高德线路失败, msg:{}", e.getMessage());
        }

        return 0;
    }

    /**
     * 获取需要时间（min）
     */
    public int getNeedTime(BigDecimal[] currentPosition, BigDecimal destinationLongitude, BigDecimal destinationLatitude) {
        if (currentPosition == null) {return 0;}
        return getNeedTime(currentPosition[0], currentPosition[1], destinationLongitude, destinationLatitude);
    }

    /**
     * 获取需要距离（m）
     */
    public int getNeedDistance(BigDecimal[] currentPosition,
                               BigDecimal destinationLongitude, BigDecimal destinationLatitude){
        int needDistance = 0;

        if (currentPosition == null) {return 0;}
        try {
            List<GdRouteDTO> gdRouteDTOS = gdService.getRoute(currentPosition[0], currentPosition[1], destinationLongitude, destinationLatitude);

            if (!gdRouteDTOS.isEmpty()){
                needDistance = gdRouteDTOS.get(0).getDistance();
            }

        } catch (ServiceSystemException e) {
            log.info("获取高德线路失败, msg:{}", e.getMessage());
        }

        return needDistance;
    }

    public Integer getDistance(BigDecimal[] currentPosition, BigDecimal destinationLongitude, BigDecimal destinationLatitude) {
        if (currentPosition == null) {return null;}
        return GpsUtil.distanceM(currentPosition[0], currentPosition[1], destinationLongitude, destinationLatitude);
    }

    /**
     * 天气
     */
    public String getWeather(String city){
        String weather = redisTemplate.opsForValue().get(REDIS_WEATHER + city);
        if (StringUtils.isNotBlank(weather)){return weather;}

        return gdService.getWeather(city);
    }
    /**
     * 天气
     */
    public String getCity(BigDecimal longitude, BigDecimal latitude){
        return gdService.getCity(longitude,latitude);
    }

    /**
     * 获取线路
     */
    public List<GdPosDTO> getRoute(BigDecimal originLongitude, BigDecimal originLatitude,
                                   BigDecimal destinationLongitude, BigDecimal destinationLatitude){
        try {
            List<GdRouteDTO> gdRouteDTOS = gdService.getRoute(originLongitude, originLatitude, destinationLongitude, destinationLatitude);

            if (!gdRouteDTOS.isEmpty()){
                return gdRouteDTOS.get(0).getPosList();
            }

        } catch (ServiceSystemException e) {
            log.info("获取高德线路失败, msg:{}", e.getMessage());
        }

        return new ArrayList<>();
    }
}
