package com.clx.performance.service.impl;

import com.alibaba.fastjson.JSON;
import com.clx.open.sdk.callback.OpenCallBackClient;
import com.clx.open.sdk.callback.message.OrderChildMessage;
import com.clx.open.sdk.enums.ResultStatusEnum;
import com.clx.order.feign.OrderFeign;
import com.clx.order.vo.feign.FeignOrderVO;
import com.clx.performance.config.ThirdAppConfig;
import com.clx.performance.dao.OrderChildDao;
import com.clx.performance.model.OrderChild;
import com.clx.performance.service.LastTruckService;
import com.clx.performance.struct.OrderChildStruct;
import com.msl.common.result.Result;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Objects;

/**
 * @ClassName LastTruckServiceImpl
 * @Description
 * @Author kavin
 * @Date 2023/11/9 13:13
 * @Version 1.0
 */
@Service
@Slf4j
@AllArgsConstructor
public class LastTruckServiceImpl implements LastTruckService {

    private final OrderFeign orderFeign;
    private final OrderChildDao orderChildDao;
    private final OrderChildStruct orderChildStruct;
    private final ThirdAppConfig thirdAppConfig;
    /**
     * @Author kavin
     * @Description 判断最后一车的逻辑
     * @Param [orderNo]
     * @return
     **/
    @Override
    public OrderChild getLastTruckChild(String orderNo) {
        FeignOrderVO orderInfo = orderFeign.getOrderInfoFeign(orderNo);
        if(Objects.isNull(orderInfo)){
            log.error("计算最后一车，通过订单号:{},未查询到对应的订单",orderNo);
            return null;
        }
        BigDecimal weight = orderChildDao.listAfterArrayReceiveChild(orderNo);
        //判断当前订单下所有到达目的地之后的运单的装车吨数  >= （订单拉运吨数 -35吨）
        log.info("订单号:{} ,计算是否到达可计算最后一车标准，运单累计拉运吨数：{}, 订单拉运吨数:{}",
                orderNo,weight,orderInfo.getTransportWeight());
        if(Objects.nonNull(weight) && weight.compareTo(orderInfo.getTransportWeight().subtract(new BigDecimal("35"))) >= 0){
            //查询最后一车的运单
            OrderChild child = orderChildDao.getLastTruckChild(orderNo);
            log.info("当前订单计算出最后一车，运单编号：{} ", child.getChildNo());
            return child;
        }
        return  null;
    }
    /**
     * @Author kavin
     * @Description 同步最后一车
     * @Param [child]
     * @return
     **/
    @Override
    public Result<?> syncLastTruckChild(OrderChild child) {
        FeignOrderVO orderInfo = orderFeign.getOrderInfoFeign(child.getOrderNo());
        OrderChildMessage message = orderChildStruct.convertMessage(child);
        message.setLastFlag(ResultStatusEnum.YES.getCode());
        String data = JSON.toJSONString(message);
        log.info("开始同步最后一车的运单，运单编号:{},运单信息:{}",child.getChildNo(), data);
        OpenCallBackClient openCallBackClient = thirdAppConfig.getOpenCallBackClient(orderInfo.getOrderSource().toString());
        Result<?> result = openCallBackClient.encryptPost(data,message.topic());
        log.info("结束同步最后一车的运单，运单编号:{}，响应结果:{}",child.getChildNo(),JSON.toJSONString(result));
        return result;
    }

    @Override
    public OrderChild getOrderCompleteLastTruck(String orderNo) {
        //判断当前订单下是否存在未到达目的地之前的运单 && 排除掉已取消的运单
        Long count = orderChildDao.haveArriveReceiveChild(orderNo);
        log.info("已完成/已完结的订单号:{},查询运单状态非取消并且在到达目的地之前的运单数量为:{}",orderNo,count);
        if(count  == 0 ){
            //说明排除已取消的运单后，剩下的所有运单都已经是到达目的地之后的状态，根据到达目的地的时间倒序取出第一条运单作为最后一车
            return orderChildDao.getOrderCompleteLastTruck(orderNo);
        }
        return null;
    }



    /**
     * @Author kavin
     * @Description 订单取消、到达最晚拉运时间 同步最后一车
     * @Param [orderNo]
     * @return
     **/
    public Result<?> syncLastTruckChild4OrderTrigger(String orderNo) {
        OrderChild child = getOrderCompleteLastTruck(orderNo);
        if(Objects.nonNull(child)){
            return syncLastTruckChild(child);
        }
        return Result.ok("订单完成/完结状态下未查询最后一车，订单编号:" + orderNo);
    }
}
