package com.clx.performance.service.impl.linewarn;

import com.clx.performance.dao.OrderChildDao;
import com.clx.performance.dao.linewarn.LineWarnConfigDao;
import com.clx.performance.dao.linewarn.LineWarnInfoDao;
import com.clx.performance.dto.linewarn.LineWarnCommonInfoDTO;
import com.clx.performance.dto.linewarn.LineWarnOrderChildStatisticsDTO;
import com.clx.performance.enums.linewarn.LineWarnConfigEnum;
import com.clx.performance.enums.linewarn.LineWarnInfoEnum;
import com.clx.performance.model.OrderGoods;
import com.clx.performance.model.linewarn.LineWarnConfig;
import com.clx.performance.model.linewarn.LineWarnInfo;
import com.clx.performance.service.linewarn.LineWarnCommonService;
import com.clx.performance.service.linewarn.LineWarnTakeWarnService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

@Slf4j
@Service
public class LineWarnTakeWarnServiceImpl implements LineWarnTakeWarnService {
    // 默认时间
    private static final int DEFAULT_TIME = 60;
    @Autowired
    private LineWarnConfigDao lineWarnConfigDao;
    @Autowired
    private LineWarnInfoDao lineWarnInfoDao;
    @Autowired
    private LineWarnCommonService lineWarnCommonService;
    @Autowired
    private OrderChildDao orderChildDao;

    @Override
    public void takeWarn(OrderGoods orderGoods, LineWarnCommonInfoDTO lineWarnCommonInfo) {

        int loadTimeAvg = lineWarnCommonInfo.getLoadTimeAvg();
        int unloadTimeAvg = lineWarnCommonInfo.getUnloadTimeAvg();
        Integer arriveReceiveExpectTime = lineWarnCommonInfo.getArriveReceiveExpectTime();

        long totalTime = Duration.between(lineWarnCommonInfo.getPendingOrderTime(),
                lineWarnCommonInfo.getTransportEndTime()).toMinutes();
        int needTime = DEFAULT_TIME+loadTimeAvg+unloadTimeAvg+arriveReceiveExpectTime;
        // 接单最长时间
        long difTime = totalTime - needTime;
        // 任务量
        BigDecimal taskWeight = lineWarnCommonInfo.getTaskWeight();

        // 任务完成率
        BigDecimal taskCompleteRatio = BigDecimal.ZERO;
        BigDecimal weight = orderChildDao.sumTakeWeightByOrderGoodsNo(orderGoods.getOrderGoodsNo());
        if (weight != null){
            taskCompleteRatio = weight.divide(taskWeight, 4, BigDecimal.ROUND_UP).movePointRight(2);
        }

        List<LineWarnConfig> lineWarnConfigList = lineWarnConfigDao
                .listByWarnType(LineWarnConfigEnum.WarnType.TAKE.getCode());
        if (lineWarnConfigList.isEmpty()) {return;}

        for (LineWarnConfig item : lineWarnConfigList) {
            doTakeWarn(orderGoods, item,lineWarnCommonInfo.getPendingOrderTime(), difTime, taskCompleteRatio);
        }

    }

    private void doTakeWarn(OrderGoods orderGoods, LineWarnConfig lineWarnConfig, LocalDateTime pendingOrderTime,
                            long difTime, BigDecimal taskCompleteRatio){
        LineWarnInfo lineWarnInfo = lineWarnInfoDao
                .findByOrderGoodsNoAndWarnConfigId(orderGoods.getOrderGoodsNo(), lineWarnConfig.getId()).orNull();
        if (lineWarnInfo != null
                && Objects.equals(lineWarnInfo.getStatus(), LineWarnInfoEnum.Status.RESOLVE.getCode())){
            return;
        }

        int time = lineWarnConfig.getTimeRatio().multiply(new BigDecimal(difTime)).intValue();

        // 未达到时间占比
        if (pendingOrderTime.plusMinutes(time).isBefore(LocalDateTime.now())){return;}
        // 达到任务占比
        if (lineWarnConfig.getTaskCompleteRatio().compareTo(taskCompleteRatio) >= 0) {return;}

        // 更新
        lineWarnCommonService.lineWarnInfoUpdate(orderGoods, lineWarnInfo, lineWarnConfig);

    }

}

