package com.clx.performance.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.clx.performance.dao.OrderChildDao;
import com.clx.performance.dao.OrderChildImageDao;
import com.clx.performance.dao.OrderChildPoundAuditDao;
import com.clx.performance.dao.OrderGoodsDao;
import com.clx.performance.enums.*;
import com.clx.performance.extranal.user.DriverService;
import com.clx.performance.model.OrderChild;
import com.clx.performance.model.OrderChildImage;
import com.clx.performance.model.OrderChildPoundAudit;
import com.clx.performance.model.OrderGoods;
import com.clx.performance.param.app.*;
import com.clx.performance.param.pc.PagePoundAuditParam;
import com.clx.performance.service.OrderChildLogService;
import com.clx.performance.service.OrderChildPoundLogService;
import com.clx.performance.service.OrderChildService;
import com.clx.performance.struct.OrderChildPoundImageStruct;
import com.clx.performance.struct.OrderChildStruct;
import com.clx.performance.vo.app.OrderChildVO;
import com.clx.performance.vo.app.SaveOrderChildVO;
import com.clx.performance.vo.pc.PageOrderChildPoundAuditVO;
import com.clx.user.vo.feign.DriverTruckInfoFeignVo;
import com.msl.common.enums.ResultCodeEnum;
import com.msl.common.exception.ServiceSystemException;
import com.msl.common.utils.DateUtils;
import com.msl.user.data.UserSessionData;
import com.msl.user.utils.TokenUtil;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author: aiqinguo
 * @Description: 运单表
 * @Date: 2023/09/18 11:34:50
 * @Version: 1.0
 */

@Slf4j
@Service
@AllArgsConstructor
public class OrderChildServiceImpl implements OrderChildService {

    private final OrderGoodsDao orderGoodsDao;

    private final OrderChildDao orderChildDao;

    private final OrderChildImageDao orderChildImageDao;
    private final OrderChildPoundAuditDao orderChildPoundAuditDao;


    private final OrderChildLogService orderChildLogService;
    private final OrderChildPoundLogService orderChildPoundLogService;

    private final DriverService driverService;
    private final OrderChildStruct orderChildStruct;
    private final OrderChildPoundImageStruct orderChildPoundImageStruct;

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public SaveOrderChildVO saveOrderChild(OrderChildSaveParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = childNoGenerate();

        // 查询司机车辆信息
        DriverTruckInfoFeignVo driverTruckInfo = driverService.getUserDetailInfo(param.getDriverUserNo(), param.getTruckId()).orElseThrow(ResultCodeEnum.FAIL);
        BigDecimal truckLoad = driverTruckInfo.getLoad();

        LocalDateTime now = LocalDateTime.now();
        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(param.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.ORDER_INVALID);
        // 状态验证
        if (!Objects.equals(orderGoods.getOrderGoodsStatus(), OrderGoodsStatusEnum.Status.PAYING.getCode())
                && !Objects.equals(orderGoods.getOrderGoodsStatus(), OrderGoodsStatusEnum.Status.GO_TO_SEND.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_INVALID);
        }

        // 库存验证
        if (orderGoods.getResidueTransportWeight().compareTo(BigDecimal.ZERO) <= 0) {throw new ServiceSystemException(PerformanceResultEnum.ORDER_WEIGHT_LACK);}
        if (orderGoods.getResidueTransportWeight().compareTo(truckLoad) < 0) {throw new ServiceSystemException(PerformanceResultEnum.ORDER_WEIGHT_LACK);}

        OrderChild orderChild = new OrderChild();
        orderChild.setChildNo(childNo);
        orderChild.setUserNo(userNo);

        //--------------------
        orderChild.setOwnerUserNo(orderGoods.getUserNo());
        orderChild.setOwnerName(orderGoods.getUserName());
        //--------------------

        orderChild.setOrderNo(orderGoods.getOrderNo());
        orderChild.setOrderGoodsNo(orderGoods.getOrderGoodsNo());
        orderChild.setFreightPrice(orderGoods.getPendingOrderFreight());

        orderChild.setGoodsId(orderGoods.getGoodsId());
        orderChild.setGoodsName(orderGoods.getGoodsName());

        orderChild.setSendAddressId(orderGoods.getSendAddressId());
        orderChild.setSendAddress(orderGoods.getSendAddressShorter());
        orderChild.setReceiveAddressId(orderGoods.getReceiveAddressId());
        orderChild.setReceiveAddress(orderGoods.getReceiveAddressShorter());

        orderChild.setLoadDeadline(orderGoods.getLastArriveSendTime());

        orderChild.setDriverUserNo(driverTruckInfo.getUserNo());
        orderChild.setDriverName(driverTruckInfo.getName());
        orderChild.setDriverMobile(driverTruckInfo.getMobile());

        orderChild.setTruckId(driverTruckInfo.getTruckId());
        orderChild.setTruckNo(driverTruckInfo.getTruckNo());
        orderChild.setTruckLoad(truckLoad);

        orderChild.setWeight(orderChild.getTruckLoad());
        orderChild.setPayTime(now);
        orderChild.setStatus(OrderChildEnum.Status.CREATED.getCode());
        orderChild.setCreateTime(now);

        // 更新货单数据
        updateOrderGoodsAmount(orderGoods, truckLoad);

        // 新增运单
        orderChildDao.saveEntity(orderChild);

        // 更新司机状态
        lockDriver(orderChild.getDriverUserNo());

        // 更新车辆状态
        lockTruck(orderChild.getTruckId());

        // 新增日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.CREATED.getCode(), OrderChildLogEnum.Type.CREATED.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());

        SaveOrderChildVO result = new SaveOrderChildVO();
        result.setChildNo(childNo);
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateGotoSendAddress(OrderChildGoToSendAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_SEND.getCode())){return;}
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.CREATED.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        orderChild.setStatus(OrderChildEnum.Status.GO_TO_SEND.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.GO_TO_SEND.getCode(), OrderChildLogEnum.Type.GO_TO_SEND.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateArriveSendAddress(OrderChildArriveSendAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_SEND.getCode())){return;}
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_SEND.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }
        // 装货超时
        if (orderChild.getLoadDeadline().isBefore(LocalDateTime.now())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_LOAD_TIMEOUT);
        }

        orderChild.setStatus(OrderChildEnum.Status.ARRIVE_SEND.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.ARRIVE_SEND.getCode(), OrderChildLogEnum.Type.ARRIVE_SEND.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateGotoReceiveAddress(OrderChildGoToReceiveAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_RECEIVE.getCode())){return;}
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.LOAD.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }


        orderChild.setStatus(OrderChildEnum.Status.GO_TO_RECEIVE.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.GO_TO_RECEIVE.getCode(), OrderChildLogEnum.Type.GO_TO_RECEIVE.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateArriveReceiveAddress(OrderChildArriveReceiveAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_RECEIVE.getCode())){return;}
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_RECEIVE.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        orderChild.setStatus(OrderChildEnum.Status.ARRIVE_RECEIVE.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.ARRIVE_RECEIVE.getCode(), OrderChildLogEnum.Type.ARRIVE_RECEIVE.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateLoad(OrderChildLoadParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CANCEL.getCode())
                || Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.PLATFORM_CANCEL.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCELED);
        }

        if (orderChild.getLoadTime() == null && Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_SEND.getCode())) {
            updateLoadFirst(param, orderChild);
            // 日志
            orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.LOAD.getCode(), OrderChildLogEnum.Type.LOAD.getMsg(),
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        } else {
            updateReload(param, orderChild);
            // 日志
            orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.RELOAD.getCode(), OrderChildLogEnum.Type.RELOAD.getMsg(),
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        }
    }

    private void updateLoadFirst(OrderChildLoadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        // 装货超时
        if (orderChild.getLoadDeadline().isBefore(LocalDateTime.now())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_LOAD_TIMEOUT);
        }

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getLoadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.LOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        orderChild.setLoadRough(param.getLoadRough());
        orderChild.setLoadTare(param.getLoadTare());
        orderChild.setLoadNet(param.getLoadNet());
        orderChild.setLoadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));
        orderChild.setStatus(OrderChildEnum.Status.LOAD.getCode());
        orderChildDao.updateLoad(orderChild);

        orderChildImageDao.batchSaveEntity(imageList);
    }

    private void updateReload(OrderChildLoadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getLoadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.LOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        orderChild.setLoadRough(param.getLoadRough());
        orderChild.setLoadTare(param.getLoadTare());
        orderChild.setLoadNet(param.getLoadNet());
        orderChild.setLoadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));
        orderChildDao.updateLoad(orderChild);

        orderChildImageDao.deleteLoad(childNo);
        orderChildImageDao.batchSaveEntity(imageList);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateUnload(OrderChildUnloadParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CANCEL.getCode())
                || Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.PLATFORM_CANCEL.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCELED);
        }

        if (orderChild.getUnloadTime() == null && Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_RECEIVE.getCode())) {
            updateUnloadFirst(param, orderChild);
            // 日志
            orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.UNLOAD.getCode(), OrderChildLogEnum.Type.UNLOAD.getMsg(),
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
            orderChildPoundLogService.saveDriverOrderChildLog(childNo, OrderChildPoundAuditEnum.Status.AUDIT.getCode(), "",
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        } else {
            updateReUnload(param, orderChild);
            // 日志
            orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.REUNLOAD.getCode(), OrderChildLogEnum.Type.REUNLOAD.getMsg(),
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
            orderChildPoundLogService.saveDriverOrderChildLog(childNo, OrderChildPoundAuditEnum.Status.AUDIT.getCode(), "",
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        }
    }

    private void updateUnloadFirst(OrderChildUnloadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getUnloadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.UNLOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        orderChild.setUnloadRough(param.getUnloadRough());
        orderChild.setUnloadTare(param.getUnloadTare());
        orderChild.setUnloadNet(param.getUnloadNet());
        orderChild.setUnloadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));
        orderChild.setStatus(OrderChildEnum.Status.UNLOAD.getCode());
        orderChild.setPoundStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());

        OrderChildPoundAudit audit = new OrderChildPoundAudit();
        audit.setChildNo(childNo);
        audit.setStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());
        audit.setLoadNet(orderChild.getLoadNet());
        audit.setUnloadNet(orderChild.getUnloadNet());

        orderChildDao.updateUnload(orderChild);

        orderChildImageDao.batchSaveEntity(imageList);

        orderChildPoundAuditDao.saveEntity(audit);

    }

    private void updateReUnload(OrderChildUnloadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        // 审核中
        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.AUDIT.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_POUND_AUDIT);
        }

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getUnloadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.UNLOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        orderChild.setUnloadRough(param.getUnloadRough());
        orderChild.setUnloadTare(param.getUnloadTare());
        orderChild.setUnloadNet(param.getUnloadNet());
        orderChild.setUnloadTime(LocalDateTime.now());
        orderChild.setPoundStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());
        orderChild.setWeight(orderChildWeightCalc(orderChild));

        OrderChildPoundAudit audit = new OrderChildPoundAudit();
        audit.setChildNo(childNo);
        audit.setStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());
        audit.setLoadNet(orderChild.getLoadNet());
        audit.setUnloadNet(orderChild.getUnloadNet());

        orderChildDao.updateUnload(orderChild);

        orderChildImageDao.deleteUnload(childNo);
        orderChildImageDao.batchSaveEntity(imageList);

        orderChildPoundAuditDao.saveEntity(audit);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateLoadAndUnloadAgain(OrderChildLoadAndUnloadAgainParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();
        String childNo = param.getChildNo();
        LocalDateTime now = LocalDateTime.now();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CANCEL.getCode())
                || Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.PLATFORM_CANCEL.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCELED);
        }
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.UNLOAD.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.AUDIT.getCode())){return;}
        if (!Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.REJECT.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getLoadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.LOAD.getCode());
            image.setImage(item);
        }
        for (String item : param.getUnloadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.UNLOAD.getCode());
            image.setImage(item);
        }

        orderChild.setLoadRough(param.getLoadRough());
        orderChild.setLoadTare(param.getLoadTare());
        orderChild.setLoadNet(param.getLoadNet());
        orderChild.setUnloadRough(param.getUnloadRough());
        orderChild.setUnloadTare(param.getUnloadTare());
        orderChild.setUnloadNet(param.getUnloadNet());
        orderChild.setWeight(orderChildWeightCalc(orderChild));

        orderChild.setPoundStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());

        OrderChildPoundAudit audit = new OrderChildPoundAudit();
        audit.setChildNo(childNo);
        audit.setStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());
        audit.setLoadNet(orderChild.getLoadNet());
        audit.setUnloadNet(orderChild.getUnloadNet());

        orderChildDao.updateLoadAndUnload(orderChild);

        orderChildImageDao.deleteLoadAndUnload(childNo);
        orderChildImageDao.batchSaveEntity(imageList);

        orderChildPoundAuditDao.saveEntity(audit);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.RELOAD_AND_REUNLOAD.getCode(), OrderChildLogEnum.Type.RELOAD_AND_REUNLOAD.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());

        orderChildPoundLogService.saveDriverOrderChildLog(childNo, OrderChildPoundAuditEnum.Status.AUDIT.getCode(), "",
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDriverConfirm(OrderChildDriverConfirmParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CANCEL.getCode())
                || Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.PLATFORM_CANCEL.getCode())){
            return;
        }
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.UNLOAD.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        // 审核中
        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.AUDIT.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_POUND_AUDIT);
        }
        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.REJECT.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        orderChild.setStatus(OrderChildEnum.Status.UNSETTLE.getCode());
        orderChildDao.updateStatus(orderChild);

        // 释放司机、车辆
        releaseDriver(orderChild.getDriverUserNo());
        releaseTruck(orderChild.getTruckId());

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.DRIVER_CONFIRM.getCode(), OrderChildLogEnum.Type.DRIVER_CONFIRM.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDriverCancel(OrderChildDriverCancelParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CANCEL.getCode())
                || Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.PLATFORM_CANCEL.getCode())){
            return;
        }

        // 禁止取消
        if (orderChild.getStatus()>OrderChildEnum.Status.ARRIVE_SEND.getCode()) {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCEL_FORBID);
        }

        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);

        // 取消量验证
        if (!cancelCountCheck(userNo)){throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCEL_FORBID_COUNT);}

        orderChild.setStatus(OrderChildEnum.Status.DRIVER_CANCEL.getCode());

        orderChildDao.updateCancel(orderChild);

        // 返回吨数
        cancelReturnWeight(orderChild, orderGoods);

        // 释放司机、车辆
        releaseDriver(orderChild.getDriverUserNo());
        releaseTruck(orderChild.getTruckId());

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.DRIVER_CANCEL.getCode(), OrderChildLogEnum.Type.DRIVER_CANCEL.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    public OrderChildVO getOrderChildInfo(String childNo) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        OrderChildVO result = orderChildStruct.convert(orderChild);

        List<OrderChildImage> imageList = orderChildImageDao.listLoadAndUnload(childNo).orElse(new ArrayList<>());
        result.setLoadImageList(imageList.stream().filter(item->Objects.equals(item.getId(),OrderChildImage.Type.LOAD.getCode())).map(item->item.getImage()).collect(Collectors.toList()));
        result.setUnloadImageList(imageList.stream().filter(item->Objects.equals(item.getId(),OrderChildImage.Type.UNLOAD.getCode())).map(item->item.getImage()).collect(Collectors.toList()));

        return result;
    }

    @Override
    public IPage<OrderChildVO> pageOrderChild(PageOrderChildOfDriverParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        param.setUserNo(userNo);
        return orderChildDao.pageOrderChildOfDriver(param);

    }


    /**
     * 更新货单数据
     */
    private void updateOrderGoodsAmount(OrderGoods orderGoods, BigDecimal weight){
        if (orderGoods.getResidueTransportWeight().compareTo(weight) <= 0) {orderGoods.setOrderGoodsStatus(0);}

        orderGoodsDao.updateWeight(orderGoods.getId(), weight);
    }

    /**
     * 取消检测
     */
    private boolean cancelCountCheck(Long userNo){
        LocalDateTime startTime = DateUtils.parseDateTime(DateUtils.formatDateTime(LocalDateTime.now(), "yyyy-MM-dd").get() + " 00:00:00").get();
        LocalDateTime endTime = DateUtils.parseDateTime(DateUtils.formatDateTime(LocalDateTime.now(), "yyyy-MM-dd").get() + " 00:00:00").get();

        long count = orderChildDao.countOfCancel(userNo, startTime, endTime);

        return count<10;
    }

    /**
     * 取消返吨数
     */
    private void cancelReturnWeight(OrderChild orderChild, OrderGoods orderGoods){

    }

    /**
     * 运单拉运吨数计算
     */
    private BigDecimal orderChildWeightCalc(OrderChild orderChild){
        if (orderChild.getUnloadNet() == null && orderChild.getLoadNet() == null) {return orderChild.getTruckLoad();}
        else if (orderChild.getUnloadNet() == null) {return orderChild.getLoadNet();}
        else {return orderChild.getUnloadNet().compareTo(orderChild.getLoadNet())>0? orderChild.getLoadNet() : orderChild.getUnloadNet();}
    }


    /**
     * 接单锁定司机
     */
    private void lockDriver(Long driverNo){

    }

    /**
     * 接单锁定车辆
     */
    private void lockTruck(Integer truckId){

    }

    /**
     * 释放司机
     */
    private void releaseDriver(Long driverNo){

    }

    /**
     * 释放车辆
     */
    private void releaseTruck(Integer truckId){

    }

    /**
     * 创建运单号
     */
    private String childNoGenerate(){

        return DateTimeFormatter.ofPattern("yyyyMMddHHmmss").format(LocalDateTime.now());

    }

    @Override
    public IPage<PageOrderChildPoundAuditVO> pagePoundAuditList(PagePoundAuditParam param) {
        return orderChildDao.pagePoundAuditList(param);
    }

}
