package com.clx.performance.service.impl.settle;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.clx.performance.dao.settle.SettlementDriverDetailDao;
import com.clx.performance.enums.loan.OwnerLoanRecordEnum;
import com.clx.performance.model.OrderChild;
import com.clx.performance.model.settle.SettlementDriverDetail;
import com.clx.performance.param.pc.driver.PageCarrierSettlementDriverDetailParam;
import com.clx.performance.service.settle.SettlementDriverDetailService;
import com.clx.performance.struct.settle.SettlementDriverDetailStruct;
import com.clx.performance.vo.pc.carrier.settle.CarrierPageSettlementDriverDetailVO;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.Objects;

/**
 * @author liruixin
 * Date 2023-10-12
 * Time 09:43
 */
@Service
@Slf4j
@AllArgsConstructor
public class SettlementDriverDetailServiceImpl  implements SettlementDriverDetailService {

    private final SettlementDriverDetailDao settlementDriverDetailDao;
    private final SettlementDriverDetailStruct settlementOwnerDetailStruct;

    @Override
    public Integer saveSettlementDetail(OrderChild orderChild) {

        SettlementDriverDetail settlementDriverDetail = new SettlementDriverDetail();
        settlementDriverDetail.setChildNo(orderChild.getChildNo());
        settlementDriverDetail.setOrderGoodsNo(orderChild.getOrderGoodsNo());
        settlementDriverDetail.setOrderNo(orderChild.getOrderNo());

        settlementDriverDetail.setDriverUserNo(orderChild.getTruckOwnUserNo());
        settlementDriverDetail.setDriverName(orderChild.getTruckOwnName());
        settlementDriverDetail.setTruckNo(orderChild.getTruckNo());

        settlementDriverDetail.setGoodsId(orderChild.getGoodsId());
        settlementDriverDetail.setGoodsName(orderChild.getGoodsName());

        // 运费
        settlementDriverDetail.setFreightPrice(orderChild.getFreightPrice());     //平台运费
        settlementDriverDetail.setWeight(weightCalc(orderChild.getLoadNet(), orderChild.getUnloadNet()));
        settlementDriverDetail.setFreight(freightCalc(settlementDriverDetail.getFreightPrice(), settlementDriverDetail.getWeight()));

        // 亏吨
        settlementDriverDetail.setLossPrice(orderChild.getLossPrice());
        settlementDriverDetail.setLossWeight(lossWeightCalc(orderChild.getLoadNet(), orderChild.getUnloadNet()));
        settlementDriverDetail.setLossFreight(lossFreightCalc(settlementDriverDetail.getLossPrice(), settlementDriverDetail.getLossWeight()));

        // 预付运费
        settlementDriverDetail.setPrepayFreight(BigDecimal.ZERO);

        // 结算金额
        settlementDriverDetail.setSettlementFreight(settlementFreightCalc(settlementDriverDetail.getFreight(), settlementDriverDetail.getLossFreight()));

        settlementDriverDetail.setInvoicingCompanyId(orderChild.getInvoicingCompanyId());
        settlementDriverDetail.setInvoicingCompanyShorterName(orderChild.getInvoicingCompanyShorterName());
        settlementDriverDetail.setInvoicingCompanyGroupCode(orderChild.getInvoicingCompanyGroupCode());

        // 平台服务费： 司机结算金额*平台服务费费率
        settlementDriverDetail.setPlatformServiceFeeRate(orderChild.getPlatformServiceFeeRate()==null
                ? BigDecimal.ZERO
                : orderChild.getPlatformServiceFeeRate());
        settlementDriverDetail.setPlatformServiceFee(
                settlementDriverDetail.getSettlementFreight()
                        .multiply(settlementDriverDetail.getPlatformServiceFeeRate().movePointLeft(2))
                                .setScale(0, RoundingMode.HALF_UP));

        settlementDriverDetailDao.saveEntity(settlementDriverDetail);

        return settlementDriverDetail.getId();
    }

    private BigDecimal weightCalc(BigDecimal loadNet, BigDecimal unLoadNet){
        return unLoadNet.compareTo(loadNet) < 0? unLoadNet : loadNet;
    }

    /**
     *  运费计算
     */
    public BigDecimal freightCalc(BigDecimal freightPrice, BigDecimal weight){

        return freightPrice.multiply(weight).setScale(0, RoundingMode.HALF_UP);
    }

    /**
     * 亏吨计算
     */
    private BigDecimal lossWeightCalc(BigDecimal loadNet, BigDecimal unLoadNet){
        return unLoadNet.compareTo(loadNet) < 0? loadNet.subtract(unLoadNet) : BigDecimal.ZERO;
    }

    /**
     * 亏吨运费
     */
    private BigDecimal lossFreightCalc(BigDecimal freightPrice, BigDecimal weight){
        return freightPrice.multiply(weight).setScale(0, RoundingMode.HALF_UP);

    }

    /**
     * 结算金额
     */
    private BigDecimal settlementFreightCalc(BigDecimal freight, BigDecimal lossFreight){
        return freight.subtract(lossFreight);
    }

    @Override
    public IPage<CarrierPageSettlementDriverDetailVO> pageSettlementDriverDetail(
            PageCarrierSettlementDriverDetailParam param) {
        IPage<SettlementDriverDetail> result = settlementDriverDetailDao.pageSettlementDriverDetail(param);
        List<CarrierPageSettlementDriverDetailVO> list = settlementOwnerDetailStruct.covertList(result.getRecords());
        return new Page<CarrierPageSettlementDriverDetailVO>().setRecords(list).setTotal(result.getTotal()).setPages(result.getPages());
    }

    @Override
    public SettlementDriverDetail selectOneByChildNo(String childNo) {
        return settlementDriverDetailDao.getByChildNo(childNo).get();
    }

    public static String formatLoanMsg(Integer loanFlag) {
        if (Objects.equals(loanFlag, OwnerLoanRecordEnum.LoanFlag.LOAN.getCode())) {
            return OwnerLoanRecordEnum.LoanFlag.LOAN.getMsg();
        }else if (Objects.equals(loanFlag, OwnerLoanRecordEnum.LoanFlag.NO_LOAN.getCode())) {
            return OwnerLoanRecordEnum.LoanFlag.NO_LOAN.getMsg();
        } else if (Objects.equals(loanFlag, OwnerLoanRecordEnum.LoanFlag.RE_PAY.getCode())) {
            return OwnerLoanRecordEnum.LoanFlag.RE_PAY.getMsg();
        }
        return "无需借款";
    }
}
