package com.clx.performance.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.clx.order.enums.OrderEnum;
import com.clx.order.feign.OrderFeign;
import com.clx.order.vo.feign.FeignOrderVO;
import com.clx.performance.dao.OrderChildDao;
import com.clx.performance.dao.OrderGoodsDao;
import com.clx.performance.enums.OrderChildEnum;
import com.clx.performance.enums.OrderGoodsStatusEnum;
import com.clx.performance.enums.TruckDemandEnum;
import com.clx.performance.model.OrderChild;
import com.clx.performance.model.OrderGoods;
import com.clx.performance.service.OrderCancelService;
import com.clx.performance.service.OrderChildService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Slf4j
@Service
@AllArgsConstructor
public class OrderCancelServiceImpl implements OrderCancelService {


    private final OrderFeign orderFeign;

    private final OrderChildDao orderChildDao;

    private final OrderGoodsDao orderGoodsDao;

    @Transactional(rollbackFor = Exception.class)
    public void ownerCancelOrderPre(String orderNo) {
        FeignOrderVO orderInfoFeign = orderFeign.getOrderInfoFeign(orderNo);
        BigDecimal orderResidueWeight = new BigDecimal(orderInfoFeign.getResidueWeight());
        if (BigDecimal.ZERO.compareTo(orderResidueWeight) == 0) {
            throw new RuntimeException("需要承运吨数的车辆,已全部到达货源地,无法取消");
        }
        List<OrderGoods> orderGoodsList = orderGoodsDao.getOrderGoodsListByOrderNo(orderNo);
        if (CollectionUtil.isEmpty(orderGoodsList)) {
            //未产生货单
            orderFeign.updateOrderStatusByOrderNo(orderNo, OrderEnum.Status.CANCELED.getCode());
        }
        BigDecimal orderGoodsResidueWeight = orderGoodsList.stream().map(OrderGoods::getResidueTransportWeight).reduce(BigDecimal::add).get();
        if (BigDecimal.ZERO.compareTo(orderGoodsResidueWeight) == 0) {
            throw new RuntimeException("需要承运吨数的车辆,已全部到达货源地,无法取消");
        }
        List<OrderChild> orderChildren = orderChildDao.selectResidueWeight(orderNo);

        if (CollectionUtil.isEmpty(orderChildren)) {
            //说明此刻未产生运单
            for (OrderGoods orderGoods : orderGoodsList) {
                OrderGoods update = new OrderGoods();
                update.setId(orderGoods.getId());
                update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.CANCEL.getCode());
                orderGoodsDao.updateEntityByKey(update);
            }
            orderFeign.updateOrderStatusByOrderNo(orderNo, OrderEnum.Status.CANCELED.getCode());

        } else {
            for (OrderGoods orderGoods : orderGoodsList) {
                OrderGoods update = new OrderGoods();
                update.setId(orderGoods.getId());
                update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.COMPLETED.getCode());
                orderGoodsDao.updateEntityByKey(update);
            }
            orderFeign.updateOrderStatusByOrderNo(orderNo, OrderEnum.Status.COMPLETED.getCode());

        }

    }


    @Transactional(rollbackFor = Exception.class)
    public void ownerCancelOrder(String orderNo) {
        FeignOrderVO orderInfoFeign = orderFeign.getOrderInfoFeign(orderNo);
        BigDecimal orderResidueWeightAll = BigDecimal.ZERO;
        List<OrderGoods> orderGoodsList = orderGoodsDao.getOrderGoodsListByOrderNo(orderNo);
        Map<Integer, BigDecimal> truckDemandMap = new HashMap<>();
        truckDemandMap.put(TruckDemandEnum.TruckType.PLATFORM_CAR.getCode(), BigDecimal.ZERO);
        truckDemandMap.put(TruckDemandEnum.TruckType.PLATFORM_ANY_CAR.getCode(), BigDecimal.ZERO);
        truckDemandMap.put(TruckDemandEnum.TruckType.OWNER_CAR.getCode(), BigDecimal.ZERO);

        if (OrderEnum.Status.COMPLETED.getCode().equals(orderInfoFeign.getOrderStatus())) {
            //已完结
            List<OrderChild> orderChildrenList = orderChildDao.selectResidueWeight(orderNo);

            for (OrderChild orderChild : orderChildrenList) {
                OrderChild update = new OrderChild();
                update.setId(orderChild.getId());
                update.setStatus(OrderChildEnum.Status.OWNER_CANCEL.getCode());
                orderChildDao.updateEntityByKey(update);
            }
            Map<String, List<OrderGoods>> orderGoodsMap = orderGoodsList.stream().collect(Collectors.groupingBy(OrderGoods::getOrderGoodsNo));

            Map<String, List<OrderChild>> orderChildMap = orderChildrenList.stream().collect(Collectors.groupingBy(OrderChild::getOrderGoodsNo));

            for (Map.Entry<String, List<OrderGoods>> entry : orderGoodsMap.entrySet()) {
                String orderGoodsNo = entry.getKey();
                OrderGoods orderGoods = entry.getValue().get(0);
                BigDecimal sum = orderChildMap.get(orderGoodsNo).stream().map(OrderChild::getWeight).reduce(BigDecimal::add).get();
                OrderGoods update = new OrderGoods();
                update.setId(orderGoods.getId());
                update.setResidueTransportWeight(orderGoods.getResidueTransportWeight().add(sum));
                truckDemandMap.put(orderGoods.getTruckDemand(), truckDemandMap.get(orderGoods.getTruckDemand()).add(update.getResidueTransportWeight()));
            }

        } else if (OrderEnum.Status.CANCELED.getCode().equals(orderInfoFeign.getOrderStatus())) {
            //已取消
            for (OrderGoods orderGoods : orderGoodsList) {
                truckDemandMap.put(orderGoods.getTruckDemand(), truckDemandMap.get(orderGoods.getTruckDemand()).add(orderGoods.getResidueTransportWeight()));
            }
        }


    }
}
