package com.clx.performance.component;

import cn.hutool.core.collection.CollectionUtil;
import com.clx.order.enums.OrderEnum;
import com.clx.order.feign.OrderFeign;
import com.clx.order.param.feign.UpdateOrderInfoParam;
import com.clx.order.vo.feign.FeignOrderVO;
import com.clx.performance.constant.RedisConstants;
import com.clx.performance.dao.OrderGoodsDao;
import com.clx.performance.dao.OrderGoodsTruckBindDao;
import com.clx.performance.enums.OrderGoodsStatusEnum;
import com.clx.performance.enums.OrderGoodsTruckBindEnum;
import com.clx.performance.model.OrderGoods;
import com.msl.common.base.Optional;
import com.xxl.job.core.handler.annotation.XxlJob;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ZSetOperations;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Date;
import java.util.List;
import java.util.Set;

@Component
@Slf4j
@AllArgsConstructor
public class OrderGoodsStatusLazyComponent implements InitializingBean {

    private final RedisTemplate<String, String> redisTemplate;

    private final OrderGoodsDao orderGoodsDao;

    private final OrderFeign orderFeign;

    private final OrderGoodsTruckBindDao orderGoodsTruckBindDao;

    private final GoodsOrderTruckRecordComponent goodsOrderTruckRecordComponent;

    public void expireProduce(LocalDateTime localDateTime, String orderGoodsNo) {
        redisTemplate.opsForZSet().add(RedisConstants.ORDER_GOODS_STATUS_LAZY, orderGoodsNo, localDateTime.toInstant(ZoneOffset.of("+8")).toEpochMilli());
    }


    @XxlJob("OrderGoodsSuccess")
    @Transactional(rollbackFor = Exception.class)
    public void consuming() {
        log.info("货单已完结定时器启动");
        long nowTimeMillis = System.currentTimeMillis();
        LocalDateTime nowDateTime = new Date(nowTimeMillis).toInstant().atOffset(ZoneOffset.of("+8")).toLocalDateTime();
        Set<ZSetOperations.TypedTuple<String>> orderGoodsIds = redisTemplate.opsForZSet().rangeByScoreWithScores(
                RedisConstants.ORDER_GOODS_STATUS_LAZY,
                0, nowTimeMillis  //延时任务score最小值
                //延时任务score最大值（当前时间）
        );
        if (orderGoodsIds == null) {
            log.info("货单已完结定时器查询redis条数为null");
            return;
        }
        if (orderGoodsIds.isEmpty()) {
            log.info("货单已完结定时器查询redis条数为空");
            return;
        }
        if (!CollectionUtil.isEmpty(orderGoodsIds)) {
            log.info("货单已完结定时器查询redis条数：{}", orderGoodsIds.size());

            for (ZSetOperations.TypedTuple<String> orderGoodsId : orderGoodsIds) {
                log.info("货单" + orderGoodsId + "过了最晚拉运时间");
                String orderGoodsNo = orderGoodsId.getValue();
                log.info("处理当前货单编号:{}", orderGoodsNo);
                Optional<OrderGoods> optional = orderGoodsDao.getByOrderGoodsNo(orderGoodsNo);
                if (!optional.isPresent()) {
                    redisTemplate.opsForZSet().remove(RedisConstants.ORDER_GOODS_STATUS_LAZY, orderGoodsId.getValue());
                    log.info("当前货单编号未查询出货单数据，执行删除redis ORDER_GOODS_STATUS_LAZY id: {}", orderGoodsId.getValue());
                    continue;
                }
                OrderGoods orderGoods = optional.get();
                FeignOrderVO orderInfoFeign = orderFeign.getOrderInfoFeign(orderGoods.getOrderNo());
                if (orderInfoFeign == null) {
                    redisTemplate.opsForZSet().remove(RedisConstants.ORDER_GOODS_STATUS_LAZY, orderGoodsId.getValue());
                    log.info("当前货单编号未查询出订单数据，执行删除redis ORDER_GOODS_STATUS_LAZY id: {}", orderGoodsId.getValue());
                    continue;
                } else {
                    log.info("当前货单编号已经查出数据{}", orderGoodsNo);
                }
                if (OrderGoodsStatusEnum.Status.SUCCESS.getCode().equals(orderGoods.getOrderGoodsStatus())) {
                    log.info("当前货单状态判断为已完成");
                } else {
                    log.info("当前货单状态判断不是已完成：货单状态为{}", orderGoods.getOrderGoodsStatus());
                }

                if (OrderGoodsStatusEnum.Status.SUCCESS.getCode().equals(orderGoods.getOrderGoodsStatus())) {
                    log.info("当前货单状态为已完成");
                    //如果当前货单已完成，则判断之前所有货单是否等于订单总吨数，等于则更新订单为已完成
                    List<OrderGoods> list = orderGoodsDao.getOrderGoodsListByOrderNoAndLastArriveSendTime(orderGoods.getOrderNo(), nowDateTime);
                    BigDecimal childSum = list.stream().map(OrderGoods::getExtractWeight).reduce(BigDecimal.ZERO, BigDecimal::add);
                    log.info("算出所有货单总量:{}, 订单总吨数：{}", childSum, orderInfoFeign.getTransportWeight());

                    if (childSum.compareTo(new BigDecimal(orderInfoFeign.getTransportWeight())) == 0) {
                        log.info("已完成的货单已经等于订单的拉运吨数，提前更新订单状态 已完成");
                        //已完成的货单已经等于订单的拉运吨数，提前更新订单状态 已完成
                        UpdateOrderInfoParam updateOrderInfoParam = new UpdateOrderInfoParam();
                        updateOrderInfoParam.setOrderId(orderInfoFeign.getId());
                        updateOrderInfoParam.setOrderStatus(OrderEnum.Status.SUCCESS.getCode());
                        orderFeign.updateOrderInfo(updateOrderInfoParam);
                    } else {
                        log.info("判断吨数不一致：childSum{}, transportWeight {}", childSum, new BigDecimal(orderInfoFeign.getTransportWeight()));

                    }
                } else {
                    log.info("当前货单状态为已完结");
                    //更新货单已完结
                    orderGoodsDao.updateOrderGoodsStatusByOrderGoodsNo(orderGoodsNo, OrderGoodsStatusEnum.Status.COMPLETED.getCode());
                    orderGoodsTruckBindDao.updateOrderGoodsBindStatus(orderGoodsNo, OrderGoodsTruckBindEnum.Status.EXPIRE.getCode());
                    goodsOrderTruckRecordComponent.deleteTruckRecord(orderGoodsNo);

                    BigDecimal residueTransportWeight = orderGoods.getResidueTransportWeight();
                    UpdateOrderInfoParam updateOrderInfoParam = new UpdateOrderInfoParam();
                    updateOrderInfoParam.setOrderId(orderInfoFeign.getId());
                    updateOrderInfoParam.setResidueWeight(residueTransportWeight.add(new BigDecimal(orderInfoFeign.getResidueWeight())));
                    updateOrderInfoParam.setVehicleUsage(orderGoods.getVehicleUsage());

                    log.info("订单ID：{},返还订单剩余吨数{}", orderInfoFeign.getId(), residueTransportWeight);

                    //返还订单剩余吨数
                    orderFeign.updateOrderInfo(updateOrderInfoParam);
                }
                redisTemplate.opsForZSet().remove(RedisConstants.ORDER_GOODS_STATUS_LAZY, orderGoodsId.getValue());
                log.info("删除redis ORDER_GOODS_STATUS_LAZY id: {}", orderGoodsId.getValue());

            }
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        Optional<List<OrderGoods>> optional = orderGoodsDao.getOrderGoodsList();
        if (optional.isPresent()) {
            for (OrderGoods orderGoods : optional.get()) {
                redisTemplate.opsForZSet().add(RedisConstants.ORDER_GOODS_STATUS_LAZY, orderGoods.getOrderGoodsNo(), orderGoods.getLastArriveSendTime().toInstant(ZoneOffset.of("+8")).toEpochMilli());
            }
        }
    }
}
