package com.clx.performance.service.impl.breakcontract;

import com.clx.performance.constant.RedissonConstants;
import com.clx.performance.dao.breakcontract.BreakContractSettlementDriverDao;
import com.clx.performance.enums.BreakContractSettlementDriverEnum;
import com.clx.performance.enums.PerformanceResultEnum;
import com.clx.performance.model.breakcontract.BreakContractSettlementDriver;
import com.clx.performance.service.breakcontract.BreakContractJobHandlerService;
import com.clx.performance.service.payment.PayService;
import com.clx.performance.utils.RedisUtil;
import com.msl.common.exception.ServiceSystemException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Objects;
import java.util.UUID;

@Slf4j
@Service
public class BreakContractJobHandlerServiceImpl implements BreakContractJobHandlerService {

    @Autowired
    private BreakContractSettlementDriverDao breakContractSettlementDriverDao;

    @Autowired
    private PayService payService;

    @Autowired
    RedisUtil redisUtil;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public String pay(Integer settlementId){
        String uuid = UUID.randomUUID().toString();
        String key = RedissonConstants.BREAK_CONTRACT_SETTLEMENT_OWNER_BATCH_PAY_ID_LOCK + settlementId;
        try{
            boolean flag = redisUtil.lock(key, uuid,180);
            if(!flag){
                return "获取支付锁失败，操作频繁，请稍后重试";
            }
            String msg = null;
            BreakContractSettlementDriver breakContractSettlementDriver = breakContractSettlementDriverDao.getEntityByKey(settlementId).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);

            if (Objects.equals(breakContractSettlementDriver.getSettleStatus(), BreakContractSettlementDriverEnum.SettleStatus.YES.getMsg())){
                return msg;
            }

            try {
                pay(breakContractSettlementDriver);
            }catch (ServiceSystemException e){
                breakContractSettlementDriver.setPayRemark(e.getMessage());
                breakContractSettlementDriverDao.updatePayFail(breakContractSettlementDriver);
                msg = e.getMessage();
            }
            return msg;
        }catch (Exception e){
            log.warn("违约结算单支付上锁失败，结算单ID：{},异常原因:{}", settlementId, ExceptionUtils.getStackTrace(e));
            //解锁
            redisUtil.unlock(key,uuid);
            throw new ServiceSystemException(PerformanceResultEnum.TRY_LOCK_ERROR);
        }



    }

    /**
     * 支付
     */
    private void pay(BreakContractSettlementDriver settlement){

        // 车主
        if (Objects.equals(settlement.getBreakContractPartyType(), BreakContractSettlementDriverEnum.BreakContractPartyType.DRIVER.getCode())){

            payService.driverPay(settlement.getSettlementNo(), settlement.getTruckOwnUserNo(), settlement.getFigure().intValue());
        }
        // 平台
        else if (Objects.equals(settlement.getBreakContractPartyType(), BreakContractSettlementDriverEnum.BreakContractPartyType.PLATFORM.getCode())) {
            payService.payDriver(settlement.getSettlementNo(), settlement.getTruckOwnUserNo(), settlement.getFigure().intValue());

        }
    }

}
