package com.clx.performance.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.clx.order.enums.OrderEnum;
import com.clx.order.feign.OrderFeign;
import com.clx.order.vo.feign.FeignAddressVO;
import com.clx.order.vo.feign.FeignOrderInfoVO;
import com.clx.performance.constant.RedissonConstants;
import com.clx.performance.dao.*;
import com.clx.performance.enums.*;
import com.clx.performance.extranal.user.AddressService;
import com.clx.performance.extranal.user.DriverService;
import com.clx.performance.extranal.user.OrderService;
import com.clx.performance.model.*;
import com.clx.performance.param.app.*;
import com.clx.performance.param.pc.OrderChildCarrierCancelParam;
import com.clx.performance.param.pc.PageCarrierOrderChildParam;
import com.clx.performance.param.pc.PagePoundAuditParam;
import com.clx.performance.service.OrderChildLogService;
import com.clx.performance.service.OrderChildPoundLogService;
import com.clx.performance.service.OrderChildService;
import com.clx.performance.service.OrderGoodsService;
import com.clx.performance.struct.*;
import com.clx.performance.utils.spring.ApplicationContextUtils;
import com.clx.performance.vo.app.*;
import com.clx.performance.vo.pc.*;
import com.clx.user.enums.driver.DriverInfoEnum;
import com.clx.user.vo.feign.DriverTruckInfoFeignVo;
import com.msl.common.base.Optional;
import com.msl.common.enums.ResultCodeEnum;
import com.msl.common.exception.ServiceSystemException;
import com.msl.common.utils.DateUtils;
import com.msl.common.utils.LocalDateTimeUtils;
import com.msl.common.utils.gps.GpsUtil;
import com.msl.user.data.UserSessionData;
import com.msl.user.utils.TokenUtil;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.amqp.core.AmqpTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * @Author: aiqinguo
 * @Description: 运单表
 * @Date: 2023/09/18 11:34:50
 * @Version: 1.0
 */

@Slf4j
@Service
@AllArgsConstructor
public class OrderChildServiceImpl implements OrderChildService {
    private final AmqpTemplate rabbitTemplate;

    private final OrderGoodsDao orderGoodsDao;
    private final OrderGoodsTruckBindDao orderGoodsTruckBindDao;

    private final OrderChildDao orderChildDao;

    private final OrderChildImageDao orderChildImageDao;
    private final OrderChildPoundAuditDao orderChildPoundAuditDao;

    private final OrderChildLogService orderChildLogService;
    private final OrderChildPoundLogService orderChildPoundLogService;
    private final AddressService addressService;
    private final OrderService orderService;

    private final DriverService driverService;
    private final OrderChildStruct orderChildStruct;
    private final OrderGoodsService orderGoodsService;
    private final OrderChildPoundStruct orderChildPoundStruct;
    private final OrderChildFreightStruct orderChildFreightStruct;
    private final AddressStruct addressStruct;
    private final OrderChildLogStruct orderChildLogStruct;

    private final UniqueOrderNumService uniqueOrderNumService;
    private final OrderFeign orderFeign;
    private final RedissonClient redissonClient;

    @Override
    public SaveOrderChildVO saveOrderChild(OrderChildSaveParam param) {

        // 司机车辆锁
        saveOrderChildOrderUserTruckLock(param);

        // 货单锁
        return saveOrderChildOrderGoodsLock(param, ()-> ApplicationContextUtils.getBean(OrderChildService.class).doSaveOrderChild(param));

    }

    /**
     * 用户车辆锁
     */
    public void saveOrderChildOrderUserTruckLock(OrderChildSaveParam param) {
        Long userNo = param.getDriverUserNo();
        Integer truckId = param.getTruckId();
        RLock lock = redissonClient.getLock(RedissonConstants.ORDER_CHILD_SAVE_USER_TRUCK_LOCK+ userNo+":"+truckId);

        boolean flag;
        try{
            flag = lock.tryLock( 1, 3, TimeUnit.SECONDS);
        }catch (Exception e){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_SAVE_FAIL);
        }
        if (!flag){
            log.warn("接单异常，orderGoodsNo:{}, userNo:{}", param.getOrderGoodsNo(), userNo);
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_SAVE_FREQUENCY_ERROR);
        }
    }

    /**
     * 货单锁
     */
    public SaveOrderChildVO saveOrderChildOrderGoodsLock(OrderChildSaveParam param, Supplier<SaveOrderChildVO> function) {
        String orderGoodsNo = param.getOrderGoodsNo();
        RLock lock = redissonClient.getLock(RedissonConstants.ORDER_CHILD_SAVE_ORDER_GOODS_NO_LOCK+orderGoodsNo);

        try{
            boolean flag = lock.tryLock(15, 30, TimeUnit.SECONDS);
            if (!flag){throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_SAVE_FAIL);}

            return function.get();
        }catch (ServiceSystemException e){
            throw e;
        }catch (Exception e){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_SAVE_FAIL);
        }finally {
            if (lock.isLocked() && lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
        }

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public SaveOrderChildVO doSaveOrderChild(OrderChildSaveParam param) {

        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = childNoGenerate();

        // 查询司机车辆信息
        DriverTruckInfoFeignVo driverTruckInfo = driverService.getUserDetailInfo(param.getDriverUserNo(), param.getTruckId()).orElseThrow(ResultCodeEnum.FAIL);
        BigDecimal truckLoad = driverTruckInfo.getLoad();
        String truckNo = driverTruckInfo.getTruckNo();

        LocalDateTime now = LocalDateTime.now();
        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(param.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.ORDER_INVALID);
        // 状态验证
        if (!Objects.equals(orderGoods.getOrderGoodsStatus(), OrderGoodsStatusEnum.Status.PAYING.getCode())
                && !Objects.equals(orderGoods.getOrderGoodsStatus(), OrderGoodsStatusEnum.Status.GO_TO_SEND.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_INVALID1);
        }

//        // 车型限制
//        FeignOrderInfoVO orderInfo = orderService.getOrderInfo(orderGoods.getOrderNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
//        if (StringUtils.isNotBlank(orderInfo.getTruckModelList())){
//            List<JSONObject> truckModelList = JSON.parseArray(orderInfo.getTruckModelList(), JSONObject.class);
//            List<String> truckModelNameList = truckModelList.stream().map(item -> item.getString("truckModelName")).collect(Collectors.toList());
//            if (!truckModelNameList.contains(driverTruckInfo.getModel())){
//                throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_TRUCK_MODEL_ERROR);
//            }
//        }

        // 定向
        OrderGoodsTruckBind bind = null;
        List<OrderGoodsTruckBind> orderGoodsTruckBindList = orderGoodsTruckBindDao.getValidByTruckNo(truckNo).orNull();
        if (Objects.equals(orderGoods.getPendingOrderWay(), OrderGoodsPendingOrderWayStatusEnum.Status.EXCLUSIVE.getCode())){
            if (CollectionUtils.isEmpty(orderGoodsTruckBindList)){throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_DIRECT_ORDER_TRUCK_ERROR);}
            //筛选出当前选择的定向货单
            bind = orderGoodsTruckBindList.stream().filter(s -> orderGoods.getOrderGoodsNo().equals(s.getOrderGoodsNo())).findFirst().orElse(null);
            if (Objects.isNull(bind)){throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_DIRECT_ORDER_TRUCK_ERROR);}
        }
        else {
            if (CollectionUtils.isNotEmpty(orderGoodsTruckBindList)){throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_DIRECT_ORDER_TRUCK_ERROR1);}
        }

        // 库存验证
        if (orderGoods.getResidueTransportWeight().compareTo(BigDecimal.ZERO) <= 0) {throw new ServiceSystemException(PerformanceResultEnum.ORDER_WEIGHT_LACK);}
//        if (orderGoods.getResidueTransportWeight().compareTo(truckLoad) < 0) {throw new ServiceSystemException(PerformanceResultEnum.ORDER_WEIGHT_LACK);}

        OrderChild orderChild = new OrderChild();
        orderChild.setChildNo(childNo);
        orderChild.setUserNo(userNo);

        orderChild.setOwnerUserNo(orderGoods.getUserNo());
        orderChild.setOwnerName(orderGoods.getUserName());

        orderChild.setOrderNo(orderGoods.getOrderNo());
        orderChild.setOrderGoodsNo(orderGoods.getOrderGoodsNo());
        orderChild.setFreightPrice(orderGoods.getPendingOrderFreight());
        orderChild.setLossPrice(orderGoods.getLossPrice());
        orderChild.setOrderFreightPrice(BigDecimal.ZERO);

        orderChild.setGoodsId(orderGoods.getGoodsId());
        orderChild.setGoodsName(orderGoods.getGoodsName());

        orderChild.setSendAddressId(orderGoods.getSendAddressId());
        orderChild.setSendAddress(orderGoods.getSendAddressShorter());
        orderChild.setReceiveAddressId(orderGoods.getReceiveAddressId());
        orderChild.setReceiveAddress(orderGoods.getReceiveAddressShorter());

        orderChild.setLoadDeadline(orderGoods.getLastArriveSendTime());

        orderChild.setDriverUserNo(driverTruckInfo.getUserNo());
        orderChild.setDriverName(driverTruckInfo.getName());
        orderChild.setDriverMobile(driverTruckInfo.getMobile());

        orderChild.setTruckOwnUserNo(driverTruckInfo.getTruckOwnUserNo());
        orderChild.setTruckId(driverTruckInfo.getTruckId());
        orderChild.setTruckNo(driverTruckInfo.getTruckNo());
        orderChild.setTruckLoad(truckLoad);
        orderChild.setTruckModel(driverTruckInfo.getModel());

        orderChild.setWeight(orderChild.getTruckLoad());
        orderChild.setFreight(orderChildFreightCalc(orderChild));
        orderChild.setPayTime(now);
        orderChild.setStatus(OrderChildEnum.Status.CREATED.getCode());
        orderChild.setCreateTime(now);

        // 更新货单数据
        updateOrderGoodsAmount(orderGoods, truckLoad);

        // 更新定向单状态
        updateOrderGoodsDirect(bind);

        // 新增运单
        orderChildDao.saveEntity(orderChild);

        // 更新出车状态
        updateDriverOrderStatusLock(orderChild.getDriverUserNo(), orderChild.getTruckId());

        // 新增日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.CREATED.getCode(), OrderChildLogEnum.Type.CREATED.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());

        SaveOrderChildVO result = new SaveOrderChildVO();
        result.setChildNo(childNo);
        return result;
    }

    @Override
    public void updateReject(OrderChildRejectParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(param.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.ORDER_INVALID);
        // 状态验证
        if (!Objects.equals(orderGoods.getOrderGoodsStatus(), OrderGoodsStatusEnum.Status.PAYING.getCode())
                && !Objects.equals(orderGoods.getOrderGoodsStatus(), OrderGoodsStatusEnum.Status.GO_TO_SEND.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_INVALID);
        }

        // 定向派单
        if (!Objects.equals(orderGoods.getPendingOrderWay(), 2)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_INVALID);
        }

        OrderGoodsTruckBind orderGoodsTruckBind = orderGoodsTruckBindDao.getValidByOrderGoodsNoAndTruckNo(param.getTruckNo())
                .orElseThrow(PerformanceResultEnum.ORDER_CHILD_DIRECT_REJECT_TRUCK_ERROR);

        if (!Objects.equals(orderGoodsTruckBind.getOrderGoodsNo(), orderGoods.getOrderGoodsNo())){throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_DIRECT_REJECT_TRUCK_ERROR);}

        // 更新定向派单
        updateOrderGoodsDirectReject(orderGoodsTruckBind);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateGotoSendAddress(OrderChildGoToSendAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (Objects.equals(orderChild.getDriverUserNo(), userNo)){
        }
        else if (Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_OPERATION_FORBID);
        }
        else {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_SEND.getCode())){return;}
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.CREATED.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        orderChild.setStatus(OrderChildEnum.Status.GO_TO_SEND.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.GO_TO_SEND.getCode(), OrderChildLogEnum.Type.GO_TO_SEND.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateArriveSendAddress(OrderChildArriveSendAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (Objects.equals(orderChild.getDriverUserNo(), userNo)){
        }
        else if (Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_OPERATION_FORBID);
        }
        else {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_SEND.getCode())){return;}
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_SEND.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }
        // 装货超时
        if (orderChild.getLoadDeadline().isBefore(LocalDateTime.now())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_LOAD_TIMEOUT);
        }

        // 距离验证
        FeignOrderInfoVO orderInfo = orderService.getOrderInfo(orderChild.getOrderNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
        double distance = GpsUtil.distance(orderGoods.getSendLongitude().doubleValue(), orderGoods.getSendLatitude().doubleValue(),
                param.getLongitude().doubleValue(), param.getLatitude().doubleValue())/1000;
        if (distance > orderInfo.getSendDriverArriveRange().doubleValue()){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_ARRIVE_SEND_ADDRESS_DISTANCE_ERROR);
        }

        orderChild.setStatus(OrderChildEnum.Status.ARRIVE_SEND.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.ARRIVE_SEND.getCode(), OrderChildLogEnum.Type.ARRIVE_SEND.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateGotoReceiveAddress(OrderChildGoToReceiveAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (Objects.equals(orderChild.getDriverUserNo(), userNo)){
        }
        else if (Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_OPERATION_FORBID);
        }
        else {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_RECEIVE.getCode())){return;}
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.LOAD.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }


        orderChild.setStatus(OrderChildEnum.Status.GO_TO_RECEIVE.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.GO_TO_RECEIVE.getCode(), OrderChildLogEnum.Type.GO_TO_RECEIVE.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateArriveReceiveAddress(OrderChildArriveReceiveAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (Objects.equals(orderChild.getDriverUserNo(), userNo)){
        }
        else if (Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_OPERATION_FORBID);
        }
        else {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_RECEIVE.getCode())){return;}
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_RECEIVE.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        // 距离验证
        FeignOrderInfoVO orderInfo = orderService.getOrderInfo(orderChild.getOrderNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
        double distance = GpsUtil.distance(orderGoods.getReceiveLongitude().doubleValue(), orderGoods.getReceiveLatitude().doubleValue(),
                param.getLongitude().doubleValue(), param.getLatitude().doubleValue())/1000;
        if (distance > orderInfo.getReveiveDriverArriveRange().doubleValue()){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_ARRIVE_RECEIVE_ADDRESS_DISTANCE_ERROR);
        }

        orderChild.setStatus(OrderChildEnum.Status.ARRIVE_RECEIVE.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.ARRIVE_RECEIVE.getCode(), OrderChildLogEnum.Type.ARRIVE_RECEIVE.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateLoad(OrderChildLoadParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (Objects.equals(orderChild.getDriverUserNo(), userNo)){
        }
        else if (Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_OPERATION_FORBID);
        }
        else {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CANCEL.getCode())
                || Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.PLATFORM_CANCEL.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCELED);
        }

        if (!loadCheck(param.getLoadNet(), orderChild.getTruckLoad())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_LOAD_WEIGHT_ERROR);
        }

        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);

        if (orderChild.getLoadTime() == null && Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_SEND.getCode())) {
            updateLoadFirst(param, orderChild, orderGoods);
            // 日志
            orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.LOAD.getCode(), OrderChildLogEnum.Type.LOAD.getMsg(),
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        } else {
            updateReload(param, orderChild, orderGoods);
            // 日志
            orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.RELOAD.getCode(), OrderChildLogEnum.Type.RELOAD.getMsg(),
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        }
    }

    private void updateLoadFirst(OrderChildLoadParam param, OrderChild orderChild, OrderGoods orderGoods) {
        String childNo = param.getChildNo();

//        // 装车时间验证
//        FeignOrderInfoVO orderInfo = orderService.getOrderInfo(orderChild.getOrderNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
//        LocalDateTime now = LocalDateTime.now();
//        LocalDateTime beginTime = DateUtils.parseDateTime(DateUtils.formatDateTime(now, "yyyy-MM-dd ").get() + orderInfo.getLoadBeginTime()).get();
//        LocalDateTime endTime = DateUtils.parseDateTime(DateUtils.formatDateTime(now, "yyyy-MM-dd ").get() + orderInfo.getLoadEndTime()).get();
//        if (now.isBefore(beginTime) || now.isAfter(endTime)){
//            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_LOAD_TIME_ERROR);
//        }

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getLoadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.LOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        BigDecimal dif = param.getLoadNet().subtract(orderChild.getWeight());

        orderChild.setLoadRough(param.getLoadRough());
        orderChild.setLoadTare(param.getLoadTare());
        orderChild.setLoadNet(param.getLoadNet());
        orderChild.setLoadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));
        orderChild.setFreight(orderChildFreightCalc(orderChild));
        orderChild.setStatus(OrderChildEnum.Status.LOAD.getCode());

        // 更新装车净重
        updateOrderGoodsAmountLoad(orderGoods, dif);

        orderChildDao.updateLoad(orderChild);

        orderChildImageDao.batchSaveEntity(imageList);
    }

    private void updateReload(OrderChildLoadParam param, OrderChild orderChild, OrderGoods orderGoods) {
        String childNo = param.getChildNo();

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getLoadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.LOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        BigDecimal dif = param.getLoadNet().subtract(orderChild.getWeight());

        orderChild.setLoadRough(param.getLoadRough());
        orderChild.setLoadTare(param.getLoadTare());
        orderChild.setLoadNet(param.getLoadNet());
        orderChild.setLoadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));
        orderChild.setFreight(orderChildFreightCalc(orderChild));

        // 更新装车净重
        updateOrderGoodsAmountLoad(orderGoods, dif);

        orderChildDao.updateLoad(orderChild);

        orderChildImageDao.deleteLoad(childNo);
        orderChildImageDao.batchSaveEntity(imageList);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateUnload(OrderChildUnloadParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (Objects.equals(orderChild.getDriverUserNo(), userNo)){
        }
        else if (Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_OPERATION_FORBID);
        }
        else {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CANCEL.getCode())
                || Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.PLATFORM_CANCEL.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCELED);
        }

//        if (!loadCheck(param.getUnloadNet(), orderChild.getTruckLoad())){
//            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_UNLOAD_WEIGHT_ERROR);
//        }

        if (orderChild.getUnloadTime() == null && Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_RECEIVE.getCode())) {
            updateUnloadFirst(param, orderChild);
            // 日志
            orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.UNLOAD.getCode(), OrderChildLogEnum.Type.UNLOAD.getMsg(),
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
            orderChildPoundLogService.saveDriverOrderChildLog(childNo, OrderChildPoundAuditEnum.Status.AUDIT.getCode(), OrderChildPoundAuditEnum.Status.AUDIT.getMsg(),
                    OrderChildLogEnum.CreateType.DRIVER.getCode(),loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        } else {
            updateReUnload(param, orderChild);
            // 日志
            orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.REUNLOAD.getCode(), OrderChildLogEnum.Type.REUNLOAD.getMsg(),
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
            orderChildPoundLogService.saveDriverOrderChildLog(childNo, OrderChildPoundAuditEnum.Status.AUDIT.getCode(), OrderChildPoundAuditEnum.Status.AUDIT.getMsg(),
                    OrderChildLogEnum.CreateType.DRIVER.getCode(),loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        }
    }

    private void updateUnloadFirst(OrderChildUnloadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

//        // 装车时间验证
//        FeignOrderInfoVO orderInfo = orderService.getOrderInfo(orderChild.getOrderNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
//        LocalDateTime now = LocalDateTime.now();
//        LocalDateTime beginTime = DateUtils.parseDateTime(DateUtils.formatDateTime(now, "yyyy-MM-dd ").get() + orderInfo.getUnloadBeginTime()).get();
//        LocalDateTime endTime = DateUtils.parseDateTime(DateUtils.formatDateTime(now, "yyyy-MM-dd ").get() + orderInfo.getUnloadEndTime()).get();
//        if (now.isBefore(beginTime) || now.isAfter(endTime)){
//            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_UNLOAD_TIME_ERROR);
//        }

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getUnloadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.UNLOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        orderChild.setUnloadPoundNo(param.getUnloadPoundNo());
        orderChild.setUnloadRough(param.getUnloadRough());
        orderChild.setUnloadTare(param.getUnloadTare());
        orderChild.setUnloadNet(param.getUnloadNet());
        orderChild.setUnloadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));
        orderChild.setStatus(OrderChildEnum.Status.UNLOAD.getCode());
        orderChild.setPoundStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());
        orderChild.setFreight(orderChildFreightCalc(orderChild));

        OrderChildPoundAudit audit = new OrderChildPoundAudit();
        audit.setChildNo(childNo);
        audit.setStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());
        audit.setLoadNet(orderChild.getLoadNet());
        audit.setUnloadNet(orderChild.getUnloadNet());

        orderChildDao.updateUnload(orderChild);

        orderChildImageDao.batchSaveEntity(imageList);

        orderChildPoundAuditDao.saveEntity(audit);

        // 更新出车状态
        updateDriverOrderStatusUnload(orderChild.getDriverUserNo(), orderChild.getTruckId());

    }

    private void updateReUnload(OrderChildUnloadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        // 审核中
        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.AUDIT.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_POUND_AUDIT);
        }

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getUnloadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.UNLOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        orderChild.setUnloadPoundNo(param.getUnloadPoundNo());
        orderChild.setUnloadRough(param.getUnloadRough());
        orderChild.setUnloadTare(param.getUnloadTare());
        orderChild.setUnloadNet(param.getUnloadNet());
        orderChild.setUnloadTime(LocalDateTime.now());
        orderChild.setPoundStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());
        orderChild.setFreight(orderChildFreightCalc(orderChild));

        OrderChildPoundAudit audit = new OrderChildPoundAudit();
        audit.setChildNo(childNo);
        audit.setStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());
        audit.setLoadNet(orderChild.getLoadNet());
        audit.setUnloadNet(orderChild.getUnloadNet());
        audit.setUnloadPoundNo(orderChild.getUnloadPoundNo());

        orderChildDao.updateUnload(orderChild);

        orderChildImageDao.deleteUnload(childNo);
        orderChildImageDao.batchSaveEntity(imageList);

        orderChildPoundAuditDao.saveEntity(audit);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateLoadAndUnloadAgain(OrderChildLoadAndUnloadAgainParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();
        String childNo = param.getChildNo();
        LocalDateTime now = LocalDateTime.now();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (Objects.equals(orderChild.getDriverUserNo(), userNo)){
        }
        else if (Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_OPERATION_FORBID);
        }
        else {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CANCEL.getCode())
                || Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.PLATFORM_CANCEL.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCELED);
        }
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.UNLOAD.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.AUDIT.getCode())){return;}
        if (!Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.REJECT.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        if (!loadCheck(param.getLoadNet(), orderChild.getTruckLoad())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_LOAD_WEIGHT_ERROR);
        }
//        if (!loadCheck(param.getUnloadNet(), orderChild.getTruckLoad())){
//            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_UNLOAD_WEIGHT_ERROR);
//        }

        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.ORDER_INVALID);

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getLoadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.LOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }
        for (String item : param.getUnloadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.UNLOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        BigDecimal dif = param.getLoadNet().subtract(orderChild.getWeight());

        orderChild.setLoadRough(param.getLoadRough());
        orderChild.setLoadTare(param.getLoadTare());
        orderChild.setLoadNet(param.getLoadNet());
        orderChild.setUnloadRough(param.getUnloadRough());
        orderChild.setUnloadTare(param.getUnloadTare());
        orderChild.setUnloadNet(param.getUnloadNet());
        orderChild.setWeight(orderChildWeightCalc(orderChild));
        orderChild.setFreight(orderChildFreightCalc(orderChild));

        orderChild.setPoundStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());

        OrderChildPoundAudit audit = new OrderChildPoundAudit();
        audit.setChildNo(childNo);
        audit.setStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());
        audit.setLoadNet(orderChild.getLoadNet());
        audit.setUnloadNet(orderChild.getUnloadNet());
        audit.setUnloadPoundNo(orderChild.getUnloadPoundNo());

        // 更新装车净重
        updateOrderGoodsAmountLoad(orderGoods, dif);

        orderChildDao.updateLoadAndUnload(orderChild);

        orderChildImageDao.deleteLoadAndUnload(childNo);
        orderChildImageDao.batchSaveEntity(imageList);

        orderChildPoundAuditDao.saveEntity(audit);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.RELOAD_AND_REUNLOAD.getCode(), OrderChildLogEnum.Type.RELOAD_AND_REUNLOAD.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());

        orderChildPoundLogService.saveDriverOrderChildLog(childNo, OrderChildPoundAuditEnum.Status.AUDIT.getCode(), OrderChildPoundAuditEnum.Status.AUDIT.getMsg(),
                OrderChildLogEnum.CreateType.DRIVER.getCode(),loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDriverConfirm(OrderChildDriverConfirmParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (Objects.equals(orderChild.getDriverUserNo(), userNo)){
        }
        else if (Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_OPERATION_FORBID);
        }
        else {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.UNLOAD.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        // 审核中
        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.AUDIT.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_POUND_AUDIT);
        }
        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.REJECT.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_POUND_REJECT);
        }

        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
        long count = orderChildDao.countOfTransitByOrderGoodsNo(orderChild.getOrderGoodsNo());

        // 更新货单完成状态
        updateOrderGoodsComplete(orderGoods, count);

        orderChild.setStatus(OrderChildEnum.Status.UNSETTLE.getCode());
        orderChildDao.updateDriverConfirm(orderChild);

        // 更新车辆状态
        updateDriverOrderStatusDriverConfirm(orderChild.getTruckId());

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.DRIVER_CONFIRM.getCode(), OrderChildLogEnum.Type.DRIVER_CONFIRM.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDriverCancel(OrderChildDriverCancelParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();
        LocalDateTime now = LocalDateTime.now();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (Objects.equals(orderChild.getDriverUserNo(), userNo)){
        }
        else if (Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_OPERATION_FORBID);
        }
        else {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        if (OrderChildEnum.CANCEL_lIST.contains(orderChild.getStatus())){
            return;
        }

        // 禁止取消
        if (orderChild.getStatus()>=OrderChildEnum.Status.LOAD.getCode()) {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCEL_FORBID);
        }

        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);

        // 取消量验证
        if (!cancelCountCheck(userNo)){throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCEL_FORBID_COUNT);}

        // 定向
        OrderGoodsTruckBind orderGoodsTruckBind = null;
        if (Objects.equals(orderGoods.getPendingOrderWay(), OrderGoodsPendingOrderWayStatusEnum.Status.EXCLUSIVE.getCode())){
            orderGoodsTruckBind = orderGoodsTruckBindDao.getByOrderGoodsNoAndTruckNo(orderGoods.getOrderGoodsNo(), orderChild.getTruckNo())
                    .orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
        }

        orderChild.setCancelRemark(param.getRemark());
        orderChild.setCancelTime(now);
        orderChild.setFinishTime(now);
        orderChild.setStatus(OrderChildEnum.Status.DRIVER_CANCEL.getCode());

        // 返回吨数
        updateOrderGoodsAmountReturn(orderChild, orderGoods);

        // 取消定向
        updateOrderGoodsDirectCancel(orderGoodsTruckBind);

        orderChildDao.updateCancel(orderChild);

        // 更新出车状态
        updateDriverOrderStatusCancel(orderChild.getDriverUserNo(), orderChild.getTruckId());

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.DRIVER_CANCEL.getCode(), OrderChildLogEnum.Type.DRIVER_CANCEL.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    public void updateCarrierCancel(OrderChildCarrierCancelParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();
        LocalDateTime now = LocalDateTime.now();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);

        if (OrderChildEnum.CANCEL_lIST.contains(orderChild.getStatus())){
            return;
        }

        // 禁止取消 (卸车前取消)
        if (orderChild.getStatus()>=OrderChildEnum.Status.UNLOAD.getCode()) {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCEL_FORBID);
        }

        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);

        // 定向
        OrderGoodsTruckBind orderGoodsTruckBind = null;
        if (Objects.equals(orderGoods.getPendingOrderWay(), OrderGoodsPendingOrderWayStatusEnum.Status.EXCLUSIVE.getCode())){
            orderGoodsTruckBind = orderGoodsTruckBindDao.getByOrderGoodsNoAndTruckNo(orderGoods.getOrderGoodsNo(), orderChild.getTruckNo())
                    .orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
        }

        orderChild.setCancelRemark(param.getRemark());
        orderChild.setCancelTime(now);
        orderChild.setFinishTime(now);
        orderChild.setStatus(OrderChildEnum.Status.PLATFORM_CANCEL.getCode());


        // 返回吨数
        updateOrderGoodsAmountReturn(orderChild, orderGoods);

        // 取消定向
        updateOrderGoodsDirectCancel(orderGoodsTruckBind);

        orderChildDao.updateCancel(orderChild);

        // 更新出车状态
        updateDriverOrderStatusCancel(orderChild.getDriverUserNo(), orderChild.getTruckId());

        // 日志
        orderChildLogService.saveCarrierOrderChildLog(childNo, OrderChildLogEnum.Type.PLATFORM_CANCEL.getCode(), OrderChildLogEnum.Type.PLATFORM_CANCEL.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    public OrderChildVO getOrderChildInfo(String childNo) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getDriverUserNo(), userNo)
                && !Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        OrderChildVO result = orderChildStruct.convert(orderChild);

        // 磅单
        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.REJECT.getCode())){
            Optional<OrderChildPoundAudit> poundAuditDetail = orderChildPoundAuditDao.getPoundAuditDetail(childNo);
            result.setPoundRemark(poundAuditDetail.orNull().getRemark());
            result.setPoundRejectType(poundAuditDetail.orNull().getRejectType());
        }
        List<OrderChildImage> imageList = orderChildImageDao.listLoadAndUnload(childNo).orElse(new ArrayList<>());
        result.setLoadImageList(imageList.stream().filter(item->Objects.equals(item.getType(),OrderChildImage.Type.LOAD.getCode())).map(item->item.getImage()).collect(Collectors.toList()));
        result.setUnloadImageList(imageList.stream().filter(item->Objects.equals(item.getType(),OrderChildImage.Type.UNLOAD.getCode())).map(item->item.getImage()).collect(Collectors.toList()));

        // 地址
        Optional<FeignAddressVO> sendAndReceiveAddress = addressService.getSendAndReceiveAddress(orderChild.getSendAddressId(), orderChild.getReceiveAddressId());
        if (sendAndReceiveAddress.isPresent()){
            result.setSendAddressInfo(addressStruct.convert(sendAndReceiveAddress.get().getSendAddress()));
        }
        if (sendAndReceiveAddress.isPresent()){
            result.setReceiveAddressInfo(addressStruct.convert(sendAndReceiveAddress.get().getReceiveAddress()));
        }

        return result;
    }

    @Override
    public IPage<OrderChildVO> pageOrderChild(PageOrderChildOfDriverParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        param.setUserNo(userNo);
        return orderChildDao.pageOrderChildOfDriver(param);

    }

    @Override
    public IPage<OrderChildVO> pageSearchOrderChild(PageOrderChildOfDriverSearchParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        param.setUserNo(userNo);
        return orderChildDao.pageSearchOrderChildOfDriver(param);
    }

    @Override
    public OrderChildEstimatedFreightVO getEstimatedFreight(OrderChildEstimatedFreightParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();
        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getDriverUserNo(), userNo)
                && !Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        OrderChildEstimatedFreightVO result = new OrderChildEstimatedFreightVO();
        result.setFreight(orderChild.getFreightPrice().multiply(param.getLoadNet()).setScale(0, RoundingMode.HALF_UP));

        return result;
    }

    @Override
    public OrderChildPoundInfoVO getOrderChildPoundInfo(String childNo) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getDriverUserNo(), userNo)
                && !Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        OrderChildPoundInfoVO result = orderChildPoundStruct.convert(orderChild);

        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.REJECT.getCode())){
            result.setPoundRemark(orderChildPoundAuditDao.getPoundAuditDetail(childNo).orNull().getRemark());
        }

        List<OrderChildImage> imageList = orderChildImageDao.listLoadAndUnload(childNo).orElse(new ArrayList<>());
        result.setLoadImageList(imageList.stream().filter(item->Objects.equals(item.getType(),OrderChildImage.Type.LOAD.getCode())).map(item->item.getImage()).collect(Collectors.toList()));
        result.setUnloadImageList(imageList.stream().filter(item->Objects.equals(item.getType(),OrderChildImage.Type.UNLOAD.getCode())).map(item->item.getImage()).collect(Collectors.toList()));

        return result;
    }

    @Override
    public OrderChildFreightInfoVO getOrderChildFreightInfo(String childNo) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getDriverUserNo(), userNo)
                && !Objects.equals(orderChild.getTruckOwnUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        OrderChildFreightInfoVO result = orderChildFreightStruct.convert(orderChild);

        return result;
    }


    /**
     * 更新货单数据
     */
    private void updateOrderGoodsAmount(OrderGoods orderGoods, BigDecimal weight){

        Integer status = OrderGoodsStatusEnum.Status.GO_TO_SEND.getCode();

        orderGoodsService.updateOrderGoodsReduceWeightAndStatus(orderGoods.getId(), weight, status);
        orderFeign.updateOrderStatusByOrderNo(orderGoods.getOrderNo(), OrderEnum.Status.IN_TRANSIT.getCode());
    }

    /**
     * 取消返吨数
     */
    private void updateOrderGoodsAmountReturn(OrderChild orderChild, OrderGoods orderGoods){
        int count = orderChildDao.countValidByOrderGoodsNo(orderGoods.getOrderGoodsNo())-1;

        Integer status = null;
        if (count == 0){
            status = OrderGoodsStatusEnum.Status.PAYING.getCode();
        }
        else {
            status = OrderGoodsStatusEnum.Status.GO_TO_SEND.getCode();
        }

        orderGoodsService.updateOrderGoodsReduceWeightAndStatus(orderGoods.getId(), orderChild.getWeight().negate(), status);
    }

    /**
     * 装车补偿
     */
    private void updateOrderGoodsAmountLoad(OrderGoods orderGoods, BigDecimal dif){
        if (dif.compareTo(BigDecimal.ZERO) == 0){return;}

        Integer status = OrderGoodsStatusEnum.Status.GO_TO_SEND.getCode();

        orderGoodsService.updateOrderGoodsReduceWeightAndStatus(orderGoods.getId(), dif, status);
    }

    /**
     * 更新定向单状态 (接单)
     */
    private void updateOrderGoodsDirect(OrderGoodsTruckBind orderGoodsTruckBind){
        if (orderGoodsTruckBind == null) {return;}

        orderGoodsTruckBind.setStatus(OrderGoodsTruckBind.Status.ORDER.getCode());
        orderGoodsTruckBindDao.updateStatus(orderGoodsTruckBind);
    }

    /**
     * 更新定向派单状态 (拒绝)
     */
    private void updateOrderGoodsDirectReject(OrderGoodsTruckBind orderGoodsTruckBind){
        orderGoodsTruckBind.setStatus(OrderGoodsTruckBind.Status.CANCEL.getCode());
        orderGoodsTruckBindDao.updateStatus(orderGoodsTruckBind);
    }

    /**
     * 更新定向派单状态 (取消)
     */
    private void updateOrderGoodsDirectCancel(OrderGoodsTruckBind orderGoodsTruckBind){
        if (orderGoodsTruckBind == null) {return;}

        orderGoodsTruckBind.setStatus(OrderGoodsTruckBind.Status.CANCEL.getCode());
        orderGoodsTruckBindDao.updateStatus(orderGoodsTruckBind);
    }

    /**
     * 更新货单完成状态
     */
    private void updateOrderGoodsComplete(OrderGoods orderGoods, long count){
        if (count > 1) {return;}

        if (orderGoods.getResidueTransportWeight().compareTo(BigDecimal.ZERO) <= 0){
            orderGoodsDao.updateOrderGoodsStatusByOrderGoodsNo(orderGoods.getOrderGoodsNo(), OrderGoodsStatusEnum.Status.SUCCESS.getCode());
            orderGoodsTruckBindDao.updateOrderGoodsBindStatus(orderGoods.getOrderGoodsNo(), OrderGoodsTruckBindEnum.Status.EXPIRE.getCode());
        }

    }

    /**
     * 拉运吨数检测
     */
    private boolean loadCheck(BigDecimal weight, BigDecimal truckLoad){
        if (weight.compareTo(truckLoad.multiply(new BigDecimal("1.5"))) > 0) {return false;}
        return true;
    }

    /**
     * 取消检测
     */
    private boolean cancelCountCheck(Long userNo){
        LocalDateTime startTime = DateUtils.parseDateTime(DateUtils.formatDateTime(LocalDateTime.now(), "yyyy-MM-dd").get() + " 00:00:00").get();
        LocalDateTime endTime = DateUtils.parseDateTime(DateUtils.formatDateTime(LocalDateTime.now(), "yyyy-MM-dd").get() + " 23:59:59").get();

        long count = orderChildDao.countOfCancel(userNo, startTime, endTime);

        return count<10;
    }

    /**
     * 运单拉运吨数计算
     */
    private BigDecimal orderChildWeightCalc(OrderChild orderChild){
        if (orderChild.getLoadNet() == null) {return orderChild.getTruckLoad();}
        else {return orderChild.getLoadNet();}
    }

    /**
     * 运费计算
     */
    private BigDecimal orderChildFreightCalc(OrderChild orderChild){
        BigDecimal totalFreight = orderChild.getFreightPrice().multiply(orderChild.getWeight());
        if (orderChild.getUnloadNet() != null){
            BigDecimal dif = orderChild.getUnloadNet().subtract(orderChild.getWeight());
            if (dif.compareTo(BigDecimal.ZERO) < 0){
                totalFreight = totalFreight.add(orderChild.getLossPrice().multiply(dif));
            }
        }

        return totalFreight.setScale(0, RoundingMode.HALF_UP);
    }


    /**
     * 更新出车状态 (接单， 运单重车)
     */
    private void updateDriverOrderStatusLock(Long driverUserNo, Integer truckId){
        driverService.updateOrderStatus(driverUserNo, DriverInfoEnum.DriverStatus.YES.getCode(), truckId, 3);

    }

    /**
     * 更新出车状态（运单卸车，运单空车）
     */
    private void updateDriverOrderStatusUnload(Long driverUserNo, Integer truckId){
        driverService.updateOrderStatus(driverUserNo, DriverInfoEnum.DriverStatus.NO.getCode(), truckId, 2);
    }

    /**
     * 更新出车状态 (运单取消，空车)
     */
    private void updateDriverOrderStatusCancel(Long driverUserNo, Integer truckId){
        driverService.updateOrderStatus(driverUserNo, DriverInfoEnum.DriverStatus.NO.getCode(), truckId, 1);
    }

    /**
     * 更新出车状态 (司机确认收货，空车)
     */
    private void updateDriverOrderStatusDriverConfirm(Integer truckId){
        driverService.updateOrderStatus(0L, 0, truckId, 1);
    }

    /**
     * 创建运单号
     */
    private String childNoGenerate(){
        return "CYD"+uniqueOrderNumService.getUniqueOrderNum(LocalDateTimeUtils.convertLocalDateTimeToString(LocalDateTime.now(),LocalDateTimeUtils.DATE_DAY));
    }

    @Override
    public IPage<PageOrderChildPoundAuditVO> pagePoundAuditList(PagePoundAuditParam param) {
        return orderChildDao.pagePoundAuditList(param);
    }

    @Override
    public IPage<PageCarrierOrderChildVO> pageCarrierOrderChildList(PageCarrierOrderChildParam param) {
        return orderChildDao.pageCarrierOrderChildList(param);
    }

    @Override
    public List<OrderChildPCVO> getOrderChildInfoByOrderGoodsNo(String orderGoodsNo) {
        List<OrderChild> orderChildren = orderChildDao.getOrderChildInfoByOrderGoodsNo(orderGoodsNo)
                .orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        return orderChildStruct.convertList(orderChildren);
    }

    @Override
    public CarrierOrderChildDetailVO getCarrierOrderChildDetail(String childNo) {
        //运单数据
        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        CarrierOrderChildDetailVO carrierOrderChildDetailVO = orderChildStruct.carrierConvert(orderChild);

        //货单数据
        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.ORDER_NO_FOUND);
        carrierOrderChildDetailVO.setExtractWeight(orderGoods.getExtractWeight());

        //磅单审核数据
        List<OrderChildPoundAudit> poundAuditList = orderChildPoundAuditDao.getPoundAuditList(childNo);
        carrierOrderChildDetailVO.setPoundAudits(orderChildPoundStruct.convert(poundAuditList));

        //运单日志
        List<OrderChildLog> orderChildLog = orderChildLogService.getOrderChildNodeLog(childNo);
        carrierOrderChildDetailVO.setChildLogs(orderChildLogStruct.convert(orderChildLog));

        //应付运费
        BigDecimal payableFreight=BigDecimal.ZERO;
        if(Objects.nonNull(orderChild.getLoadNet())){
            payableFreight = orderChild.getLoadNet()
                    .multiply(orderChild.getFreightPrice())
                    .setScale(0,BigDecimal.ROUND_HALF_UP);
        }

        //亏吨扣款（元）
        BigDecimal lossDeduction=BigDecimal.ZERO;
        if(Objects.nonNull(orderChild.getLoadNet()) && Objects.nonNull(orderChild.getUnloadNet())){
            if(orderChild.getLoadNet().compareTo(orderChild.getUnloadNet()) >0){
                lossDeduction = carrierOrderChildDetailVO.getLossPrice()
                        .multiply(orderChild.getLoadNet().subtract(orderChild.getUnloadNet()))
                        .setScale(0,BigDecimal.ROUND_HALF_UP);
            }
        }

        //实付运费（元）
        BigDecimal realFreight=payableFreight.subtract(lossDeduction);
        carrierOrderChildDetailVO.setPayableFreight(payableFreight);
        carrierOrderChildDetailVO.setLossDeduction(lossDeduction);
        carrierOrderChildDetailVO.setRealFreight(realFreight);

        //磅单图片
        List<OrderChildImage> loadImages = orderChildImageDao.getImages(childNo, OrderChildImage.Type.LOAD.getCode());
        List<OrderChildImage> unloadImages = orderChildImageDao.getImages(childNo, OrderChildImage.Type.UNLOAD.getCode());
        carrierOrderChildDetailVO.setLoadImageList(loadImages.stream().map(OrderChildImage::getImage).collect(Collectors.toList()));
        carrierOrderChildDetailVO.setUnloadImageList(unloadImages.stream().map(OrderChildImage::getImage).collect(Collectors.toList()));
        return carrierOrderChildDetailVO;
    }

    @Override
    public GoingOrderChildVO getGoingLatestOrderChild() {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        OrderChild orderChild = orderChildDao.getGoingLatestOrderChild(loginUserInfo.getUserNo());
        return orderChildStruct.convertGoingOrder(orderChild);
    }

    @Override
    public Integer getOrderChildTotalByUserNo(Long userNo) {
        return orderChildDao.getOrderChildTotalByUserNo(userNo);
    }

    @Override
    public OrderChildBussInfoVO getOrderChildBussInfo(String truckNo) {
        List<OrderChild> childList = orderChildDao.getOrderChildBussInfo(truckNo);
        if(CollectionUtils.isEmpty(childList)){
            OrderChildBussInfoVO vo = OrderChildBussInfoVO.builder().completeNum(0).historyLoadWeight(new BigDecimal("0"))
                    .completeCarryWeight(new BigDecimal("0")).loseWeight(new BigDecimal("0")).loseRate(new BigDecimal("0"))
                    .overRate(new BigDecimal("0")).build();
            return vo;
        }


        //运单完成数量
        int complateNum = childList.size();
        //历史装车总量
        BigDecimal historyLoadWeight = BigDecimal.ZERO;

        java.util.Optional<BigDecimal> reduce = childList.stream().map(OrderChild::getLoadNet).filter(
                Objects::nonNull).reduce(BigDecimal::add);
        if (reduce.isPresent()) {
            historyLoadWeight = reduce.get();
        }
        //完成拉运总量
        BigDecimal complateCarryWeight = BigDecimal.ZERO;

        java.util.Optional<BigDecimal> reduce1 = childList.stream().map(OrderChild::getUnloadNet).filter(
                Objects::nonNull).reduce(BigDecimal::add);
        if (reduce1.isPresent()) {
            complateCarryWeight =  reduce1.get();
        }
        //亏吨数量总计


        BigDecimal loseWeight = BigDecimal.ZERO;
        BigDecimal overWeight = BigDecimal.ZERO;
        //卸车总量 大于 装车总量
        if(complateCarryWeight.subtract(historyLoadWeight).compareTo(BigDecimal.ZERO) >= 0){
            overWeight = complateCarryWeight.subtract(historyLoadWeight); //超吨
        }else{
            loseWeight = historyLoadWeight.subtract(complateCarryWeight);//亏吨
        }
        BigDecimal loseRate = BigDecimal.ZERO;
        BigDecimal overRate = BigDecimal.ZERO;

        //亏吨率
        if(historyLoadWeight.compareTo(BigDecimal.ZERO) != 0){
             loseRate = loseWeight.divide(historyLoadWeight,2,RoundingMode.HALF_UP).multiply(new BigDecimal("100"));
             overRate = overWeight.divide(historyLoadWeight,2,RoundingMode.HALF_UP).multiply(new BigDecimal("100"));
        }
        OrderChildBussInfoVO vo = OrderChildBussInfoVO.builder().completeNum(complateNum).historyLoadWeight(historyLoadWeight).completeCarryWeight(complateCarryWeight)
                .loseWeight(loseWeight).loseRate(loseRate).overRate(overRate).build();
        return vo;
    }
}
