package com.clx.performance.service.impl;


import com.clx.performance.dao.*;
import com.clx.performance.enums.*;
import com.clx.performance.model.*;
import com.clx.performance.param.pc.OrderChildCarrierCancelParam;
import com.clx.performance.param.pc.PoundAuditParam;
import com.clx.performance.service.OrderChildLogService;
import com.clx.performance.service.OrderChildPoundAuditService;
import com.clx.performance.vo.pc.OrderChildPoundAuditDetailVO;
import com.msl.common.exception.ServiceSystemException;
import com.msl.user.data.UserSessionData;
import com.msl.user.utils.TokenUtil;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author liruixin
 * Date 2023-09-19
 * Time 13:25
 */
@Slf4j
@Service
@AllArgsConstructor
public class OrderChildPoundAuditServiceImpl  implements OrderChildPoundAuditService {

    private final OrderGoodsTruckBindDao orderGoodsTruckBindDao;
    private final OrderChildImageDao orderChildImageDao;

    private final OrderChildPoundAuditDao orderChildPoundAuditDao;

    private final OrderChildPoundLogDao orderChildPoundLogDao;

    private final OrderChildDao orderChildDao;

    private final OrderGoodsDao orderGoodsDao;

    private final OrderChildLogService orderChildLogService;


    @Override
    public OrderChildPoundAuditDetailVO getPoundAuditDetail(String childNo) {
        OrderChildPoundAudit poundAuditDetail = orderChildPoundAuditDao.getPoundAuditDetail(childNo).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
        List<OrderChildImage> loadImages = orderChildImageDao.getImages(childNo, OrderChildImage.Type.LOAD.getCode());
        List<OrderChildImage> unloadImages = orderChildImageDao.getImages(childNo, OrderChildImage.Type.UNLOAD.getCode());
        OrderChildPoundAuditDetailVO vo = new OrderChildPoundAuditDetailVO();
        vo.setChildNo(childNo);
        vo.setLoadImages(loadImages.stream().map(OrderChildImage::getImage).collect(Collectors.toList()));
        vo.setUnloadImages(unloadImages.stream().map(OrderChildImage::getImage).collect(Collectors.toList()));
        vo.setLoadNet(poundAuditDetail.getLoadNet());
        vo.setUnloadNet(poundAuditDetail.getUnloadNet());
        vo.setUnloadPoundNo(poundAuditDetail.getUnloadPoundNo());
        vo.setStatus(poundAuditDetail.getStatus());
        vo.setRemark(poundAuditDetail.getRemark());
        vo.setRejectType(poundAuditDetail.getRejectType());
        return vo;
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updatePoundAudit(PoundAuditParam param) {

        OrderChild orderChild = orderChildDao.getByChildNo(param.getChildNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);

        OrderChildPoundAudit poundAuditDetail = orderChildPoundAuditDao.getPoundAuditDetail(param.getChildNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
        poundAuditDetail.setStatus(param.getStatus());
        poundAuditDetail.setRejectType(param.getRejectType());
        poundAuditDetail.setRemark(param.getRemark());
        poundAuditDetail.setId(null);

        Integer type = OrderChildLogEnum.Type.POUND_AUDIT.getCode();
        if(Objects.equals(param.getStatus(), OrderChildPoundAuditEnum.Status.REJECT.getCode())){
            poundAuditDetail.setLoadNet(orderChild.getLoadNet());
            poundAuditDetail.setUnloadNet(orderChild.getUnloadNet());
            type = OrderChildLogEnum.Type.POUND_AUDIT_REJECT.getCode();
        }else{
            if(Objects.nonNull(param.getLoadNet())){
                poundAuditDetail.setLoadNet(param.getLoadNet());
            }else{
                poundAuditDetail.setLoadNet(orderChild.getLoadNet());
            }

            if(Objects.nonNull(param.getUnloadNet())){
                poundAuditDetail.setUnloadNet(param.getUnloadNet());
            }else{
                poundAuditDetail.setUnloadNet(orderChild.getUnloadNet());
            }
        }


        orderChildPoundAuditDao.saveEntity(poundAuditDetail);

        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        OrderChildPoundLog poundLog = OrderChildPoundLog.builder().childNo(param.getChildNo()).status(param.getStatus()).createType(OrderChildLogEnum.CreateType.PLATFORM.getCode())
                .remark(OrderChildPoundAuditEnum.Status.getByCode(param.getStatus()).get().getMsg()).createBy(loginUserInfo.getUserNo()).createName(loginUserInfo.getUserName()).build();

        orderChildPoundLogDao.saveEntity(poundLog);

        if(Objects.nonNull(param.getLoadNet())){
            orderChild.setLoadNet(param.getLoadNet());
        }
        if(Objects.nonNull(param.getUnloadNet())){
            orderChild.setUnloadNet(param.getUnloadNet());
        }
        orderChild.setPoundStatus(param.getStatus());
        orderChildDao.updatePoundAuditStatus(orderChild);

        orderChildLogService.saveOrderChildLog(param.getChildNo(),type,OrderChildLogEnum.Type.getByCode(type).get().getMsg(),
                OrderChildLogEnum.CreateType.PLATFORM.getCode(), loginUserInfo.getUserNo(),loginUserInfo.getUserName());

    }

    @Override
    public void updateCarrierCancel(OrderChildCarrierCancelParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();
        LocalDateTime now = LocalDateTime.now();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);

        if (OrderChildEnum.CANCEL_lIST.contains(orderChild.getStatus())){
            return;
        }

        // 禁止取消 (卸车前取消)
        if (orderChild.getStatus()>OrderChildEnum.Status.UNLOAD.getCode()) {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCEL_FORBID);
        }

        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);

        // 定向
        OrderGoodsTruckBind orderGoodsTruckBind = null;
        if (Objects.equals(orderGoods.getPendingOrderWay(), OrderGoodsPendingOrderWayStatusEnum.Status.EXCLUSIVE.getCode())){
            orderGoodsTruckBind = orderGoodsTruckBindDao.getByOrderGoodsNoAndTruckNo(orderGoods.getOrderGoodsNo(), orderChild.getTruckNo())
                    .orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
        }

        orderChild.setCancelRemark(param.getRemark());
        orderChild.setCancelTime(now);
        orderChild.setFinishTime(now);
        orderChild.setStatus(OrderChildEnum.Status.PLATFORM_CANCEL.getCode());

        orderChildDao.updateCancel(orderChild);

        // 返回吨数
        updateOrderGoodsAmountReturn(orderChild, orderGoods);

        // 取消定向
        updateOrderGoodsDirectCancel(orderGoodsTruckBind);

        // 释放司机、车辆
        releaseDriver(orderChild.getDriverUserNo());
        releaseTruck(orderChild.getTruckId());

        // 日志
        orderChildLogService.saveCarrierOrderChildLog(childNo, OrderChildLogEnum.Type.PLATFORM_CANCEL.getCode(), OrderChildLogEnum.Type.PLATFORM_CANCEL.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    /**
     * 取消返吨数
     */
    private void updateOrderGoodsAmountReturn(OrderChild orderChild, OrderGoods orderGoods){
        orderGoodsDao.updateOrderGoodsReduceWeightAndStatus(orderGoods.getId(), orderChild.getWeight().negate());
    }

    /**
     * 更新定向派单状态 (取消)
     */
    private void updateOrderGoodsDirectCancel(OrderGoodsTruckBind orderGoodsTruckBind){
        if (orderGoodsTruckBind == null) {return;}

        orderGoodsTruckBind.setStatus(OrderGoodsTruckBind.Status.CANCEL.getCode());
        orderGoodsTruckBindDao.updateStatus(orderGoodsTruckBind);
    }

    /**
     * 释放司机
     */
    private void releaseDriver(Long driverNo){

    }

    /**
     * 释放车辆
     */
    private void releaseTruck(Integer truckId){

    }

}
