package com.clx.performance.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.clx.performance.dao.OrderChildDao;
import com.clx.performance.dao.OrderChildImageDao;
import com.clx.performance.dao.OrderGoodsDao;
import com.clx.performance.enums.OrderChildEnum;
import com.clx.performance.enums.OrderChildLogEnum;
import com.clx.performance.model.OrderChild;
import com.clx.performance.model.OrderChildImage;
import com.clx.performance.model.OrderGoods;
import com.clx.performance.param.app.*;
import com.clx.performance.param.pc.PagePoundAuditParam;
import com.clx.performance.service.OrderChildLogService;
import com.clx.performance.service.OrderChildService;
import com.clx.performance.vo.pc.OrderChildPoundAuditVO;
import com.msl.common.enums.ResultCodeEnum;
import com.msl.common.exception.ServiceSystemException;
import com.msl.common.utils.DateUtils;
import com.msl.user.data.UserSessionData;
import com.msl.user.utils.TokenUtil;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @Author: aiqinguo
 * @Description: 运单表
 * @Date: 2023/09/18 11:34:50
 * @Version: 1.0
 */

@Slf4j
@Service
@AllArgsConstructor
public class OrderChildServiceImpl implements OrderChildService {

    private final OrderGoodsDao orderGoodsDao;

    private final OrderChildDao orderChildDao;

    private final OrderChildImageDao orderChildImageDao;

    private final OrderChildLogService orderChildLogService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public SaveOrderChildVo saveOrderChild(OrderChildSaveParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();

        Long userNo = loginUserInfo.getUserNo();

        String childNo = "1";

        BigDecimal truckLoad = BigDecimal.valueOf(1L);

        LocalDateTime now = LocalDateTime.now();
        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(param.getOrderGoodsNo()).orElseThrow(ResultCodeEnum.FAIL);
        // 状态验证
        if (!Objects.equals(orderGoods.getOrderGoodsStatus(), 1)
                && !Objects.equals(orderGoods.getOrderGoodsStatus(), 1)){
            throw new ServiceSystemException(ResultCodeEnum.FAIL);
        }

        // 库存验证
        if (orderGoods.getResidueTransportWeight().compareTo(BigDecimal.ZERO) <= 0) {throw new ServiceSystemException(ResultCodeEnum.FAIL);}
        if (orderGoods.getResidueTransportWeight().compareTo(truckLoad) < 0) {throw new ServiceSystemException(ResultCodeEnum.FAIL);}

        OrderChild orderChild = new OrderChild();
        orderChild.setChildNo(childNo);
        orderChild.setUserNo(userNo);
        orderChild.setOwnerUserNo(orderGoods.getUserNo());
        orderChild.setOrderNo(orderGoods.getOrderNo());
        orderChild.setOrderGoodsNo(orderGoods.getOrderGoodsNo());
        orderChild.setFreightPrice(orderGoods.getPendingOrderFreight());

        orderChild.setGoodsId(0);
        orderChild.setGoodsName(orderGoods.getGoodsName());

        orderChild.setSendAddressId(orderGoods.getSendAddressId());
        orderChild.setSendAddress(orderGoods.getSendAddressShorter());
        orderChild.setReceiveAddressId(orderGoods.getReceiveAddressId());
        orderChild.setReceiveAddress(orderGoods.getReceiveAddressShorter());

        orderChild.setDriverUserNo(param.getDriverUserNo());
        orderChild.setDriverName("");
        orderChild.setDriverMobile("");

        orderChild.setTruckId(param.getTruckId());
        orderChild.setTruckNo("");
        orderChild.setTruckLoad(truckLoad);

        orderChild.setWeight(orderChild.getTruckLoad());
        orderChild.setPayTime(now);
        orderChild.setStatus(OrderChildEnum.Status.CREATED.getCode());
        orderChild.setCreateTime(now);

        // 更新货单数据
        updateOrderGoodsAmount(orderGoods, truckLoad);

        // 新增运单
        orderChildDao.saveEntity(orderChild);

        // 更新司机状态

        // 更新车辆状态

        // 新增日志
        orderChildLogService.saveOrderChildLog(childNo, OrderChildLogEnum.Type.CREATED.getCode(), OrderChildLogEnum.Type.CREATED.getMsg(),
                1, loginUserInfo.getUserNo(), loginUserInfo.getUserName());

        SaveOrderChildVo result = new SaveOrderChildVo();
        result.setChildNo(childNo);
        return result;
    }

    @Override
    public void updateGotoSendAddress(OrderChildGoToSendAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(ResultCodeEnum.FAIL);
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_RECEIVE.getCode())){
            throw new ServiceSystemException(ResultCodeEnum.FAIL);
        }

        orderChild.setStatus(OrderChildEnum.Status.GO_TO_SEND.getCode());
        orderChildDao.updateLoad(orderChild);

        // 日志
        orderChildLogService.saveOrderChildLog(childNo, OrderChildLogEnum.Type.ARRIVE_RECEIVE.getCode(), OrderChildLogEnum.Type.ARRIVE_RECEIVE.getMsg(),
                1, loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    public void updateArriveSendAddress(OrderChildArriveSendAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(ResultCodeEnum.FAIL);
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_RECEIVE.getCode())){
            throw new ServiceSystemException(ResultCodeEnum.FAIL);
        }

        orderChild.setStatus(OrderChildEnum.Status.ARRIVE_RECEIVE.getCode());
        orderChildDao.updateLoad(orderChild);

        // 日志
        orderChildLogService.saveOrderChildLog(childNo, OrderChildLogEnum.Type.ARRIVE_RECEIVE.getCode(), OrderChildLogEnum.Type.ARRIVE_RECEIVE.getMsg(),
                1, loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    public void updateGotoReceiveAddress(OrderChildGoToReceiveAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(ResultCodeEnum.FAIL);
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_RECEIVE.getCode())){
            throw new ServiceSystemException(ResultCodeEnum.FAIL);
        }

        orderChild.setStatus(OrderChildEnum.Status.GO_TO_RECEIVE.getCode());
        orderChildDao.updateLoad(orderChild);

        // 日志
        orderChildLogService.saveOrderChildLog(childNo, OrderChildLogEnum.Type.ARRIVE_RECEIVE.getCode(), OrderChildLogEnum.Type.ARRIVE_RECEIVE.getMsg(),
                1, loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    public void updateArriveReceiveAddress(OrderChildArriveReceiveAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(ResultCodeEnum.FAIL);
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_RECEIVE.getCode())){
            throw new ServiceSystemException(ResultCodeEnum.FAIL);
        }

        orderChild.setStatus(OrderChildEnum.Status.ARRIVE_RECEIVE.getCode());
        orderChildDao.updateLoad(orderChild);

        // 日志
        orderChildLogService.saveOrderChildLog(childNo, OrderChildLogEnum.Type.ARRIVE_RECEIVE.getCode(), OrderChildLogEnum.Type.ARRIVE_RECEIVE.getMsg(),
                1, loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateLoad(OrderChildLoadParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(ResultCodeEnum.FAIL);
        if (orderChild.getLoadTime() == null) {
            updateLoadFirst(param, orderChild);
            // 日志
            orderChildLogService.saveOrderChildLog(childNo, OrderChildLogEnum.Type.LOAD.getCode(), OrderChildLogEnum.Type.LOAD.getMsg(),
                    1, loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        } else {
            updateReload(param, orderChild);
            // 日志
            orderChildLogService.saveOrderChildLog(childNo, OrderChildLogEnum.Type.RELOAD.getCode(), OrderChildLogEnum.Type.RELOAD.getMsg(),
                    1, loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        }
    }

    private void updateLoadFirst(OrderChildLoadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getLoadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.LOAD.getCode());
            image.setImage(item);
        }

        orderChild.setLoadRough(param.getLoadRough());
        orderChild.setLoadTare(param.getLoadTare());
        orderChild.setLoadNet(param.getLoadNet());
        orderChild.setLoadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));
        orderChildDao.updateLoad(orderChild);

        orderChildImageDao.batchSaveEntity(imageList);
    }

    private void updateReload(OrderChildLoadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getLoadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.LOAD.getCode());
            image.setImage(item);
        }

        orderChild.setLoadRough(param.getLoadRough());
        orderChild.setLoadTare(param.getLoadTare());
        orderChild.setLoadNet(param.getLoadNet());
        orderChild.setLoadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));
        orderChildDao.updateLoad(orderChild);

        orderChildImageDao.deleteLoad(childNo);
        orderChildImageDao.batchSaveEntity(imageList);

    }

    @Override
    public void updateUnload(OrderChildUnloadParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(ResultCodeEnum.FAIL);
        if (orderChild.getUnloadTime() == null) {
            updateUnloadFirst(param, orderChild);

            // 日志
            orderChildLogService.saveOrderChildLog(childNo, OrderChildLogEnum.Type.UNLOAD.getCode(), OrderChildLogEnum.Type.UNLOAD.getMsg(),
                    1, loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        } else {
            updateReUnload(param, orderChild);
            // 日志
            orderChildLogService.saveOrderChildLog(childNo, OrderChildLogEnum.Type.REUNLOAD.getCode(), OrderChildLogEnum.Type.REUNLOAD.getMsg(),
                    1, loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        }
    }

    private void updateUnloadFirst(OrderChildUnloadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getUnloadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.UNLOAD.getCode());
            image.setImage(item);
        }

        orderChild.setUnloadRough(param.getUnloadRough());
        orderChild.setUnloadTare(param.getUnloadTare());
        orderChild.setUnloadNet(param.getUnloadNet());
        orderChild.setUnloadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));

        orderChild.setStatus(OrderChildEnum.Status.DRIVER_CONFIRM.getCode());
        orderChildDao.updateUnload(orderChild);

        orderChildImageDao.batchSaveEntity(imageList);

    }

    private void updateReUnload(OrderChildUnloadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getUnloadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.UNLOAD.getCode());
            image.setImage(item);
        }

        orderChild.setUnloadRough(param.getUnloadRough());
        orderChild.setUnloadTare(param.getUnloadTare());
        orderChild.setUnloadNet(param.getUnloadNet());
        orderChild.setUnloadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));

        orderChildDao.updateUnload(orderChild);

        orderChildImageDao.deleteUnload(childNo);
        orderChildImageDao.batchSaveEntity(imageList);

    }

    @Override
    public void updateLoadAndUnloadAgain(OrderChildLoadAndUnloadAgainParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();
        String childNo = param.getChildNo();
        LocalDateTime now = LocalDateTime.now();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(ResultCodeEnum.FAIL);
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.POUND_AUDIT_FAIL.getCode())){
            throw new ServiceSystemException(ResultCodeEnum.FAIL);
        }

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getLoadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.LOAD.getCode());
            image.setImage(item);
        }
        for (String item : param.getUnloadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.UNLOAD.getCode());
            image.setImage(item);
        }

        orderChild.setLoadRough(param.getLoadRough());
        orderChild.setLoadTare(param.getLoadTare());
        orderChild.setLoadNet(param.getLoadNet());
//        orderChild.setLoadTime(now);
        orderChild.setUnloadRough(param.getUnloadRough());
        orderChild.setUnloadTare(param.getUnloadTare());
        orderChild.setUnloadNet(param.getUnloadNet());
//        orderChild.setUnloadTime(now);
        orderChild.setWeight(orderChildWeightCalc(orderChild));

        orderChild.setStatus(OrderChildEnum.Status.DRIVER_CONFIRM.getCode());
        orderChildDao.updateLoadAndUnload(orderChild);

        orderChildImageDao.deleteLoadAndUnload(childNo);
        orderChildImageDao.batchSaveEntity(imageList);

        // 日志
        orderChildLogService.saveOrderChildLog(childNo, OrderChildLogEnum.Type.RELOAD_AND_REUNLOAD.getCode(), OrderChildLogEnum.Type.RELOAD_AND_REUNLOAD.getMsg(),
                1, loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    public void updateConfirm(OrderChildConfirmParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(ResultCodeEnum.FAIL);
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CONFIRM.getCode())){
            throw new ServiceSystemException(ResultCodeEnum.FAIL);
        }

        orderChild.setStatus(OrderChildEnum.Status.COMPLETE.getCode());
        orderChildDao.updateDriverConfirm(orderChild);

        // 日志
        orderChildLogService.saveOrderChildLog(childNo, OrderChildLogEnum.Type.DRIVER_CONFIRM.getCode(), OrderChildLogEnum.Type.DRIVER_CONFIRM.getMsg(),
                1, loginUserInfo.getUserNo(), loginUserInfo.getUserName());

    }

    @Override
    public void updateCancel(OrderChildCancelParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(ResultCodeEnum.FAIL);
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.CREATED.getCode())
                ||Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_SEND.getCode())
                ||Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_SEND.getCode())
        ){}
        else {
            throw new ServiceSystemException(ResultCodeEnum.FAIL);
        }

        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(ResultCodeEnum.FAIL);

        // 取消量验证
        if (!cancelCountCheck(userNo)){throw new ServiceSystemException(ResultCodeEnum.FAIL);}

        orderChild.setStatus(OrderChildEnum.Status.DRIVER_CANCEL.getCode());

        // 取消
        orderChildDao.updateCancel(orderChild);

        // 返回吨数
        cancelReturnWeight(orderChild, orderGoods);

        // 日志
        orderChildLogService.saveOrderChildLog(childNo, OrderChildLogEnum.Type.DRIVER_CANCEL.getCode(), OrderChildLogEnum.Type.DRIVER_CANCEL.getMsg(),
                1, loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    /**
     * 更新货单数据
     */
    private void updateOrderGoodsAmount(OrderGoods orderGoods, BigDecimal weight){
        if (orderGoods.getResidueTransportWeight().compareTo(weight) <= 0) {orderGoods.setOrderGoodsStatus(0);}
//        orderGoodsDao.updateEntityByKey(orderGoods, weight);
    }

    /**
     * 取消检测
     */
    private boolean cancelCountCheck(Long userNo){
        LocalDateTime startTime = DateUtils.parseDateTime(DateUtils.formatDateTime(LocalDateTime.now(), "yyyy-MM-dd").get() + " 00:00:00").get();
        LocalDateTime endTime = DateUtils.parseDateTime(DateUtils.formatDateTime(LocalDateTime.now(), "yyyy-MM-dd").get() + " 00:00:00").get();

        long count = orderChildDao.countOfCancel(userNo, startTime, endTime);

        return count<10;
    }

    /**
     * 取消返吨数
     */
    private void cancelReturnWeight(OrderChild orderChild, OrderGoods orderGoods){

    }

    /**
     * 运单拉运吨数计算
     */
    private BigDecimal orderChildWeightCalc(OrderChild orderChild){

        return orderChild.getLoadNet();
    }

    @Override
    public IPage<OrderChildPoundAuditVO> pagePoundAuditList(PagePoundAuditParam param) {
        return orderChildDao.pagePoundAuditList(param);
    }
}
