package com.clx.performance.component;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.json.JSONUtil;
import com.clx.order.enums.OrderEnum;
import com.clx.order.feign.OrderFeign;
import com.clx.order.param.feign.RollbackOrderWeightParam;
import com.clx.order.vo.feign.FeignOrderVO;
import com.clx.performance.dao.OrderChildDao;
import com.clx.performance.dao.OrderGoodsDao;
import com.clx.performance.enums.*;
import com.clx.performance.extranal.user.DriverService;
import com.clx.performance.model.OrderChild;
import com.clx.performance.model.OrderGoods;
import com.clx.performance.param.pc.OrderCancelParam;
import com.clx.performance.service.OrderChildService;
import com.clx.performance.service.OrderGoodsTruckBindService;
import com.clx.performance.service.breakcontract.BreakContractSettlementOwnerService;
import com.clx.user.enums.driver.DriverInfoEnum;
import com.msl.common.exception.ServiceSystemException;
import com.msl.common.result.Result;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Component
@Slf4j
@AllArgsConstructor
public class OrderCancelComponent implements InitializingBean {

    private final OrderChildDao orderChildDao;

    private final OrderGoodsDao orderGoodsDao;

    private final OrderGoodsTruckBindService orderGoodsTruckBindService;

    private final GoodsOrderTruckRecordComponent goodsOrderTruckRecordComponent;

    private final OrderFeign orderFeign;

    private final DriverService driverService;

    private final BreakContractSettlementOwnerService breakContractSettlementOwnerService;

    public static Map<Integer,String> statusCodeMap = new HashMap<>();

    public void ownerProcess(OrderCancelParam param){
        String orderNo = param.getOrderNo();
        List<OrderGoods> orderGoodsList = null;
        Integer orderStatusCode = OrderEnum.Status.CANCELED.getCode();

        switch (param.getCancelType()) {
            case 1: //单纯取消订单
                log.info("1:货主单纯取消订单");
                if (CollectionUtil.isNotEmpty(orderChildDao.getOrderChildByOrderNo(orderNo))) {
                    throw new ServiceSystemException(PerformanceResultEnum.HTTP_ERROR, "当前订单已发生变化,请重新选择取消方式");
                }

                orderGoodsList = orderGoodsDao.getOrderGoodsListByOrderNo(orderNo);
                if (CollectionUtil.isNotEmpty(orderGoodsList)) {
                    List<Integer> ids = new LinkedList<>();
                    for (OrderGoods orderGoods : orderGoodsList) {
                        if (orderGoods.getPendingOrderWay().equals(PendingOrderWayStatusEnum.Status.DX.getCode())) {
                            orderGoodsTruckBindService.updateCancelTruckBind(orderGoods.getOrderGoodsNo());
                        }
                        goodsOrderTruckRecordComponent.deleteTruckRecord(orderGoods.getOrderGoodsNo());

                        ids.add(orderGoods.getId());
                    }
                    orderGoodsDao.batchUpdateOrderGoodsStatus(OrderGoodsStatusEnum.Status.CANCEL.getCode(),
                            ids);
                }
                break;
            case 2: //取消未拉运吨数的货单
                log.info("2:货主取消未拉运吨数的货单");

                orderGoodsList = orderGoodsDao.getOrderGoodsListByOrderNo(orderNo);

                for (OrderGoods orderGoods : orderGoodsList) {
                    OrderGoods update = new OrderGoods();
                    update.setId(orderGoods.getId());
                    if (orderGoods.getResidueTransportWeight().compareTo(orderGoods.getExtractWeight()) == 0) {
                        update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.CANCEL.getCode());
                    } else {
                        orderStatusCode = OrderEnum.Status.COMPLETED.getCode();
                        update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.COMPLETED.getCode());
                    }
                    if (orderGoods.getPendingOrderWay().equals(PendingOrderWayStatusEnum.Status.DX.getCode())) {
                        orderGoodsTruckBindService.updateCancelTruckBind(orderGoods.getOrderGoodsNo());
                    }
                    goodsOrderTruckRecordComponent.deleteTruckRecord(orderGoods.getOrderGoodsNo());

                    orderGoodsDao.updateEntityByKey(update);
                }

                break;

            case 3: //取消未拉运+未到达货源地吨数
                log.info("3:货主取消未拉运+未到达货源地吨数");

                List<OrderChild> orderChildren = orderChildDao.selectResidueWeight(orderNo, OrderChildEnum.Status.ARRIVE_SEND.getCode());
                log.info("执行货主取消未拉运+未到达货源地吨数,当前运单个数"+ orderChildren.size());
                List<Integer> ids = new LinkedList<>();
                for (OrderChild orderChild : orderChildren) {
                    log.info("取消的运单：{}", JSONUtil.parse(orderChild));
                    //释放车辆
                    driverService.updateOrderStatus(orderChild.getDriverUserNo(), DriverInfoEnum.DriverStatus.NO.getCode(), orderChild.getTruckId(), 1);
                    ids.add(orderChild.getId());
                }

                Map<String, List<OrderChild>> orderChildMap = new HashMap<>();
                if (CollectionUtil.isNotEmpty(ids)) {
                    log.info("ids:{}", ids);
                    orderChildDao.batchUpdateOrderChildStatus(
                            OrderChildEnum.Status.OWNER_CANCEL.getCode(), "货主取消",
                            ids);
                    orderChildMap = orderChildren.stream()
                            .collect(Collectors.groupingBy(OrderChild::getOrderGoodsNo));

                }
                orderGoodsList = orderGoodsDao.getOrderGoodsListByOrderNo(orderNo);

                for (OrderGoods orderGoods : orderGoodsList) {
                    OrderGoods update = new OrderGoods();
                    update.setId(orderGoods.getId());
                    if (CollectionUtil.isEmpty(orderChildMap.get(orderGoods.getOrderGoodsNo()))) {
                        update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.CANCEL.getCode());
                    } else {
                        BigDecimal reduce = BigDecimal.ZERO;
                        for (OrderChild orderChild : orderChildMap.get(orderGoods.getOrderGoodsNo())) {
                            reduce = reduce.add(orderChild.getWeight());
                        }

                        if (orderGoods.getExtractWeight().compareTo(orderGoods.getResidueTransportWeight().add(reduce)) == 0) {
                            update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.CANCEL.getCode());
                        } else {
                            orderStatusCode = OrderEnum.Status.COMPLETED.getCode();
                            update.setOrderGoodsStatus(OrderGoodsStatusEnum.Status.COMPLETED.getCode());
                        }
                    }
                    if (orderGoods.getPendingOrderWay().equals(PendingOrderWayStatusEnum.Status.DX.getCode())) {
                        orderGoodsTruckBindService.updateCancelTruckBind(orderGoods.getOrderGoodsNo());
                    }
                    goodsOrderTruckRecordComponent.deleteTruckRecord(orderGoods.getOrderGoodsNo());

                    orderGoodsDao.updateEntityByKey(update);
                }

                break;

            default:
                throw new ServiceSystemException(PerformanceResultEnum.HTTP_ERROR, "参数有误");

        }

        this.breakContractSettlementOwner(orderNo, orderStatusCode);

        log.info("执行货主取消订单更改状态");
        if (!orderFeign.updateOrderStatusByOrderNo(orderNo, orderStatusCode).succeed()) {
            throw new ServiceSystemException(PerformanceResultEnum.HTTP_ERROR, "货主取消订单失败,原因：远程调用取消订单失败");
        }


    }

    public void backFillTonnage(OrderGoods orderGoods, BigDecimal weight) {
        if (orderGoods.getOrderGoodsStatus().equals(OrderGoodsStatusEnum.Status.CANCEL.getCode())
                || orderGoods.getOrderGoodsStatus().equals(OrderGoodsStatusEnum.Status.COMPLETED.getCode())
        ) {
            Integer truckDemand = orderGoods.getTruckDemand();
            //返回到货单
            orderGoodsDao.updateOrderGoodsSetResidueWeightAndAlreadyWeight(weight,weight, orderGoods.getId());

            RollbackOrderWeightParam rollbackOrderWeightParam = new RollbackOrderWeightParam();
            rollbackOrderWeightParam.setOrderNo(orderGoods.getOrderNo());
            BigDecimal residueWeight = orderGoods.getResidueTransportWeight().add(weight);
            rollbackOrderWeightParam.setResidueWeight(residueWeight);

            //归还吨数
            if (!truckDemand.equals(TruckDemandEnum.TruckType.PLATFORM_ANY_CAR.getCode())) {
                if (truckDemand.equals(TruckDemandEnum.TruckType.PLATFORM_CAR.getCode())) {
                    rollbackOrderWeightParam.setOwnResidueCarryWeight(BigDecimal.ZERO);
                    rollbackOrderWeightParam.setPlatformResidueCarryWeight(residueWeight);
                }else if (truckDemand.equals(TruckDemandEnum.TruckType.OWNER_CAR.getCode())) {
                    rollbackOrderWeightParam.setOwnResidueCarryWeight(residueWeight);
                    rollbackOrderWeightParam.setPlatformResidueCarryWeight(BigDecimal.ZERO);
                }
            } else {
                rollbackOrderWeightParam.setPlatformResidueCarryWeight(residueWeight);
                rollbackOrderWeightParam.setOwnResidueCarryWeight(residueWeight);
            }

            Result<Object> result = orderFeign.rollbackOrderWeight(rollbackOrderWeightParam);
            if (!result.succeed()) {
                throw new ServiceSystemException(PerformanceResultEnum.HTTP_ERROR, "远程调用更新订单余量失败");
            }
        }
    }

    public void backFillTonnageForArtificial(String orderGoodsNo, BigDecimal weight) {
        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderGoodsNo).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
        if (orderGoods.getOrderGoodsStatus().equals(OrderGoodsStatusEnum.Status.CANCEL.getCode())
                || orderGoods.getOrderGoodsStatus().equals(OrderGoodsStatusEnum.Status.COMPLETED.getCode())
        ) {
            Integer truckDemand = orderGoods.getTruckDemand();
            //返回到货单
            orderGoodsDao.updateOrderGoodsSetResidueWeightAndAlreadyWeight(weight,weight, orderGoods.getId());

            RollbackOrderWeightParam rollbackOrderWeightParam = new RollbackOrderWeightParam();
            rollbackOrderWeightParam.setOrderNo(orderGoods.getOrderNo());
            BigDecimal residueWeight = orderGoods.getResidueTransportWeight().add(weight);
            rollbackOrderWeightParam.setResidueWeight(residueWeight);

            //归还吨数
            if (!truckDemand.equals(TruckDemandEnum.TruckType.PLATFORM_ANY_CAR.getCode())) {
                if (truckDemand.equals(TruckDemandEnum.TruckType.PLATFORM_CAR.getCode())) {
                    rollbackOrderWeightParam.setOwnResidueCarryWeight(BigDecimal.ZERO);
                    rollbackOrderWeightParam.setPlatformResidueCarryWeight(residueWeight);
                }else if (truckDemand.equals(TruckDemandEnum.TruckType.OWNER_CAR.getCode())) {
                    rollbackOrderWeightParam.setOwnResidueCarryWeight(residueWeight);
                    rollbackOrderWeightParam.setPlatformResidueCarryWeight(BigDecimal.ZERO);
                }
            } else {
                rollbackOrderWeightParam.setPlatformResidueCarryWeight(residueWeight);
                rollbackOrderWeightParam.setOwnResidueCarryWeight(residueWeight);
            }

            Result<Object> result = orderFeign.rollbackOrderWeight(rollbackOrderWeightParam);
            if (!result.succeed()) {
                throw new ServiceSystemException(PerformanceResultEnum.HTTP_ERROR, "远程调用更新订单余量失败");
            }
        } else {
            //返回到货单
            orderGoodsDao.updateOrderGoodsSetResidueWeightAndAlreadyWeight(weight,weight, orderGoods.getId());

        }
    }


    public void breakContractSettlementOwner(String orderNo, Integer updateStatus) {
        FeignOrderVO orderInfoFeign = orderFeign.getOrderInfoFeign(orderNo);
        Integer orderStatus = orderInfoFeign.getOrderStatus();
        log.info("执行处理违约,订单号：{}, updateStatus:{}, 订单状态{}", orderNo, updateStatus, orderStatus);
        if (OrderEnum.Status.COMPLETED.getCode().equals(updateStatus) || statusCodeMap.containsKey(orderStatus)) {
            try {
                breakContractSettlementOwnerService.saveSettlementOwnerForCancelOrder(orderNo);
                log.info("执行处理违约完毕");

            }catch (Exception e) {
                log.info("执行处理违约失败{}", e.getMessage());
            }
        }

    }

    @Override
    public void afterPropertiesSet() throws Exception {
        statusCodeMap.put(OrderEnum.Status.QUOTATION_CONFIRMED.getCode(), OrderEnum.Status.QUOTATION_CONFIRMED.getName());
        statusCodeMap.put(OrderEnum.Status.PLATFORM_UNDERTAKING.getCode(), OrderEnum.Status.PLATFORM_UNDERTAKING.getName());
        statusCodeMap.put(OrderEnum.Status.POST_ORDER.getCode(), OrderEnum.Status.POST_ORDER.getName());
        //statusCodeMap.put(OrderEnum.Status.ON_ORDER.getCode(), OrderEnum.Status.ON_ORDER.getName());
    }
}
