package com.clx.performance.controller.feign;

import com.clx.open.sdk.request.action.GetOrderBreakContractOwnerRuleFileAction;
import com.clx.open.sdk.request.action.QueryPerformanceProgressAction;
import com.clx.open.sdk.request.action.QueryPerformanceProgressLogAction;
import com.clx.open.sdk.request.dto.BatchOrderStatisticsDTO;
import com.clx.open.sdk.request.dto.BatchOrderStatisticsInfoDTO;
import com.clx.open.sdk.request.dto.OrderCanExtractWeightDTO;
import com.clx.open.sdk.request.dto.OwnerCancelResidueWeightDTO;
import com.clx.performance.extranal.user.OrderService;
import com.clx.performance.param.pc.OrderCancelParam;
import com.clx.performance.service.*;
import com.clx.performance.service.breakcontract.BreakContractOwnerRuleService;
import com.clx.performance.vo.feign.OrderExtractWeightVO;
import com.clx.performance.vo.pc.OwnerAccountAllVO;
import com.clx.performance.vo.pc.PerformanceProgressOperationLogVO;
import com.clx.performance.vo.pc.PerformanceProgressVO;
import com.clx.performance.vo.pc.breakcontract.carrier.BreakContractOwnerRuleFileVO;
import com.msl.common.result.Result;
import com.msl.user.utils.TokenUtil;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Operation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import jakarta.validation.constraints.NotBlank;
import jakarta.validation.constraints.NotEmpty;
import java.math.BigDecimal;
import java.util.List;

import static com.clx.performance.enums.PerformanceProgressEnum.LogType.PERFORMANCE_ABNORMAL_REASON;

@Slf4j
@RestController
@RequestMapping("/feign/sdk")
@Validated
@Tag(name = "对外sdk Feign")
@AllArgsConstructor
public class PerformanceSdkFeignController {


    private final OrderCancelService orderCancelService;

    private final OrderGoodsService orderGoodsService;

    private final OwnerAccountService ownerAccountService;

    private final BreakContractOwnerRuleService breakContractOwnerRuleService;

    private final OrderWeightService orderWeightService;



    private final PerformanceProgressService performanceProgressService;

    private final OrderService orderService;


    @Operation(summary = "货主端取消订单", description = "<br>By:胡宇帆")
    @PostMapping("/ownCancelOrderPre")
    public Result<Integer> ownCancelOrderPre(@RequestBody OrderCancelParam param) {
        log.info("取消订单参数：{}", param);
        return Result.ok(orderCancelService.ownerCancelOrderPre(param));
    }

    @Operation(summary = "获取当前订单可取消吨数", description = "<br>By:胡宇帆")
    @GetMapping("/orderCancelResidueWeight")
    public Result<OwnerCancelResidueWeightDTO> orderCancelResidueWeight(@RequestParam(value = "orderNo") String orderNo, @RequestParam(value = "optionType") String optionType) {
        log.info("当前订单可取消吨数参数：{}", TokenUtil.getLoginUserInfo());
        OwnerCancelResidueWeightDTO dto = new OwnerCancelResidueWeightDTO();
        dto.setWeight(orderGoodsService.orderCancelResidueWeight(orderNo, optionType).toString());
        return Result.ok(dto);
    }

    @Operation(summary = "获取当前用户所有账户信息", description = "<br>By:胡宇帆")
    @GetMapping("/accountAllInfo")
    public Result<OwnerAccountAllVO> accountAllInfo(@RequestParam Long userNo) {
        return Result.ok(ownerAccountService.accountAllInfo(userNo));
    }

    @Operation(summary = "查询规则文档信息", description = "<br>By:艾庆国")
    @PostMapping("/getOwnerRuleFile")
    Result<BreakContractOwnerRuleFileVO> getOwnerRuleFile(@RequestBody GetOrderBreakContractOwnerRuleFileAction action){
        return Result.ok(breakContractOwnerRuleService.getRuleFile(action.getId()));
    }

    @Operation(summary = "查询履约进度", description = "<br>By:杨启发")
    @GetMapping("/queryPerformanceProgress")
    Result<List<PerformanceProgressVO>> queryPerformanceProgress(@RequestBody QueryPerformanceProgressAction action){
        return Result.ok(performanceProgressService.queryPerformanceProgress(action.getOrderNoList()));
    }

    @Operation(summary = "查询履约进度操作日志", description = "<br>By:杨启发")
    @GetMapping("/queryPerformanceProgressLog")
    Result<List<PerformanceProgressOperationLogVO>> queryPerformanceProgressLog(@RequestBody QueryPerformanceProgressLogAction action){
        return Result.ok(performanceProgressService.getOperationLog(action.getOrderNo(),PERFORMANCE_ABNORMAL_REASON.getCode()));
    }
    @Operation(summary = "获取当前订单的可提取吨数", description = "<br>By:刘海泉")
    @GetMapping("/getOrderCanExtractWeight")
    public Result<OrderCanExtractWeightDTO> getOrderCanExtractWeight(@RequestParam(value = "orderNo")
    @NotBlank(message = "订单编号不能为空") String orderNo) {
        OrderExtractWeightVO orderCanExtractWeight = orderWeightService.getOrderCanExtractWeight(orderNo);
        OrderCanExtractWeightDTO data  = new OrderCanExtractWeightDTO();
        BeanUtils.copyProperties(orderCanExtractWeight,data);
        return Result.ok(data);
    }




    @Operation(summary = "查询订单相关统计信息", description = "<br>By:刘海泉")
    @GetMapping("/queryTaskStatisticsInfo")
    Result<BatchOrderStatisticsInfoDTO> queryTaskStatisticsInfo(@RequestParam("orderNoList")
    @NotEmpty(message = "订单编号集合不能为空") List<String> orderNoList){
        return Result.ok(orderService.queryTaskStatisticsInfo(orderNoList));
    }



    @Operation(summary = "获取当前订单的剩余吨数", description = "<br>By:刘海泉")
    @GetMapping("/getOrderResidueWeight")
    public Result<BigDecimal> getOrderResidueWeight(@RequestParam(value = "orderNo")
    @NotBlank(message = "订单编号不能为空") String orderNo) {
        BigDecimal residueWeight = orderWeightService.getOrderResidueWeight(orderNo);
        return Result.ok(residueWeight);
    }

    @Operation(summary = "查询订单已拉运量",description = "<br>By:刘海泉")
    @GetMapping("/getAlreadyWeight")
    public Result<BigDecimal> getAlreadyWeight(@RequestParam("orderNo")
    @NotBlank(message = "订单编号不能为空") String orderNo) {
        return Result.ok(orderWeightService.getAlreadyWeight(orderNo));
    }

    @Operation(summary = "查询订单相关统计", description = "<br>By:刘海泉")
    @GetMapping("/queryOrderStatistics")
    Result<BatchOrderStatisticsDTO> queryOrderStatistics(@RequestParam("orderNoList")
    @NotEmpty(message = "订单编号集合不能为空") List<String> orderNoList){
        return Result.ok(orderService.queryOrderStatistics(orderNoList));
    }


}
