package com.clx.performance.service.impl.vehiclewarn;

import com.clx.performance.dao.vehiclewarn.VehicleWarnConfigDao;
import com.clx.performance.dao.vehiclewarn.VehicleWarnInfoDao;
import com.clx.performance.dto.vehiclewarn.VehicleWarnCommonInfoDTO;
import com.clx.performance.enums.vehiclewarn.VehicleWarnConfigEnum;
import com.clx.performance.enums.vehiclewarn.VehicleWarnInfoEnum;
import com.clx.performance.model.OrderChild;
import com.clx.performance.model.vehiclewarn.VehicleWarnChild;
import com.clx.performance.model.vehiclewarn.VehicleWarnConfig;
import com.clx.performance.model.vehiclewarn.VehicleWarnInfo;
import com.clx.performance.service.vehiclewarn.VehicleWarnCommonService;
import com.clx.performance.service.vehiclewarn.VehicleWarnSendAddressWarnService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;

@Slf4j
@Service
public class VehicleWarnSendAddressWarnServiceImpl implements VehicleWarnSendAddressWarnService {

    @Autowired
    private VehicleWarnConfigDao vehicleWarnConfigDao;
    @Autowired
    private VehicleWarnInfoDao vehicleWarnInfoDao;
    @Autowired
    private VehicleWarnCommonService vehicleWarnCommonService;

    /**
     * 前往货源地超时
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void gotoSendAddressTimeout(OrderChild orderChild, VehicleWarnChild vehicleWarnChild
            , VehicleWarnCommonInfoDTO commonInfo){
        if (vehicleWarnChild.getArriveSendExpectTime() == null) {return;}

        long expectTime = vehicleWarnChild.getArriveSendExpectTime();

        // 计算需要的时间
        Integer needTime = commonInfo.getSendAddressTime();

        List<VehicleWarnConfig> vehicleWarnConfigList = vehicleWarnConfigDao
                .listByWarnType(VehicleWarnConfigEnum.WarnType.GOTO_SEND_TIMEOUT.getCode());
        if (vehicleWarnConfigList.isEmpty()) {return;}

        for (VehicleWarnConfig item : vehicleWarnConfigList) {
            doGotoSendAddressTimeout(orderChild, vehicleWarnChild, item, expectTime, needTime);
        }

    }

    private void doGotoSendAddressTimeout(OrderChild orderChild, VehicleWarnChild vehicleWarnChild
            , VehicleWarnConfig vehicleWarnConfig, long expectTime, Integer needTime){
        VehicleWarnInfo vehicleWarnInfo = vehicleWarnInfoDao
                .findByChildNoAndWarnConfigId(vehicleWarnChild.getChildNo(), vehicleWarnConfig.getId()).orNull();
        if (vehicleWarnInfo != null
                && Objects.equals(vehicleWarnInfo.getStatus(), VehicleWarnInfoEnum.Status.RESOLVE.getCode())){
            return;
        }

        // 暂停检测
        boolean suspend = vehicleWarnCommonService.suspendCheck(vehicleWarnChild, vehicleWarnConfig.getId());
        if (suspend) {return;}

        // 超时判断
        int time = vehicleWarnConfig.getTimeoutRatio()
                .multiply(new BigDecimal(expectTime))
                .divide(new BigDecimal("100"), 0, RoundingMode.HALF_UP).intValue();
        log.info("前往货源地超时， childNo:{}, takeTime:{}, expectTime:{}, time:{}, needTime:{}", orderChild.getChildNo(), orderChild.getPayTime(), expectTime, time, needTime);
        if (orderChild.getPayTime().plusMinutes(expectTime)
                .plusMinutes(time).isAfter(LocalDateTime.now().plusMinutes(needTime))){return;}

        // 更新
        vehicleWarnCommonService.vehicleWarnInfoUpdate(orderChild, vehicleWarnInfo, vehicleWarnConfig);

    }

    /**
     * 到达货源地延误
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void arriveSendAddressTimeout(OrderChild orderChild, VehicleWarnChild vehicleWarnChild
            , VehicleWarnCommonInfoDTO commonInfo){
        if (vehicleWarnChild.getArriveSendExpectTime() == null) {return;}

        // 计算需要的时间
        Integer needTime = commonInfo.getSendAddressTime();
        LocalDateTime lastArriveSendTime = commonInfo.getLastArriveSendTime();

        log.info("到达货源地延误, childNo:{}, lastArriveSendTime:{}, needTime:{}", orderChild.getChildNo(), lastArriveSendTime, needTime);
        // 超时判断
        LocalDateTime now = LocalDateTime.now();
        if (lastArriveSendTime.isAfter(now.plusMinutes(needTime))) {return;}

        List<VehicleWarnConfig> vehicleWarnConfigList = vehicleWarnConfigDao
                .listByWarnType(VehicleWarnConfigEnum.WarnType.ARRIVE_SEND_DELAY.getCode());
        if (vehicleWarnConfigList.isEmpty()) {return;}

        for (VehicleWarnConfig item : vehicleWarnConfigList) {
            doArriveSendAddressTimeout(orderChild, vehicleWarnChild, item);
        }

    }

    private void doArriveSendAddressTimeout(OrderChild orderChild, VehicleWarnChild vehicleWarnChild
            , VehicleWarnConfig vehicleWarnConfig){
        VehicleWarnInfo vehicleWarnInfo = vehicleWarnInfoDao
                .findByChildNoAndWarnConfigId(vehicleWarnChild.getChildNo(), vehicleWarnConfig.getId()).orNull();
        if (vehicleWarnInfo != null
                && Objects.equals(vehicleWarnInfo.getStatus(), VehicleWarnInfoEnum.Status.RESOLVE.getCode())){
            return;
        }

        // 暂停检测
        boolean suspend = vehicleWarnCommonService.suspendCheck(vehicleWarnChild, vehicleWarnConfig.getId());
        if (suspend) {return;}

        // 更新
        vehicleWarnCommonService.vehicleWarnInfoUpdate(orderChild, vehicleWarnInfo, vehicleWarnConfig);

    }

}
