package com.clx.performance.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.clx.order.enums.ResultEnum;
import com.clx.order.feign.OrderFeign;
import com.clx.performance.dao.OrderGoodsAdjustmentPriceDao;
import com.clx.performance.extranal.user.OrderService;
import com.clx.performance.model.OrderGoodsAdjustmentPrice;
import com.clx.performance.service.OrderGoodsAdjustmentPriceService;
import com.clx.performance.vo.pc.DriverFreightPriceChartVO;
import com.msl.common.base.Optional;
import com.msl.common.exception.ServiceSystemException;
import com.msl.common.utils.DateUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * @author kavin
 * Date 2024-07-18
 * Time 17:04
 */
@Service
@Slf4j
@AllArgsConstructor
public class OrderGoodsAdjustmentPriceServiceImpl  implements OrderGoodsAdjustmentPriceService {

    private final OrderGoodsAdjustmentPriceDao orderGoodsAdjustmentPriceDao;
    private final OrderFeign orderFeign;
    private final OrderService orderService;


    @Transactional(rollbackFor = Exception.class)
    @Override
    public void saveOrderGoodsAdjustmentPrice(OrderGoodsAdjustmentPrice item){
        orderGoodsAdjustmentPriceDao.saveEntity(item);
    }





    @Override
    public List<DriverFreightPriceChartVO> getDriverFreightPriceChart(String orderNo) {
        //查询近一个月与该订单收发货地线路一致的订单

        Optional<List<String>> result = orderService.getMonthSameLineOrderList(orderNo);
        if(!result.isPresent()){
            log.warn("通过订单号：{} 查找相同线路的订单列列表，返回结果：{}",orderNo,result);
            throw new ServiceSystemException(ResultEnum.DATA_ERROR,"获取相同线路的订单列表失败");
        }
        //相邻位置相同则取同一条
        return this.getAdjacentPosDuplicateRemoveAdjustmentPriceRecord(result.get());
    }


    @Override
    public List<DriverFreightPriceChartVO> getAdjacentPosDuplicateRemoveAdjustmentPriceRecord(List<String> orderNoList){
        List<OrderGoodsAdjustmentPrice> list  = orderGoodsAdjustmentPriceDao.selectByOrderNoList(orderNoList);
        List<DriverFreightPriceChartVO> chartResult = new ArrayList<>();
        //相邻的多条记录调价相同，则保留一条，不相同则生成新的一条
        if(CollectionUtil.isNotEmpty(list)){
            DriverFreightPriceChartVO chart;
            BigDecimal preAdjustmentPrice = BigDecimal.ZERO;
            for (OrderGoodsAdjustmentPrice item : list) {
                //上次的价格和本次的价格发生了变化
                if(preAdjustmentPrice.compareTo(item.getAmount()) != 0){
                    chart = new DriverFreightPriceChartVO();
                    chart.setPendingOrderFreight(item.getAmount());
                    chart.setCreatTime(DateUtils.formatDateTime(item.getCreateTime()).get());
                    chartResult.add(chart);
                    preAdjustmentPrice = item.getAmount();
                }
            }
        }
        return chartResult;
    }

    @Override
    public List<BigDecimal> getNoRepeatAdjustmentPriceRecord(List<String> orderNoList){
        List<OrderGoodsAdjustmentPrice> list  = orderGoodsAdjustmentPriceDao.selectByOrderNoList(orderNoList);
        List<BigDecimal> chartResult = new ArrayList<>();
        //相邻的多条记录调价相同，则保留一条，不相同则生成新的一条
        if(CollectionUtil.isNotEmpty(list)){
            for (OrderGoodsAdjustmentPrice item : list) {
                //去重操作
                if(!chartResult.contains(item.getAmount())){
                    chartResult.add(item.getAmount());
                }
            }
        }
        return chartResult;
    }





}
