package com.clx.performance.encryption.oldmsl;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.bouncycastle.util.encoders.Base64;

import javax.crypto.Cipher;
import java.security.*;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.HashMap;
import java.util.Map;

/**
 * RSA非对称加密工具类
 * 
 * @author xujianke
 * @date 2017年4月13日
 * @description
 */
@Slf4j
public class RSACoder {
	public static final String KEY_ALGORITHM = "RSA";
	public static final String SIGNATURE_ALGORITHM = "MD5withRSA";

	private static final String PUBLIC_KEY = "RSAPublicKey";
	private static final String PRIVATE_KEY = "RSAPrivateKey";

	/**
	 * 用字符串私钥对信息生成数字签名
	 * 
	 * @param data 加密数据
	 * @param privateKey 私钥字符串
	 * @return
	 * @throws Exception
	 */
	public static String sign(byte[] data, String privateKey) throws Exception {
		// 解密由base64编码的私钥
		byte[] keyBytes = Base64.decode(privateKey);
		return sign(data,keyBytes);
	}
	/**
	 * 用字节私钥对信息生成数字签名
	 * 
	 * @param data 加密数据
	 * @param keyBytes 私钥字节
	 * @return
	 * @throws Exception
	 */
	public static String sign(byte[] data, byte[] keyBytes){
		String ret = null;
		try {
			// 构造PKCS8EncodedKeySpec对象
			PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(keyBytes);
	
			// KEY_ALGORITHM 指定的加密算法
			KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
	
			// 取私钥匙对象
			PrivateKey priKey = keyFactory.generatePrivate(pkcs8KeySpec);
			
	
			// 用私钥对信息生成数字签名
			Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
			signature.initSign(priKey);
			signature.update(data);
	
			byte[] dataBytes = Base64.encode(signature.sign());
			ret = new String(dataBytes);
		} catch (Exception e) {
			log.error(ExceptionUtils.getStackTrace(e));
		}
		return ret;
	}

	/**
	 * 校验数字签名
	 * 
	 * @param data 加密数据
	 * @param publicKey 公钥
	 * @param sign 数字签名
	 * @return 校验成功返回true 失败返回false
	 * @throws Exception
	 * 
	 */
	public static boolean verify(byte[] data, String publicKey, String sign){

		// 解密由base64编码的公钥
		byte[] keyBytes = Base64.decode(publicKey);
		return verify(data,keyBytes,sign);
	}
	/**
	 * 校验数字签名
	 * 
	 * @param data 加密数据
	 * @param publicKey 公钥
	 * @param sign 数字签名
	 * @return 校验成功返回true 失败返回false
	 * @throws Exception
	 * 
	 */
	public static boolean verify(byte[] data, byte[] keyBytes, String sign){
		boolean ret = false;
		try {
			// 构造X509EncodedKeySpec对象
			X509EncodedKeySpec keySpec = new X509EncodedKeySpec(keyBytes);
	
			// KEY_ALGORITHM 指定的加密算法
			KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
	
			// 取公钥匙对象
			PublicKey pubKey = keyFactory.generatePublic(keySpec);
	
			Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
			
			signature.initVerify(pubKey);
			signature.update(data);

			// 验证签名是否正常
			ret = signature.verify(Base64.decode(sign));
		} catch (Exception e) {
			log.error(ExceptionUtils.getStackTrace(e));
		}
		return ret;
	}

	/**
	 * 解密<br>
	 * 用字符串私钥解密
	 * 
	 * @param data
	 * @param key
	 * @return byte[]
	 * @throws Exception
	 */
	public static byte[] decryptByPrivateKey(byte[] data, String key)
			throws Exception {
		// 对密钥解密
		byte[] keyBytes = Base64.decode(key);
		return decryptByPrivateKey(data, keyBytes);
	}

	/**
	 * 解密<br>
	 * 用字符串私钥解密
	 * 
	 * @param data
	 * @param key
	 * @return String
	 * @throws Exception
	 */
	public static String decryptByPrivateKeyToString(byte[] data, String key){
		// 对密钥解密
		byte[] keyBytes = Base64.decode(key);
		byte[] bytes = decryptByPrivateKey(data, keyBytes);
		if (bytes == null) {
			return null;
		}
		return new String(Base64.encode(bytes));
	}

	/**
	 * 解密<br>
	 * 用字节私钥解密
	 * 
	 * @param data
	 * @param keyBytes
	 * @return byte[]
	 * @throws Exception
	 */
	public static byte[] decryptByPrivateKey(byte[] data, byte[] keyBytes) {
		byte[] ret = null;
		try {
			// 取得私钥
			PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(keyBytes);
			KeyFactory keyFactory;
			keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
			Key privateKey = keyFactory.generatePrivate(pkcs8KeySpec);

			// 对数据解密
			Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
			cipher.init(Cipher.DECRYPT_MODE, privateKey);
			ret = cipher.doFinal(data);
		} catch (Exception e) {
			log.error(ExceptionUtils.getStackTrace(e));
		}
		return ret;
	}

	/**
	 * 解密<br>
	 * 用字节私钥解密
	 * 
	 * @param data
	 * @param keyBytes
	 * @return String
	 * @throws Exception
	 */
	public static String decryptByPrivateKeyToString(byte[] data, byte[] keyBytes) {
		byte[] bytes = decryptByPrivateKey(data, keyBytes);
		if (bytes == null) {
			return null;
		}
		return new String(Base64.encode(bytes));
	}

	/**
	 * 解密<br>
	 * 用字符串公钥解密
	 * 
	 * @param data
	 * @param key
	 * @return byte[]
	 * @throws Exception
	 */
	public static byte[] decryptByPublicKey(byte[] data, String key) {
		// 对密钥解密
		byte[] bytes = Base64.decode(key);
		return decryptByPublicKey(data, bytes);
	}

	/**
	 * 解密<br>
	 * 用字符串公钥解密
	 * 
	 * @param data
	 * @param key
	 * @return String
	 * @throws Exception
	 */
	public static String decryptByPublicKeyToString(byte[] data, String key) {
		// 对密钥解密
		byte[] keyBytes = Base64.decode(key);
		byte[] bytes = decryptByPublicKey(data, keyBytes);
		if (bytes == null) {
			return null;
		}
		return new String(Base64.encode(bytes));
	}

	/**
	 * 解密<br>
	 * 用字节公钥解密
	 * 
	 * @param data
	 * @param keyBytes
	 * @return byte[]
	 * @throws Exception
	 */
	public static byte[] decryptByPublicKey(byte[] data, byte[] keyBytes) {
		byte[] ret = null;
		try {
			// 取得公钥
			X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(keyBytes);
			KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
			Key publicKey = keyFactory.generatePublic(x509KeySpec);

			// 对数据解密
			Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
			cipher.init(Cipher.DECRYPT_MODE, publicKey);
			ret = cipher.doFinal(data);
		} catch (Exception e) {
			log.error(ExceptionUtils.getStackTrace(e));
		}
		return ret;
	}

	/**
	 * 加密<br>
	 * 用字符串公钥加密
	 * 
	 * @param data
	 * @param key
	 * @return byte[]
	 * @throws Exception
	 */
	public static byte[] encryptByPublicKey(byte[] data, String key) {
		// 对公钥解密
		byte[] keyBytes = Base64.decode(key);
		return encryptByPublicKey(data, keyBytes);
	}
	/**
	 * 加密<br>
	 * 用字符串公钥加密
	 * 
	 * @param data
	 * @param key
	 * @return String
	 * @throws Exception
	 */
	public static String encryptByPublicKeyToString(byte[] data, String key) {
		// 对公钥解密
		byte[] keyBytes = Base64.decode(key);
		byte[] bytes = encryptByPublicKey(data, keyBytes);
		if (bytes == null) {
			return null;
		}
		return new String(Base64.encode(bytes));
	}

	/**
	 * 加密<br>
	 * 用字节公钥加密
	 * 
	 * @param data
	 * @param keyBytes
	 * @return byte[]
	 * @throws Exception
	 */
	public static byte[] encryptByPublicKey(byte[] data, byte[] keyBytes) {
		byte[] ret = null;
		try {
			// 取得公钥
			X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(keyBytes);
			KeyFactory keyFactory;
			keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);

			Key publicKey = keyFactory.generatePublic(x509KeySpec);

			// 对数据加密
			Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
			cipher.init(Cipher.ENCRYPT_MODE, publicKey);
			ret = cipher.doFinal(data);
		} catch (Exception e) {
			log.error(ExceptionUtils.getStackTrace(e));
		}
		return ret;
	}
	/**
	 * 加密<br>
	 * 用字节公钥加密
	 * 
	 * @param data
	 * @param keyBytes
	 * @return String
	 * @throws Exception
	 */
	public static String encryptByPublicKeyToString(byte[] data, byte[] keyBytes) {
		// 对公钥解密
		byte[] bytes = encryptByPublicKey(data, keyBytes);
		if (bytes == null) {
			return null;
		}
		return new String(Base64.encode(bytes));
	}

	/**
	 * 加密<br>
	 * 用字符串私钥加密
	 * 
	 * @param data
	 * @param key
	 * @return byte[]
	 * @throws Exception
	 */
	public static byte[] encryptByPrivateKey(byte[] data, String key){
		// 对密钥解密
		byte[] keyBytes = Base64.decode(key);
		return encryptByPrivateKey(data, keyBytes);
	}
	
	/**
	 * 加密<br>
	 * 用字符串私钥加密
	 * 
	 * @param data
	 * @param key
	 * @return byte[]
	 * @throws Exception
	 */
	public static String encryptByPrivateKeyToString(byte[] data, String key){
		// 对密钥解密
		byte[] keyBytes = Base64.decode(key);
		byte[] bytes = encryptByPrivateKey(data, keyBytes);
		if (bytes == null) {
			return null;
		}
		return new String(Base64.encode(bytes));
	}

	/**
	 * 加密<br>
	 * 用字节私钥加密
	 * 
	 * @param data
	 * @param keyBytes
	 * @return byte[]
	 * @throws Exception
	 */
	public static byte[] encryptByPrivateKey(byte[] data, byte[] keyBytes){
		byte[] ret = null;
		try {
			// 取得私钥
			PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(keyBytes);
			KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
			Key privateKey = keyFactory.generatePrivate(pkcs8KeySpec);
			// 对数据加密
			Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
			cipher.init(Cipher.ENCRYPT_MODE, privateKey);
			ret = cipher.doFinal(data);
		} catch (Exception e) {
			log.error(ExceptionUtils.getStackTrace(e));
		}
		return ret;
	}
	/**
	 * 加密<br>
	 * 用字节私钥加密
	 * 
	 * @param data
	 * @param keyBytes
	 * @return String
	 * @throws Exception
	 */
	public static String encryptByPrivateKeyToString(byte[] data, byte[] keyBytes){
		byte[] bytes = encryptByPrivateKey(data, keyBytes);
		if (bytes == null) {
			return null;
		}
		return new String(Base64.encode(bytes));
	}

	/**
	 * 取得私钥
	 * 
	 * @param keyMap
	 * @return
	 * @throws Exception
	 */
	public static String getPrivateKey(Map<String, Object> keyMap)
			throws Exception {
		Key key = (Key) keyMap.get(PRIVATE_KEY);
		byte[] keyBytes = Base64.encode(key.getEncoded());
		return new String(keyBytes);
	}

	/**
	 * 取得公钥
	 * 
	 * @param keyMap
	 * @return
	 * @throws Exception
	 */
	public static String getPublicKey(Map<String, Object> keyMap) throws Exception {
		Key key = (Key) keyMap.get(PUBLIC_KEY);
		byte[] keyBytes = Base64.encode(key.getEncoded());
		return new String(keyBytes);
	}

	/**
	 * 初始化密钥
	 * 
	 * @return
	 * @throws Exception
	 */
	public static Map<String, Object> initKey(int keySize) throws Exception {
		KeyPairGenerator keyPairGen = KeyPairGenerator.getInstance(KEY_ALGORITHM);
		keyPairGen.initialize(keySize);

		KeyPair keyPair = keyPairGen.generateKeyPair();

		// 公钥
		RSAPublicKey publicKey = (RSAPublicKey) keyPair.getPublic();

		// 私钥
		RSAPrivateKey privateKey = (RSAPrivateKey) keyPair.getPrivate();

		Map<String, Object> keyMap = new HashMap<String, Object>(2);

		keyMap.put(PUBLIC_KEY, publicKey);
		keyMap.put(PRIVATE_KEY, privateKey);
		return keyMap;
	}
}
