package com.clx.performance.service.impl;

import com.clx.performance.dao.*;
import com.clx.performance.model.IntegralRecord;
import com.clx.performance.model.IntegralStatistics;
import com.clx.performance.model.IntegralTruck;
import com.clx.performance.param.mq.*;
import com.clx.performance.service.IntegralMqHandlerService;
import com.clx.performance.service.IntegralMqService;
import com.clx.performance.service.IntegralRecordService;
import com.clx.performance.service.IntegralStatisticsService;
import com.clx.performance.utils.LocalDateTimeUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@Service
public class IntegralMqHandlerServiceImpl implements IntegralMqHandlerService {

    @Autowired
    private IntegralMqService integralMqService;
    @Autowired
    private IntegralStatisticsDao integralStatisticsDao;
    @Autowired
    private IntegralRecordDao integralRecordDao;
    @Autowired
    private IntegralTruckDao integralTruckDao;
    @Autowired
    private IntegralOrderDao integralOrderDao;
    @Autowired
    private OrderChildDao orderChildDao;

    @Autowired
    private OrderGoodsDao orderGoodsDao;

    @Autowired
    private IntegralStatisticsService integralStatisticsService;

    @Autowired
    private IntegralRecordService integralRecordService;

    /**
     * 车辆新增
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void truckAdd(IntegralTuckAddMqParam mq){


    }

    /**
     * 积分记录新增
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void recordAdd(IntegralRecordAddMqParam mq){


    }

    /**
     * 下周车辆初始化（新增统计车辆） （周3、周7 晚上）
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void nextWeekTruckInit(IntegralNextWeekTruckInitMqParam mq){
        String statisticsDateCurrent = mq.getStatisticsDate();

        String statisticsDateNext = LocalDateTimeUtils.formatTime(LocalDateTimeUtils.parseTime(statisticsDateCurrent).plusDays(7));

        // 查找本周车辆
        List<IntegralStatistics> list = integralStatisticsDao.selectByStatisticsDate(statisticsDateCurrent);

        // 查找下周车辆
        List<IntegralStatistics> nextList = integralStatisticsDao.selectByStatisticsDate(statisticsDateNext);
        List<String> truckNoNextList = nextList.stream().map(item -> item.getTruckNo()).collect(Collectors.toList());

        List<IntegralStatistics> newList = new ArrayList<>();
        for (IntegralStatistics item : list) {
            if (truckNoNextList.contains(item.getTruckNo())) {continue;}

            IntegralStatistics statistics = new IntegralStatistics();
            statistics.setTruckNo(item.getTruckNo());
            statistics.setStatisticsDate(statisticsDateNext);
            statistics.setIntegralBase(0);
            statistics.setIntegral(0);
            statistics.setRank(1);

            newList.add(statistics);
        }
        if (newList.isEmpty()) {return;}

        // 新增下周车辆
        integralStatisticsDao.batchSaveEntity(newList);

    }

    /**
     * 车辆周统计 (统计周初始积分)
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void weekTruckStatistics(IntegralWeekTruckStatisticsMqParam mq){

        if (StringUtils.isNotBlank(mq.getTruckNo())){
            // 基础分初始化
            truckBaseIntegralInit(mq.getTruckNo(), mq.getStatisticsDate());
            return;
        }

        // 更新本周实时排名
        integralRealTimeRankStatistics(mq.getStatisticsDate());
    }

    /**
     * 新一周基础积分初始化
     */
    private void truckBaseIntegralInit(String truckNo, String statisticsDate){
        IntegralTruck truck = integralTruckDao.getByTruckNo(truckNo).orNull();
        if (null == truck) {return;}

        IntegralStatistics statistics = integralStatisticsDao.selectByTruckNoAndStatisticsDate(truck.getTruckNo(), statisticsDate).orNull();
        if (null == statistics) {return;}

        // 新车保护分
        newTruckProtection(truck, statistics);

        // 周结算分
        truckWeekSettlement(truck, statistics);

        integralStatisticsDao.updateIntegral(statistics);

        truck.setIntegral(statistics.getIntegral());
        integralTruckDao.updateIntegral(truck);

    }

    /**
     * 新车保护分
     */
    private void newTruckProtection(IntegralTruck truck, IntegralStatistics statistics){
        String statisticsDate = statistics.getStatisticsDate();

        // 超过保护期
        if (LocalDateTimeUtils.parseTime(truck.getCreateTime()).isBefore(LocalDateTimeUtils.parseTime(statisticsDate).minusDays(7))) {return;}

        // 已保护不处理
        IntegralRecord record = integralRecordDao.selectByStatisticsIdAndType(statistics.getId(), IntegralRecord.type.PROTECTION.getValue());
        if (null != record) {return;}

        // 计算新车积分
        int integral = integralStatisticsService.protectionIntegralCalc(truck.getCreateTime(), statisticsDate);
        if (integral == 0) {return;}

        statistics.setIntegral(statistics.getIntegral()+integral);
        integralStatisticsDao.updateIntegral(statistics);

        integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecord.type.PROTECTION.getValue(), integral, "新用户");

    }

    /**
     * 上周结算分
     */
    private void truckWeekSettlement(IntegralTruck truck, IntegralStatistics statistics){
        int integral;

        // 已结算不处理
        IntegralRecord record = integralRecordDao.selectByStatisticsIdAndType(statistics.getId(), IntegralRecord.type.SETTLEMENT.getValue());
        if (null != record) {return;}

        String statisticsDatePre = LocalDateTimeUtils.formatTime(LocalDateTimeUtils.parseTime(statistics.getStatisticsDate()).minusDays(7));

        // 查找上周统计数据
        IntegralStatistics statisticsPre = integralStatisticsDao.selectByTruckNoAndStatisticsDate(truck.getTruckNo(), statisticsDatePre).orNull();
        if (null == statisticsPre) {return;}

        // 查找总车数
        int truckSum = integralStatisticsDao.countByStatisticsDate(statisticsDatePre);

        if (statisticsPre.getRank() <= truckSum*0.1){
            integral = 300;
        }
        else if ( (statisticsPre.getRank() > truckSum*0.1) && (statisticsPre.getRank() <= truckSum*0.3)){
            integral = 200;
        }
        else if ( (statisticsPre.getRank() > truckSum*0.3) && (statisticsPre.getRank() <= truckSum*0.6)){
            integral = 100;
        }
        else if ( (statisticsPre.getRank() > truckSum*0.6) && (statisticsPre.getRank() <= truckSum*0.9)){
            integral = 60;
        }
        else {
            integral = 0;
        }
        if (integral == 0) {return;}

        statistics.setIntegral(statistics.getIntegral()+integral);
        integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecord.type.SETTLEMENT.getValue(), integral, "上周结算");

    }

    /**
     * 新手保护取消
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void truckProtectionCancel(IntegralTruckProtectionCancelMqParam mq){
        IntegralTruck truck = integralTruckDao.getByTruckNo(mq.getTruckNo()).orNull();
        if (null == truck) {return;}

        IntegralStatistics statistics = integralStatisticsDao.selectByTruckNoAndStatisticsDate(truck.getTruckNo(), mq.getStatisticsDate()).orNull();
        if (null == statistics) {return;}

        IntegralRecord record = integralRecordDao.selectByStatisticsIdAndType(statistics.getId(), IntegralRecord.type.PROTECTION.getValue());
        IntegralRecord recordCancel = integralRecordDao.selectByStatisticsIdAndType(statistics.getId(), IntegralRecord.type.PROTECTION_CANCEL.getValue());
        if (null != recordCancel) {return;}
        int integral = -record.getIntegral();

        integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecord.type.PROTECTION_CANCEL.getValue(), integral, "新用户过保护期");

        statistics.setIntegral(statistics.getIntegral()+integral);
        integralStatisticsDao.updateIntegral(statistics);

        truck.setIntegral(statistics.getIntegral());
        integralTruckDao.updateIntegral(truck);

        // 更新本周实时排名
        integralRealTimeRankStatistics(mq.getStatisticsDate());

    }

    /**
     * 运单结束
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void orderChildFinish(IntegralOrderChildFinishMqParam mq) {

    }

    /**
     * 更新实时积分
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void integralRealTimeStatistics(Integer statisticsId){

    }

    /**
     * 更新实时排名
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void integralRealTimeRankStatistics(String statisticsDate){

    }

}
