package com.clx.performance.controller.pc.export;

import com.clx.performance.param.pc.export.*;
import com.clx.performance.service.export.ExportFieldService;
import com.clx.performance.vo.pc.export.ExportFieldVo;
import com.clx.performance.vo.pc.export.ExportTemplateDetailVo;
import com.clx.performance.vo.pc.export.ExportTemplateVo;
import com.msl.common.result.Result;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import jakarta.validation.constraints.NotBlank;
import java.util.List;

@Slf4j
@RestController
@Validated
@Tag(name = "模板")
@RequestMapping(value="/pc/carrier/template")

public class TemplateController {
    @Autowired
    private ExportFieldService exportFieldService;

    @GetMapping("/listTemplateByFunctionCodes")
    @Operation(summary = "列表查询登陆用户某功能下的模板", description = "<br>by 艾庆国")
    @Validated
    public Result<List<ExportTemplateVo>> listTemplateByFunctionCodes(@Parameter(name = "功能code") @NotBlank(message = "功能code不能为空") String functionCode) {
        return Result.ok(exportFieldService.listUserFunctionTemplate(functionCode));
    }

    @PostMapping("/listTemplateByFunctionCodes")
    @Operation(summary = "列表查询登陆用户某些功能下的模板", description = "<br>by 艾庆国")
    public Result<List<ExportTemplateVo>> listTemplateByFunctionCodes(@RequestBody ListExportTemplateParam param) {
        return Result.ok(exportFieldService.listUserFunctionsTemplate(param.getFunctionCodes()));
    }

    @PostMapping("/saveTemplate")
    @Operation(summary = "新增模板", description = "<br>by 艾庆国")
    public Result<Integer> saveTemplate(@Validated @RequestBody SaveExportTemplateParam param) {
        return Result.ok(exportFieldService.saveTemplate(param));
    }

    @DeleteMapping("/deleteTemplate")
    @Operation(summary = "删除模板", description = "<br>by 艾庆国")
    public Result<Void> deleteTemplate(@Parameter(name = "模板id") @RequestParam("id") Integer id) {
        exportFieldService.deleteTemplate(id);
        return Result.ok();
    }

    @PostMapping("/updateTemplateName")
    @Operation(summary = "修改模板名称", description = "<br>by 艾庆国")
    public Result<Void> updateTemplateName(@Validated @RequestBody UpdateTemplateNameParam param) {
        exportFieldService.updateTemplateName(param.getId(), param.getName());
        return Result.ok();
    }

    @PostMapping("/updateTemplateDefaultFlag")
    @Operation(summary = "设置默认模板", description = "<br>by 艾庆国")
    public Result<Integer> updateTemplateDefaultFlag(@Validated @RequestBody UpdateTemplateDefaultFlagParam param) {
        exportFieldService.updateTemplateDefaultFlag(param.getId());
        return Result.ok();
    }

    @PostMapping("/updateTemplateFields")
    @Operation(summary = "修改模板下的字段列表", description = "<br>by 艾庆国")
    public Result<Integer> updateTemplateFields(@Validated @RequestBody UpdateTemplateFieldsParam param) {
        exportFieldService.updateTemplateFields(param.getId(), param.getFieldIds());
        return Result.ok();
    }

    @GetMapping("/getTemplateDetail")
    @Operation(summary = "获取模板详情", description = "<br>by 艾庆国")
    public Result<ExportTemplateDetailVo> getTemplateDetail(@Parameter(name = "模板id") Integer id) {
        return Result.ok(exportFieldService.getTemplateDetail(id).orNull());
    }

    @GetMapping("/listFieldByTemplate")
    @Operation(summary = "查询模板下的所有字段", description = "<br>by 艾庆国")
    public Result<List<ExportFieldVo>> listFieldByTemplate(@Parameter(name = "模板id") @RequestParam Integer id) {
        return Result.ok(exportFieldService.listFieldByTemplate(id));
    }


    @GetMapping("/listFieldByFunctionCode")
    @Operation(summary = "查询功能下的所有字段", description = "<br>by 艾庆国")
    public Result<List<ExportFieldVo>> listFieldByFunctionCode(@Parameter(name = "功能code") String functionCode) {
        return Result.ok(exportFieldService.listFieldByFunctionCode(functionCode));
    }
}
