package com.clx.performance.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.clx.order.vo.feign.FeignAddressVO;
import com.clx.performance.dao.*;
import com.clx.performance.enums.*;
import com.clx.performance.extranal.user.AddressService;
import com.clx.performance.extranal.user.DriverService;
import com.clx.performance.model.*;
import com.clx.performance.param.app.*;
import com.clx.performance.param.pc.PageCarrierOrderChildParam;
import com.clx.performance.param.pc.PagePoundAuditParam;
import com.clx.performance.service.OrderChildLogService;
import com.clx.performance.service.OrderChildPoundLogService;
import com.clx.performance.service.OrderChildService;
import com.clx.performance.service.OrderGoodsService;
import com.clx.performance.struct.*;
import com.clx.performance.vo.app.*;
import com.clx.performance.vo.pc.CarrierOrderChildDetailVO;
import com.clx.performance.vo.pc.OrderChildPCVO;
import com.clx.performance.vo.pc.PageCarrierOrderChildVO;
import com.clx.performance.vo.pc.PageOrderChildPoundAuditVO;
import com.clx.user.vo.feign.DriverTruckInfoFeignVo;
import com.msl.common.base.Optional;
import com.msl.common.enums.ResultCodeEnum;
import com.msl.common.exception.ServiceSystemException;
import com.msl.common.utils.DateUtils;
import com.msl.common.utils.LocalDateTimeUtils;
import com.msl.user.data.UserSessionData;
import com.msl.user.utils.TokenUtil;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author: aiqinguo
 * @Description: 运单表
 * @Date: 2023/09/18 11:34:50
 * @Version: 1.0
 */

@Slf4j
@Service
@AllArgsConstructor
public class OrderChildServiceImpl implements OrderChildService {

    private final OrderGoodsDao orderGoodsDao;
    private final OrderGoodsTruckBindDao orderGoodsTruckBindDao;

    private final OrderChildDao orderChildDao;

    private final OrderChildImageDao orderChildImageDao;
    private final OrderChildPoundAuditDao orderChildPoundAuditDao;

    private final OrderChildLogService orderChildLogService;
    private final OrderChildPoundLogService orderChildPoundLogService;
    private final AddressService addressService;

    private final DriverService driverService;
    private final OrderChildStruct orderChildStruct;
    private final OrderChildPoundImageStruct orderChildPoundImageStruct;
    private final OrderGoodsService orderGoodsService;
    private final OrderChildPoundStruct orderChildPoundStruct;
    private final OrderChildFreightStruct orderChildFreightStruct;
    private final AddressStruct addressStruct;
    private final OrderChildLogStruct orderChildLogStruct;

    private final UniqueOrderNumService uniqueOrderNumService;


    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public SaveOrderChildVO saveOrderChild(OrderChildSaveParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = childNoGenerate();

        // 查询司机车辆信息
        DriverTruckInfoFeignVo driverTruckInfo = driverService.getUserDetailInfo(param.getDriverUserNo(), param.getTruckId()).orElseThrow(ResultCodeEnum.FAIL);
        BigDecimal truckLoad = driverTruckInfo.getLoad();
        String truckNo = driverTruckInfo.getTruckNo();

        LocalDateTime now = LocalDateTime.now();
        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(param.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.ORDER_INVALID);
        // 状态验证
        if (!Objects.equals(orderGoods.getOrderGoodsStatus(), OrderGoodsStatusEnum.Status.PAYING.getCode())
                && !Objects.equals(orderGoods.getOrderGoodsStatus(), OrderGoodsStatusEnum.Status.GO_TO_SEND.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_INVALID);
        }

        // 定向
        OrderGoodsTruckBind orderGoodsTruckBind = orderGoodsTruckBindDao.getValidByOrderGoodsNoAndTruckNo(orderGoods.getOrderGoodsNo(), truckNo).orNull();
        if (Objects.equals(orderGoods.getPendingOrderWay(), OrderGoodsPendingOrderWayStatusEnum.Status.EXCLUSIVE.getCode())){
            if (orderGoodsTruckBind == null){throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_DIRECT_ORDER_TRUCK_ERROR);}
        }
        else {
            if (orderGoodsTruckBind != null){throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_DIRECT_ORDER_TRUCK_ERROR1);}
        }

        // 库存验证
        if (orderGoods.getResidueTransportWeight().compareTo(BigDecimal.ZERO) <= 0) {throw new ServiceSystemException(PerformanceResultEnum.ORDER_WEIGHT_LACK);}
        if (orderGoods.getResidueTransportWeight().compareTo(truckLoad) < 0) {throw new ServiceSystemException(PerformanceResultEnum.ORDER_WEIGHT_LACK);}

        OrderChild orderChild = new OrderChild();
        orderChild.setChildNo(childNo);
        orderChild.setUserNo(userNo);

        orderChild.setOwnerUserNo(orderGoods.getUserNo());
        orderChild.setOwnerName(orderGoods.getUserName());

        orderChild.setOrderNo(orderGoods.getOrderNo());
        orderChild.setOrderGoodsNo(orderGoods.getOrderGoodsNo());
        orderChild.setFreightPrice(orderGoods.getPendingOrderFreight());
        orderChild.setLossPrice(orderGoods.getLossPrice());
        orderChild.setOrderFreightPrice(BigDecimal.ZERO);

        orderChild.setGoodsId(orderGoods.getGoodsId());
        orderChild.setGoodsName(orderGoods.getGoodsName());

        orderChild.setSendAddressId(orderGoods.getSendAddressId());
        orderChild.setSendAddress(orderGoods.getSendAddressShorter());
        orderChild.setReceiveAddressId(orderGoods.getReceiveAddressId());
        orderChild.setReceiveAddress(orderGoods.getReceiveAddressShorter());

        orderChild.setLoadDeadline(orderGoods.getLastArriveSendTime());

        orderChild.setDriverUserNo(driverTruckInfo.getUserNo());
        orderChild.setDriverName(driverTruckInfo.getName());
        orderChild.setDriverMobile(driverTruckInfo.getMobile());

        orderChild.setTruckId(driverTruckInfo.getTruckId());
        orderChild.setTruckNo(driverTruckInfo.getTruckNo());
        orderChild.setTruckLoad(truckLoad);
        orderChild.setTruckModel(driverTruckInfo.getModel());

        orderChild.setWeight(orderChild.getTruckLoad());
        orderChild.setFreight(orderChildFreightCalc(orderChild));
        orderChild.setPayTime(now);
        orderChild.setStatus(OrderChildEnum.Status.CREATED.getCode());
        orderChild.setCreateTime(now);

        // 更新货单数据
        updateOrderGoodsAmount(orderGoods, truckLoad);

        // 更新定向单状态
        updateOrderGoodsDirect(orderGoodsTruckBind);

        // 新增运单
        orderChildDao.saveEntity(orderChild);

        // 更新司机状态
        lockDriver(orderChild.getDriverUserNo());

        // 更新车辆状态
        lockTruck(orderChild.getTruckId());

        // 新增日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.CREATED.getCode(), OrderChildLogEnum.Type.CREATED.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());

        SaveOrderChildVO result = new SaveOrderChildVO();
        result.setChildNo(childNo);
        return result;
    }

    @Override
    public void updateReject(OrderChildRejectParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(param.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.ORDER_INVALID);
        // 状态验证
        if (!Objects.equals(orderGoods.getOrderGoodsStatus(), OrderGoodsStatusEnum.Status.PAYING.getCode())
                && !Objects.equals(orderGoods.getOrderGoodsStatus(), OrderGoodsStatusEnum.Status.GO_TO_SEND.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_INVALID);
        }

        // 定向派单
        if (!Objects.equals(orderGoods.getPendingOrderWay(), 2)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_INVALID);
        }

        OrderGoodsTruckBind orderGoodsTruckBind = orderGoodsTruckBindDao.getValidByOrderGoodsNoAndTruckNo(orderGoods.getOrderGoodsNo(), param.getTruckNo())
                .orElseThrow(PerformanceResultEnum.ORDER_CHILD_DIRECT_REJECT_TRUCK_ERROR);

        // 更新定向派单
        updateOrderGoodsDirectReject(orderGoodsTruckBind);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateGotoSendAddress(OrderChildGoToSendAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_SEND.getCode())){return;}
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.CREATED.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        orderChild.setStatus(OrderChildEnum.Status.GO_TO_SEND.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.GO_TO_SEND.getCode(), OrderChildLogEnum.Type.GO_TO_SEND.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateArriveSendAddress(OrderChildArriveSendAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_SEND.getCode())){return;}
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_SEND.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }
        // 装货超时
        if (orderChild.getLoadDeadline().isBefore(LocalDateTime.now())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_LOAD_TIMEOUT);
        }

        orderChild.setStatus(OrderChildEnum.Status.ARRIVE_SEND.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.ARRIVE_SEND.getCode(), OrderChildLogEnum.Type.ARRIVE_SEND.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateGotoReceiveAddress(OrderChildGoToReceiveAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_RECEIVE.getCode())){return;}
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.LOAD.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }


        orderChild.setStatus(OrderChildEnum.Status.GO_TO_RECEIVE.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.GO_TO_RECEIVE.getCode(), OrderChildLogEnum.Type.GO_TO_RECEIVE.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateArriveReceiveAddress(OrderChildArriveReceiveAddressParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_RECEIVE.getCode())){return;}
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.GO_TO_RECEIVE.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        orderChild.setStatus(OrderChildEnum.Status.ARRIVE_RECEIVE.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.ARRIVE_RECEIVE.getCode(), OrderChildLogEnum.Type.ARRIVE_RECEIVE.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateLoad(OrderChildLoadParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CANCEL.getCode())
                || Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.PLATFORM_CANCEL.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCELED);
        }

        if (loadCheck(param.getLoadNet(), orderChild.getTruckLoad())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_LOAD_WEIGHT_ERROR);
        }

        if (orderChild.getLoadTime() == null && Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_SEND.getCode())) {
            updateLoadFirst(param, orderChild);
            // 日志
            orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.LOAD.getCode(), OrderChildLogEnum.Type.LOAD.getMsg(),
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        } else {
            updateReload(param, orderChild);
            // 日志
            orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.RELOAD.getCode(), OrderChildLogEnum.Type.RELOAD.getMsg(),
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        }
    }

    private void updateLoadFirst(OrderChildLoadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        // 装货超时
        if (orderChild.getLoadDeadline().isBefore(LocalDateTime.now())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_LOAD_TIMEOUT);
        }

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getLoadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.LOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        orderChild.setLoadRough(param.getLoadRough());
        orderChild.setLoadTare(param.getLoadTare());
        orderChild.setLoadNet(param.getLoadNet());
        orderChild.setLoadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));
        orderChild.setStatus(OrderChildEnum.Status.LOAD.getCode());
        orderChildDao.updateLoad(orderChild);

        //

        orderChildImageDao.batchSaveEntity(imageList);
    }

    private void updateReload(OrderChildLoadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getLoadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.LOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        orderChild.setLoadRough(param.getLoadRough());
        orderChild.setLoadTare(param.getLoadTare());
        orderChild.setLoadNet(param.getLoadNet());
        orderChild.setLoadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));
        orderChildDao.updateLoad(orderChild);

        orderChildImageDao.deleteLoad(childNo);
        orderChildImageDao.batchSaveEntity(imageList);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateUnload(OrderChildUnloadParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CANCEL.getCode())
                || Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.PLATFORM_CANCEL.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCELED);
        }

        if (loadCheck(param.getUnloadNet(), orderChild.getTruckLoad())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_UNLOAD_WEIGHT_ERROR);
        }

        if (orderChild.getUnloadTime() == null && Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.ARRIVE_RECEIVE.getCode())) {
            updateUnloadFirst(param, orderChild);
            // 日志
            orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.UNLOAD.getCode(), OrderChildLogEnum.Type.UNLOAD.getMsg(),
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
            orderChildPoundLogService.saveDriverOrderChildLog(childNo, OrderChildPoundAuditEnum.Status.AUDIT.getCode(), "",
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        } else {
            updateReUnload(param, orderChild);
            // 日志
            orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.REUNLOAD.getCode(), OrderChildLogEnum.Type.REUNLOAD.getMsg(),
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
            orderChildPoundLogService.saveDriverOrderChildLog(childNo, OrderChildPoundAuditEnum.Status.AUDIT.getCode(), "",
                    loginUserInfo.getUserNo(), loginUserInfo.getUserName());
        }
    }

    private void updateUnloadFirst(OrderChildUnloadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getUnloadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.UNLOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        orderChild.setUnloadPoundNo(param.getUnloadPoundNo());
        orderChild.setUnloadRough(param.getUnloadRough());
        orderChild.setUnloadTare(param.getUnloadTare());
        orderChild.setUnloadNet(param.getUnloadNet());
        orderChild.setUnloadTime(LocalDateTime.now());
        orderChild.setWeight(orderChildWeightCalc(orderChild));
        orderChild.setStatus(OrderChildEnum.Status.UNLOAD.getCode());
        orderChild.setPoundStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());

        OrderChildPoundAudit audit = new OrderChildPoundAudit();
        audit.setChildNo(childNo);
        audit.setStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());
        audit.setLoadNet(orderChild.getLoadNet());
        audit.setUnloadNet(orderChild.getUnloadNet());

        orderChildDao.updateUnload(orderChild);

        orderChildImageDao.batchSaveEntity(imageList);

        orderChildPoundAuditDao.saveEntity(audit);

        // 释放司机、车辆
        releaseDriver(orderChild.getDriverUserNo());
        releaseTruck(orderChild.getTruckId());

    }

    private void updateReUnload(OrderChildUnloadParam param, OrderChild orderChild) {
        String childNo = param.getChildNo();

        // 审核中
        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.AUDIT.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_POUND_AUDIT);
        }

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getUnloadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.UNLOAD.getCode());
            image.setImage(item);
            imageList.add(image);
        }

        orderChild.setUnloadPoundNo(param.getUnloadPoundNo());
        orderChild.setUnloadRough(param.getUnloadRough());
        orderChild.setUnloadTare(param.getUnloadTare());
        orderChild.setUnloadNet(param.getUnloadNet());
        orderChild.setUnloadTime(LocalDateTime.now());
        orderChild.setPoundStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());

        OrderChildPoundAudit audit = new OrderChildPoundAudit();
        audit.setChildNo(childNo);
        audit.setStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());
        audit.setLoadNet(orderChild.getLoadNet());
        audit.setUnloadNet(orderChild.getUnloadNet());
        audit.setUnloadPoundNo(orderChild.getUnloadPoundNo());

        orderChildDao.updateUnload(orderChild);

        orderChildImageDao.deleteUnload(childNo);
        orderChildImageDao.batchSaveEntity(imageList);

        orderChildPoundAuditDao.saveEntity(audit);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateLoadAndUnloadAgain(OrderChildLoadAndUnloadAgainParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();
        String childNo = param.getChildNo();
        LocalDateTime now = LocalDateTime.now();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CANCEL.getCode())
                || Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.PLATFORM_CANCEL.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCELED);
        }
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.UNLOAD.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.AUDIT.getCode())){return;}
        if (!Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.REJECT.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        if (loadCheck(param.getLoadNet(), orderChild.getTruckLoad())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_LOAD_WEIGHT_ERROR);
        }
        if (loadCheck(param.getUnloadNet(), orderChild.getTruckLoad())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_UNLOAD_WEIGHT_ERROR);
        }

        List<OrderChildImage> imageList = new ArrayList<>();
        for (String item : param.getLoadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.LOAD.getCode());
            image.setImage(item);
        }
        for (String item : param.getUnloadImageList()) {
            OrderChildImage image = new OrderChildImage();
            image.setChildNo(orderChild.getChildNo());
            image.setType(OrderChildImage.Type.UNLOAD.getCode());
            image.setImage(item);
        }

        orderChild.setLoadRough(param.getLoadRough());
        orderChild.setLoadTare(param.getLoadTare());
        orderChild.setLoadNet(param.getLoadNet());
        orderChild.setUnloadRough(param.getUnloadRough());
        orderChild.setUnloadTare(param.getUnloadTare());
        orderChild.setUnloadNet(param.getUnloadNet());
        orderChild.setWeight(orderChildWeightCalc(orderChild));

        orderChild.setPoundStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());

        OrderChildPoundAudit audit = new OrderChildPoundAudit();
        audit.setChildNo(childNo);
        audit.setStatus(OrderChildPoundAuditEnum.Status.AUDIT.getCode());
        audit.setLoadNet(orderChild.getLoadNet());
        audit.setUnloadNet(orderChild.getUnloadNet());
        audit.setUnloadPoundNo(orderChild.getUnloadPoundNo());

        orderChildDao.updateLoadAndUnload(orderChild);

        orderChildImageDao.deleteLoadAndUnload(childNo);
        orderChildImageDao.batchSaveEntity(imageList);

        orderChildPoundAuditDao.saveEntity(audit);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.RELOAD_AND_REUNLOAD.getCode(), OrderChildLogEnum.Type.RELOAD_AND_REUNLOAD.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());

        orderChildPoundLogService.saveDriverOrderChildLog(childNo, OrderChildPoundAuditEnum.Status.AUDIT.getCode(), "",
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDriverConfirm(OrderChildDriverConfirmParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.DRIVER_CANCEL.getCode())
                || Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.PLATFORM_CANCEL.getCode())){
            return;
        }
        if (!Objects.equals(orderChild.getStatus(), OrderChildEnum.Status.UNLOAD.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        // 审核中
        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.AUDIT.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_POUND_AUDIT);
        }
        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.REJECT.getCode())){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_STATUS_CHANGED);
        }

        orderChild.setStatus(OrderChildEnum.Status.UNSETTLE.getCode());
        orderChildDao.updateStatus(orderChild);

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.DRIVER_CONFIRM.getCode(), OrderChildLogEnum.Type.DRIVER_CONFIRM.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDriverCancel(OrderChildDriverCancelParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        String childNo = param.getChildNo();
        LocalDateTime now = LocalDateTime.now();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }
        if (OrderChildEnum.CANCEL_lIST.contains(orderChild.getStatus())){
            return;
        }

        // 禁止取消
        if (orderChild.getStatus()>OrderChildEnum.Status.ARRIVE_SEND.getCode()) {
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCEL_FORBID);
        }

        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);

        // 取消量验证
        if (!cancelCountCheck(userNo)){throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_CANCEL_FORBID_COUNT);}

        // 定向
        OrderGoodsTruckBind orderGoodsTruckBind = null;
        if (Objects.equals(orderGoods.getPendingOrderWay(), OrderGoodsPendingOrderWayStatusEnum.Status.EXCLUSIVE.getCode())){
            orderGoodsTruckBind = orderGoodsTruckBindDao.getByOrderGoodsNoAndTruckNo(orderGoods.getOrderGoodsNo(), orderChild.getTruckNo())
                    .orElseThrow(PerformanceResultEnum.DATA_NOT_FIND);
        }

        orderChild.setCancelRemark(param.getRemark());
        orderChild.setCancelTime(now);
        orderChild.setFinishTime(now);
        orderChild.setStatus(OrderChildEnum.Status.DRIVER_CANCEL.getCode());

        orderChildDao.updateCancel(orderChild);

        // 返回吨数
        updateOrderGoodsAmountReturn(orderChild, orderGoods);

        // 取消定向
        updateOrderGoodsDirectCancel(orderGoodsTruckBind);

        // 释放司机、车辆
        releaseDriver(orderChild.getDriverUserNo());
        releaseTruck(orderChild.getTruckId());

        // 日志
        orderChildLogService.saveDriverOrderChildLog(childNo, OrderChildLogEnum.Type.DRIVER_CANCEL.getCode(), OrderChildLogEnum.Type.DRIVER_CANCEL.getMsg(),
                loginUserInfo.getUserNo(), loginUserInfo.getUserName());
    }

    @Override
    public OrderChildVO getOrderChildInfo(String childNo) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        OrderChildVO result = orderChildStruct.convert(orderChild);

        // 磅单
        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.REJECT.getCode())){
            Optional<OrderChildPoundAudit> poundAuditDetail = orderChildPoundAuditDao.getPoundAuditDetail(childNo);
            result.setPoundRemark(poundAuditDetail.orNull().getRemark());
            result.setPoundRejectType(poundAuditDetail.orNull().getRejectType());
        }
        List<OrderChildImage> imageList = orderChildImageDao.listLoadAndUnload(childNo).orElse(new ArrayList<>());
        result.setLoadImageList(imageList.stream().filter(item->Objects.equals(item.getType(),OrderChildImage.Type.LOAD.getCode())).map(item->item.getImage()).collect(Collectors.toList()));
        result.setUnloadImageList(imageList.stream().filter(item->Objects.equals(item.getType(),OrderChildImage.Type.UNLOAD.getCode())).map(item->item.getImage()).collect(Collectors.toList()));

        // 地址
        Optional<FeignAddressVO> sendAndReceiveAddress = addressService.getSendAndReceiveAddress(orderChild.getSendAddressId(), orderChild.getReceiveAddressId());
        if (sendAndReceiveAddress.isPresent()){
            result.setSendAddressInfo(addressStruct.convert(sendAndReceiveAddress.get().getSendAddress()));
        }
        if (sendAndReceiveAddress.isPresent()){
            result.setReceiveAddressInfo(addressStruct.convert(sendAndReceiveAddress.get().getReceiveAddress()));
        }

        return result;
    }

    @Override
    public IPage<OrderChildVO> pageOrderChild(PageOrderChildOfDriverParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        param.setUserNo(userNo);
        return orderChildDao.pageOrderChildOfDriver(param);

    }

    @Override
    public IPage<OrderChildVO> pageSearchOrderChild(PageOrderChildOfDriverSearchParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        param.setUserNo(userNo);
        return orderChildDao.pageSearchOrderChildOfDriver(param);
    }

    @Override
    public OrderChildEstimatedFreightVO getEstimatedFreight(OrderChildEstimatedFreightParam param) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();
        String childNo = param.getChildNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        OrderChildEstimatedFreightVO result = new OrderChildEstimatedFreightVO();
        result.setFreight(orderChild.getFreightPrice().multiply(param.getLoadNet()).setScale(0, RoundingMode.HALF_UP));

        return result;
    }

    @Override
    public OrderChildPoundInfoVO getOrderChildPoundInfo(String childNo) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        OrderChildPoundInfoVO result = orderChildPoundStruct.convert(orderChild);

        if (Objects.equals(orderChild.getPoundStatus(), OrderChildPoundAuditEnum.Status.REJECT.getCode())){
            result.setPoundRemark(orderChildPoundAuditDao.getPoundAuditDetail(childNo).orNull().getRemark());
        }

        List<OrderChildImage> imageList = orderChildImageDao.listLoadAndUnload(childNo).orElse(new ArrayList<>());
        result.setLoadImageList(imageList.stream().filter(item->Objects.equals(item.getType(),OrderChildImage.Type.LOAD.getCode())).map(item->item.getImage()).collect(Collectors.toList()));
        result.setUnloadImageList(imageList.stream().filter(item->Objects.equals(item.getType(),OrderChildImage.Type.UNLOAD.getCode())).map(item->item.getImage()).collect(Collectors.toList()));

        return result;
    }

    @Override
    public OrderChildFreightInfoVO getOrderChildFreightInfo(String childNo) {
        UserSessionData loginUserInfo = TokenUtil.getLoginUserInfo();
        Long userNo = loginUserInfo.getUserNo();

        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        if (!Objects.equals(orderChild.getUserNo(), userNo) && !Objects.equals(orderChild.getDriverUserNo(), userNo)){
            throw new ServiceSystemException(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        }

        OrderChildFreightInfoVO result = orderChildFreightStruct.convert(orderChild);

        return result;
    }


    /**
     * 更新货单数据
     */
    private void updateOrderGoodsAmount(OrderGoods orderGoods, BigDecimal weight){
        orderGoodsDao.updateOrderGoodsReduceWeightAndStatus(orderGoods.getId(), weight);
    }

    /**
     * 取消返吨数
     */
    private void updateOrderGoodsAmountReturn(OrderChild orderChild, OrderGoods orderGoods){
        orderGoodsDao.updateOrderGoodsReduceWeightAndStatus(orderGoods.getId(), orderChild.getWeight().negate());
    }

    /**
     * 更新定向单状态 (接单)
     */
    private void updateOrderGoodsDirect(OrderGoodsTruckBind orderGoodsTruckBind){
        if (orderGoodsTruckBind == null) {return;}

        orderGoodsTruckBind.setStatus(OrderGoodsTruckBind.Status.ORDER.getCode());
        orderGoodsTruckBindDao.updateStatus(orderGoodsTruckBind);
    }

    /**
     * 更新定向派单状态 (拒绝)
     */
    private void updateOrderGoodsDirectReject(OrderGoodsTruckBind orderGoodsTruckBind){
        orderGoodsTruckBind.setStatus(OrderGoodsTruckBind.Status.CANCEL.getCode());
        orderGoodsTruckBindDao.updateStatus(orderGoodsTruckBind);
    }

    /**
     * 更新定向派单状态 (取消)
     */
    private void updateOrderGoodsDirectCancel(OrderGoodsTruckBind orderGoodsTruckBind){
        if (orderGoodsTruckBind == null) {return;}

        orderGoodsTruckBind.setStatus(OrderGoodsTruckBind.Status.CANCEL.getCode());
        orderGoodsTruckBindDao.updateStatus(orderGoodsTruckBind);
    }

    /**
     * 拉运吨数检测
     */
    private boolean loadCheck(BigDecimal weight, BigDecimal truckLoad){
        if (weight.compareTo(truckLoad.multiply(new BigDecimal("1.5"))) > 0) {return false;}
        return true;
    }

    /**
     * 取消检测
     */
    private boolean cancelCountCheck(Long userNo){
        LocalDateTime startTime = DateUtils.parseDateTime(DateUtils.formatDateTime(LocalDateTime.now(), "yyyy-MM-dd").get() + " 00:00:00").get();
        LocalDateTime endTime = DateUtils.parseDateTime(DateUtils.formatDateTime(LocalDateTime.now(), "yyyy-MM-dd").get() + " 00:00:00").get();

        long count = orderChildDao.countOfCancel(userNo, startTime, endTime);

        return count<10;
    }

    /**
     * 运单拉运吨数计算
     */
    private BigDecimal orderChildWeightCalc(OrderChild orderChild){
        if (orderChild.getUnloadNet() == null) {return orderChild.getTruckLoad();}
        else {return orderChild.getLoadNet();}
    }

    /**
     * 运费计算
     */
    private BigDecimal orderChildFreightCalc(OrderChild orderChild){
        return orderChild.getWeight().multiply(orderChild.getFreightPrice()).setScale(0, BigDecimal.ROUND_HALF_UP);
    }


    /**
     * 接单锁定司机
     */
    private void lockDriver(Long driverNo){

    }

    /**
     * 接单锁定车辆
     */
    private void lockTruck(Integer truckId){

    }

    /**
     * 释放司机
     */
    private void releaseDriver(Long driverNo){

    }

    /**
     * 释放车辆
     */
    private void releaseTruck(Integer truckId){

    }

    /**
     * 创建运单号
     */
    private String childNoGenerate(){
        return "CYD"+uniqueOrderNumService.getUniqueOrderNum(LocalDateTimeUtils.convertLocalDateTimeToString(LocalDateTime.now(),LocalDateTimeUtils.DATE_DAY));
    }

    @Override
    public IPage<PageOrderChildPoundAuditVO> pagePoundAuditList(PagePoundAuditParam param) {
        return orderChildDao.pagePoundAuditList(param);
    }

    @Override
    public IPage<PageCarrierOrderChildVO> pageCarrierOrderChildList(PageCarrierOrderChildParam param) {
        return orderChildDao.pageCarrierOrderChildList(param);
    }

    @Override
    public List<OrderChildPCVO> getOrderChildInfoByOrderGoodsNo(String orderGoodsNo) {
        List<OrderChild> orderChildren = orderChildDao.getOrderChildInfoByOrderGoodsNo(orderGoodsNo)
                .orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        return orderChildStruct.convertList(orderChildren);
    }

    @Override
    public CarrierOrderChildDetailVO getCarrierOrderChildDetail(String childNo) {
        //运单数据
        OrderChild orderChild = orderChildDao.getByChildNo(childNo).orElseThrow(PerformanceResultEnum.ORDER_CHILD_NO_FOUND);
        CarrierOrderChildDetailVO carrierOrderChildDetailVO = orderChildStruct.carrierConvert(orderChild);

        //货单数据
        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orElseThrow(PerformanceResultEnum.ORDER_NO_FOUND);
        carrierOrderChildDetailVO.setExtractWeight(orderGoods.getExtractWeight());

        //磅单审核数据
        List<OrderChildPoundAudit> poundAuditList = orderChildPoundAuditDao.getPoundAuditList(childNo);
        carrierOrderChildDetailVO.setPoundAudits(orderChildPoundStruct.convert(poundAuditList));

        //运单日志
        List<OrderChildLog> orderChildLog = orderChildLogService.getOrderChildNodeLog(childNo);
        carrierOrderChildDetailVO.setChildLogs(orderChildLogStruct.convert(orderChildLog));

        //应付运费
        BigDecimal payableFreight=BigDecimal.ZERO;
        if(Objects.nonNull(orderChild.getLoadNet())){
            payableFreight = orderChild.getLoadNet().multiply(carrierOrderChildDetailVO.getFreight());
        }

        //亏吨扣款（元）
        BigDecimal lossDeduction=BigDecimal.ZERO;
        if(Objects.nonNull(orderChild.getLoadNet()) && Objects.nonNull(orderChild.getUnloadNet())){
            if(orderChild.getLoadNet().compareTo(orderChild.getUnloadNet()) >0){
                lossDeduction = carrierOrderChildDetailVO.getFreight().multiply(orderChild.getLoadNet().subtract(orderChild.getUnloadNet()));
            }
        }

        //实付运费（元）
        BigDecimal realFreight=payableFreight.subtract(lossDeduction);
        carrierOrderChildDetailVO.setPayableFreight(payableFreight);
        carrierOrderChildDetailVO.setLossDeduction(lossDeduction);
        carrierOrderChildDetailVO.setRealFreight(realFreight);

        //磅单图片
        List<OrderChildImage> loadImages = orderChildImageDao.getImages(childNo, OrderChildImage.Type.LOAD.getCode());
        List<OrderChildImage> unloadImages = orderChildImageDao.getImages(childNo, OrderChildImage.Type.UNLOAD.getCode());
        carrierOrderChildDetailVO.setLoadImageList(loadImages.stream().map(OrderChildImage::getImage).collect(Collectors.toList()));
        carrierOrderChildDetailVO.setUnloadImageList(unloadImages.stream().map(OrderChildImage::getImage).collect(Collectors.toList()));
        return carrierOrderChildDetailVO;
    }
}
