package com.clx.performance.service.impl;

import com.clx.performance.dao.*;
import com.clx.performance.enums.OrderChildEnum;
import com.clx.performance.enums.integral.IntegralRecordEnum;
import com.clx.performance.model.*;
import com.clx.performance.param.mq.*;
import com.clx.performance.service.*;
import com.clx.performance.service.integral.IntegralRuleService;
import com.clx.performance.utils.LocalDateTimeUtils;
import com.clx.performance.vo.pc.integral.AppIntegralRuleVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@Service
public class IntegralMqHandlerServiceImpl implements IntegralMqHandlerService {

    @Autowired
    private IntegralStatisticsDao integralStatisticsDao;
    @Autowired
    private IntegralRecordDao integralRecordDao;
    @Autowired
    private IntegralTruckDao integralTruckDao;
    @Autowired
    private IntegralOrderDao integralOrderDao;
    @Autowired
    private OrderChildDao orderChildDao;
    @Autowired
    private OrderGoodsDao orderGoodsDao;

    @Autowired
    private IntegralStatisticsService integralStatisticsService;

    @Autowired
    private IntegralRecordService integralRecordService;

    @Autowired
    private IntegralRuleService integralRuleService;

    /**
     * 车辆新增
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void truckAdd(IntegralTuckAddMqParam mq){
        String truckNo = mq.getTruckNo();

        boolean existFlag = integralTruckDao.existByTruckNo(truckNo);
        if (existFlag) {return;}

        IntegralTruck truck = new IntegralTruck();
        truck.setTruckNo(truckNo);
        truck.setDriverName(mq.getDriverName());
        truck.setDriverMobile(mq.getDriverMobile());
        truck.setEchelon(1);

        LocalDateTime statisticsDate = LocalDateTimeUtils.getStartWeek();
        int protectionIntegral = integralStatisticsService.protectionIntegralCalc(statisticsDate);
        int baseIntegral = 100;

        // 初始化统计
        IntegralStatistics statistics = new IntegralStatistics();
        statistics.setTruckNo(truckNo);
        statistics.setStatisticsDate(statisticsDate);
        statistics.setIntegralBase(0);
        statistics.setIntegral(protectionIntegral+baseIntegral);   //保护分+基础分
        statistics.setRank(1);
        integralStatisticsDao.saveEntity(statistics);

        IntegralStatistics statisticsNext = new IntegralStatistics();
        statisticsNext.setTruckNo(truckNo);
        statisticsNext.setStatisticsDate(statistics.getStatisticsDate().plusDays(7));
        statisticsNext.setIntegralBase(0);
        statisticsNext.setIntegral(0);
        statisticsNext.setRank(1);
        integralStatisticsDao.saveEntity(statisticsNext);

        // 新增车辆
        truck.setIntegral(statistics.getIntegral());
        integralTruckDao.saveEntity(truck);

        // 新增积分记录
        integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecord.type.PROTECTION.getValue(), protectionIntegral, "新用户");
        integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecord.type.BASE.getValue(), baseIntegral, "基础分");

        // 实时积分统计
        integralRealTimeStatistics(statistics.getId());

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void orderGoodsAdd(IntegralOrderAddMqParam mq) {

        boolean existFlag = integralOrderDao.existByOrderGoodsNo(mq.getOrderGoodsNo());
        if (existFlag) {return;}

        IntegralOrder order = new IntegralOrder();
        order.setOrderGoodsNo(mq.getOrderGoodsNo());
        order.setSendAddressId(mq.getSendAddressId());
        order.setSendAddress(mq.getSendAddress());
        order.setReceiveAddressId(mq.getReceiveAddressId());
        order.setReceiveAddress(mq.getReceiveAddress());
        order.setOrderType(mq.getOrderType());
        order.setTruckEchelon(1);
        integralOrderDao.saveEntity(order);

    }

    /**
     * 积分记录新增
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void recordAdd(IntegralRecordAddMqParam mq){


    }

    /**
     * 下周车辆初始化（新增统计车辆） （周3、周7 晚上）
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void nextWeekTruckInit(IntegralNextWeekTruckInitMqParam mq){
        LocalDateTime statisticsDateCurrent = LocalDateTimeUtils.parseTime(mq.getStatisticsDate());

        LocalDateTime statisticsDateNext = statisticsDateCurrent.plusDays(7);

        // 查找本周车辆
        List<IntegralStatistics> list = integralStatisticsDao.selectByStatisticsDate(statisticsDateCurrent);

        // 查找下周车辆
        List<IntegralStatistics> nextList = integralStatisticsDao.selectByStatisticsDate(statisticsDateNext);
        List<String> truckNoNextList = nextList.stream().map(item -> item.getTruckNo()).collect(Collectors.toList());

        List<IntegralStatistics> newList = new ArrayList<>();
        for (IntegralStatistics item : list) {
            if (truckNoNextList.contains(item.getTruckNo())) {continue;}

            IntegralStatistics statistics = new IntegralStatistics();
            statistics.setTruckNo(item.getTruckNo());
            statistics.setStatisticsDate(statisticsDateNext);
            statistics.setIntegralBase(0);
            statistics.setIntegral(0);
            statistics.setRank(1);

            newList.add(statistics);
        }
        log.info("初始化下周车辆，size:{}",newList.size());

        if (newList.isEmpty()) {return;}

        // 新增下周车辆
        integralStatisticsDao.batchSaveEntity(newList);

    }

    /**
     * 车辆周统计 (统计周初始积分)
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void weekTruckStatistics(IntegralWeekTruckStatisticsMqParam mq){

        if (StringUtils.isNotBlank(mq.getTruckNo())){
            // 基础分初始化
            truckBaseIntegralInit(mq.getTruckNo(), mq.getStatisticsDate());
            return;
        }

        // 更新本周实时排名
        integralRealTimeRankStatistics(LocalDateTimeUtils.parseTime(mq.getStatisticsDate()));
    }

    /**
     * 新一周基础积分初始化
     */
    private void truckBaseIntegralInit(String truckNo, String statisticsDate){
        IntegralTruck truck = integralTruckDao.getByTruckNo(truckNo).orNull();
        if (null == truck) {return;}

        IntegralStatistics statistics = integralStatisticsDao.selectByTruckNoAndStatisticsDate(truck.getTruckNo(), statisticsDate).orNull();
        if (null == statistics) {return;}

        // 新车保护分
        newTruckProtection(truck, statistics);

        // 周结算分
        truckWeekSettlement(truck, statistics);

        integralStatisticsDao.updateIntegral(statistics);

        truck.setIntegral(statistics.getIntegral());
        integralTruckDao.updateIntegral(truck);

    }

    /**
     * 新车保护分
     */
    private void newTruckProtection(IntegralTruck truck, IntegralStatistics statistics){
        LocalDateTime statisticsDate = statistics.getStatisticsDate();

        // 超过保护期
        if (truck.getCreateTime().isBefore(statisticsDate.minusDays(7))) {return;}

        // 已保护不处理
        IntegralRecord record = integralRecordDao.selectByStatisticsIdAndType(statistics.getId(), IntegralRecord.type.PROTECTION.getValue());
        if (null != record) {return;}

        // 计算新车积分
        int integral = integralStatisticsService.protectionIntegralCalc(truck.getCreateTime(), statisticsDate);
        if (integral == 0) {return;}

        statistics.setIntegral(statistics.getIntegral()+integral);
        integralStatisticsDao.updateIntegral(statistics);

        integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecord.type.PROTECTION.getValue(), integral, "新用户");

    }

    /**
     * 上周结算分
     */
    private void truckWeekSettlement(IntegralTruck truck, IntegralStatistics statistics){
        int integral;

        // 已结算不处理
        IntegralRecord record = integralRecordDao.selectByStatisticsIdAndType(statistics.getId(), IntegralRecord.type.SETTLEMENT.getValue());
        if (null != record) {return;}

        String statisticsDatePre = LocalDateTimeUtils.formatTime(statistics.getStatisticsDate().minusDays(7));

        // 查找上周统计数据
        IntegralStatistics statisticsPre = integralStatisticsDao.selectByTruckNoAndStatisticsDate(truck.getTruckNo(), statisticsDatePre).orNull();
        if (null == statisticsPre) {return;}

        // 查找总车数
        int truckSum = integralStatisticsDao.countByStatisticsDate(statisticsDatePre);

        if (statisticsPre.getRank() <= truckSum*0.1){
            integral = 300;
        }
        else if ( (statisticsPre.getRank() > truckSum*0.1) && (statisticsPre.getRank() <= truckSum*0.3)){
            integral = 200;
        }
        else if ( (statisticsPre.getRank() > truckSum*0.3) && (statisticsPre.getRank() <= truckSum*0.6)){
            integral = 100;
        }
        else if ( (statisticsPre.getRank() > truckSum*0.6) && (statisticsPre.getRank() <= truckSum*0.9)){
            integral = 60;
        }
        else {
            integral = 0;
        }
        if (integral == 0) {return;}

        statistics.setIntegral(statistics.getIntegral()+integral);
        integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecord.type.SETTLEMENT.getValue(), integral, "上周结算");

    }

    /**
     * 新手保护取消
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void truckProtectionCancel(IntegralTruckProtectionCancelMqParam mq){
        IntegralTruck truck = integralTruckDao.getByTruckNo(mq.getTruckNo()).orNull();
        if (null == truck) {return;}

        IntegralStatistics statistics = integralStatisticsDao.selectByTruckNoAndStatisticsDate(truck.getTruckNo(), mq.getStatisticsDate()).orNull();
        if (null == statistics) {return;}

        IntegralRecord record = integralRecordDao.selectByStatisticsIdAndType(statistics.getId(), IntegralRecord.type.PROTECTION.getValue());
        IntegralRecord recordCancel = integralRecordDao.selectByStatisticsIdAndType(statistics.getId(), IntegralRecord.type.PROTECTION_CANCEL.getValue());
        if (null != recordCancel) {return;}
        int integral = -record.getIntegral();

        integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecord.type.PROTECTION_CANCEL.getValue(), integral, "新用户过保护期");

        statistics.setIntegral(statistics.getIntegral()+integral);
        integralStatisticsDao.updateIntegral(statistics);

        truck.setIntegral(statistics.getIntegral());
        integralTruckDao.updateIntegral(truck);

        // 更新本周实时排名
        integralRealTimeRankStatistics(LocalDateTimeUtils.parseTime(mq.getStatisticsDate()));

    }

    /**
     * 运单结束
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void orderChildFinish(IntegralOrderChildFinishMqParam mq) {
        String statisticsDate = LocalDateTimeUtils.getWeekStart();

        // 查找规则
        AppIntegralRuleVO ruleInfo = integralRuleService.getRuleInfo();

        OrderChild orderChild = orderChildDao.getByChildNo(mq.getChildNo()).orNull();
        if (null == orderChild) {return;}

        OrderGoods orderGoods = orderGoodsDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orNull();
        if (null == orderGoods) {return;}

        IntegralOrder integralOrder = integralOrderDao.getByOrderGoodsNo(orderChild.getOrderGoodsNo()).orNull();
        if (null == integralOrder){return;}

        IntegralStatistics statistics = integralStatisticsDao.selectByTruckNoAndStatisticsDate(orderChild.getTruckNo(), statisticsDate).orNull();
        if (null == statistics) {return;}

        IntegralTruck truck = integralTruckDao.getByTruckNo(orderChild.getTruckNo()).orNull();
        if (null == truck) {return;}

        List<Integer> finishStatusList = Arrays.asList(OrderChildEnum.Status.UNSETTLE.getCode(), OrderChildEnum.Status.COMPLETE.getCode());
        List<Integer> cancelStatusList = Arrays.asList(OrderChildEnum.Status.DRIVER_CANCEL.getCode());;
        List<Integer> platformCancelStatusList = Arrays.asList(
                OrderChildEnum.Status.OWNER_CANCEL.getCode(),
                OrderChildEnum.Status.PLATFORM_CANCEL.getCode(),
                OrderChildEnum.Status.ARTIFICIAL_CANCEL.getCode()
        );

        int integral = 0;
        if (finishStatusList.contains(orderChild.getStatus())){
            // 完成
            integral = orderChildCompleteIntegralSettlement(statistics, ruleInfo, integralOrder, orderChild, orderGoods);
        }
        else if (cancelStatusList.contains(orderChild.getStatus())){
            // 取消
            integral = orderChildCancelIntegralSettlement(statistics, ruleInfo, integralOrder, orderChild, orderGoods);
        }
        else if (platformCancelStatusList.contains(orderChild.getStatus())){
            // 平台取消
            orderChildPlatformCancelIntegralSettlement(statistics, ruleInfo, integralOrder, orderChild, orderGoods);
        }

        log.info("司机积分, 运单:{}, 累计积分:{}", orderChild.getChildNo(), integral);
        if (integral == 0) {return;}

        // 更新积分
        statistics.setIntegral(statistics.getIntegral()+integral);
        integralStatisticsDao.updateIntegral(statistics);

        truck.setIntegral(statistics.getIntegral());
        integralTruckDao.updateIntegral(truck);

        // 实时积分统计
        integralRealTimeStatistics(statistics.getId());
    }

    /**
     * 运单完成
     */
    private int orderChildCompleteIntegralSettlement(IntegralStatistics statistics, AppIntegralRuleVO ruleInfo, IntegralOrder integralOrder, OrderChild orderChild, OrderGoods orderGoods){

        LocalDateTime dateTimeDayStart = LocalDateTimeUtils.getDayStart();
        LocalDateTime dateTimeDayNext = dateTimeDayStart.plusDays(1);

        String dayStartTime = LocalDateTimeUtils.formatTime(dateTimeDayStart);
        String nextDayStartTime = LocalDateTimeUtils.formatTime(dateTimeDayNext);

        int integralSum = 0;

        // 运单完成积分
        int completeIntegral = 0;
//        if (completeIntegral > 0){
//            Integer sum = integralRecordDao.sumByStatisticsIdAndTypeAndTime(statistics.getId(), IntegralRecord.type.ORDER_CHILD_COMPLETE.getValue(), dayStartTime, nextDayStartTime);
//            if (null == sum) {sum = 0;}
//            if (sum+completeIntegral > 30) {completeIntegral = 0;}
//        }
//        if (completeIntegral > 0){
//            String remark="";
//            if (IntegralOrder.OrderType.HIGH_QUALITY.getValue().equals(integralOrder.getOrderType())){remark = "完成优质单";}
//            else if (IntegralOrder.OrderType.NORMAL.getValue().equals(integralOrder.getOrderType())){remark = "完成普通单";}
//            else if (IntegralOrder.OrderType.WATER.getValue().equals(integralOrder.getOrderType())){remark = "完成流水单";}
//            else if (IntegralOrder.OrderType.SUPPORT.getValue().equals(integralOrder.getOrderType())){remark = "完成支援单";}
//            integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecord.type.ORDER_CHILD_COMPLETE.getValue(), completeIntegral, remark);
//        }

        // 完成量积分
        int completeNumIntegral = 0;
        String completeNumIntegralRemark="";
        List<String> enableOrderNoList = integralOrderDao.getAllOrderNoList();
        int finishNum = orderChildDao.countCompleteByTruckNoAndPayFinishTime(orderChild.getTruckNo(),
                LocalDateTimeUtils.formatTime(dateTimeDayStart),
                LocalDateTimeUtils.formatTime(dateTimeDayNext),
                LocalDateTimeUtils.formatTime(orderChild.getConfirmTime()),
                enableOrderNoList);
        log.info("司机积分，车辆完成量，truckNo:{}, sum:{}", orderChild.getChildNo(), finishNum);

        if (finishNum == 1){
            completeNumIntegral = ruleInfo.getOrderChildCompleteRule().getInteger1();
            completeNumIntegralRemark = "运单达量1单";
        }
        else if (finishNum == 2){
            completeNumIntegral = ruleInfo.getOrderChildCompleteRule().getInteger2();
            completeNumIntegralRemark = "运单达量2单";
        }
        else if (finishNum == 3){
            completeNumIntegral = ruleInfo.getOrderChildCompleteRule().getInteger3();
            completeNumIntegralRemark = "运单达量3单";
        }
        else if (finishNum == 4){
            completeNumIntegral = ruleInfo.getOrderChildCompleteRule().getInteger4();
            completeNumIntegralRemark = "运单达量4单";
        }
        else if (finishNum > 4){
            completeNumIntegral = 0;
        }
        if (completeNumIntegral > 0){
            integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecordEnum.Type.ORDER_CHILD_COMPLETE_NUM.getValue(), completeNumIntegral, completeNumIntegralRemark);
        }

        // 奖励积分（低价）(下期启用)
        int waterOrderIntegral = 0;
//        if (IntegralOrder.OrderType.WATER.getValue().equals(integralOrder.getOrderType())){
//            waterOrderIntegral = ruleInfo.getOrderAwardRule().getIntegerWater();
//        }
//        if (waterOrderIntegral > 0){
//            Integer sum = integralRecordDao.countByStatisticsIdAndTypeAndTime(statistics.getId(), IntegralRecord.type.ORDER_CHILD_COMPLETE_WATER.getValue(), dayStartTime, nextDayStartTime);
//            if (null == sum) {sum = 0;}
//            if ((sum + 1) > ruleInfo.getOrderAwardRule().getCountWaterLimit()) {waterOrderIntegral = 0;}
//        }
//        if (waterOrderIntegral > 0){
//            integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecord.type.ORDER_CHILD_COMPLETE_WATER.getValue(), waterOrderIntegral, "流水单奖励");
//        }

        // 奖励积分(支持单)(下期启用)
        int supportOrderIntegral = 0;
//        if (IntegralOrder.OrderType.SUPPORT.getValue().equals(integralOrder.getOrderType())){
//            supportOrderIntegral = ruleInfo.getOrderAwardRule().getIntegerSupport();
//        }
//        if (supportOrderIntegral > 0){
//            Integer sum = integralRecordDao.countByStatisticsIdAndTypeAndTime(statistics.getId(), IntegralRecord.type.ORDER_CHILD_COMPLETE_SUPPORT.getValue(), dayStartTime, nextDayStartTime);
//            if (null == sum) {sum = 0;}
//            if ((sum + 1) > ruleInfo.getOrderAwardRule().getCountSupportLimit()) {supportOrderIntegral = 0;}
//        }
//        if (supportOrderIntegral > 0){
//            integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecord.type.ORDER_CHILD_COMPLETE_SUPPORT.getValue(), supportOrderIntegral, "支援单奖励");
//        }

        // 到达货源地目的地超时
        if (null == orderGoods.getLastArriveSendTime() && null == orderGoods.getLastArriveReceiveTime()){
            integralSum = completeIntegral+completeNumIntegral+waterOrderIntegral+supportOrderIntegral;

            return integralSum;
        }

        LocalDateTime lastArriveSendTime = orderGoods.getLastArriveSendTime();
        LocalDateTime lastArriveReceiveTime = orderGoods.getLastArriveReceiveTime();

        int sendAddressTimeoutIntegral = 0;
        if (null != lastArriveSendTime && orderChild.getArriveSendTime().isAfter(lastArriveSendTime)){
            sendAddressTimeoutIntegral = -ruleInfo.getOrderChildTimeoutRule().getIntegralSendAddressTimeout();
        }
        if (sendAddressTimeoutIntegral < 0){
            integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecordEnum.Type.ORDER_CHILD_COMPLETE_TIMEOUT.getValue(), sendAddressTimeoutIntegral, "未按时到达货源地减分");
        }

        int receiveAddressTimeoutIntegral = 0;
        if (null != lastArriveReceiveTime && orderChild.getArriveReceiveTime().isAfter(lastArriveReceiveTime)){
            receiveAddressTimeoutIntegral = -ruleInfo.getOrderChildTimeoutRule().getIntegralSendAddressTimeout();
        }
        if (receiveAddressTimeoutIntegral < 0){
            integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecordEnum.Type.ORDER_CHILD_COMPLETE_TIMEOUT.getValue(), receiveAddressTimeoutIntegral, "未按时到达目的地减分");
        }

        // 装卸车超时（下期开发）

        integralSum = completeIntegral+completeNumIntegral+waterOrderIntegral+supportOrderIntegral+sendAddressTimeoutIntegral+receiveAddressTimeoutIntegral;

        return integralSum;
    }

    /**
     * 运单取消
     */
    private int orderChildCancelIntegralSettlement(IntegralStatistics statistics, AppIntegralRuleVO ruleInfo, IntegralOrder integralOrder, OrderChild orderChild, OrderGoods orderGoods){

        int integralSum = 0;

        // 运单取消积分
        Integer cancelIntegral = -ruleInfo.getOrderChildCancelRule().getIntegralCancel();
        String cancelIntegralRemark="取消运单";

        //(下期启用)
//        if (IntegralOrder.OrderType.HIGH_QUALITY.getValue().equals(integralOrder.getOrderType())){
//
//            cancelIntegralRemark = "取消优质单";
//        }
//        else if (IntegralOrder.OrderType.NORMAL.getValue().equals(integralOrder.getOrderType())){
//
//            cancelIntegralRemark = "取消普通单";
//        }
//        else if (IntegralOrder.OrderType.WATER.getValue().equals(integralOrder.getOrderType())){
//            cancelIntegral -= ruleInfo.getOrderChildCancelRule().getIntegralWaterCancel();
//            cancelIntegralRemark = "取消流水单";
//        }
//        else if (IntegralOrder.OrderType.SUPPORT.getValue().equals(integralOrder.getOrderType())){
//            cancelIntegral -= ruleInfo.getOrderChildCancelRule().getIntegralSupportCancel();
//            cancelIntegralRemark = "取消支援单";
//        }

        if (cancelIntegral < 0){
            integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecordEnum.Type.ORDER_CHILD_CANCEL.getValue(), cancelIntegral, cancelIntegralRemark);
        }


        // 超时
        if (null == orderGoods.getLastArriveSendTime() && null == orderGoods.getLastArriveReceiveTime()){
            integralSum = cancelIntegral;
            return integralSum;
        }

        LocalDateTime lastArriveSendTime = orderGoods.getLastArriveSendTime();
        LocalDateTime lastArriveReceiveTime = orderGoods.getLastArriveReceiveTime();

        int sendAddressTimeoutIntegral = 0;
        if (null != lastArriveSendTime && null != orderChild.getArriveSendTime() && orderChild.getArriveSendTime().isAfter(lastArriveSendTime)){
            sendAddressTimeoutIntegral = -ruleInfo.getOrderChildTimeoutRule().getIntegralSendAddressTimeout();
        }
        else if (null != lastArriveSendTime && null == orderChild.getArriveSendTime() && orderChild.getFinishTime().isAfter(lastArriveSendTime)){
            sendAddressTimeoutIntegral = -ruleInfo.getOrderChildTimeoutRule().getIntegralSendAddressTimeout();
        }
        if (sendAddressTimeoutIntegral < 0){
            integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecordEnum.Type.ORDER_CHILD_CANCEL_TIMEOUT.getValue(), sendAddressTimeoutIntegral, "未按时到达货源地减分");
        }

        int receiveAddressTimeoutIntegral = 0;
        if (null != lastArriveReceiveTime && null != orderChild.getArriveReceiveTime() && orderChild.getArriveReceiveTime().isAfter(lastArriveReceiveTime)){
            receiveAddressTimeoutIntegral = -ruleInfo.getOrderChildTimeoutRule().getIntegralReceiveAddressTimeout();
        }
        else if (null != lastArriveReceiveTime && null != orderChild.getLoadTime() && orderChild.getFinishTime().isAfter(lastArriveReceiveTime)){
            receiveAddressTimeoutIntegral = -ruleInfo.getOrderChildTimeoutRule().getIntegralReceiveAddressTimeout();
        }

        if (receiveAddressTimeoutIntegral < 0){
            integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecordEnum.Type.ORDER_CHILD_CANCEL_TIMEOUT.getValue(), receiveAddressTimeoutIntegral, "未按时到达目的地减分");
        }

        // 装卸车超时（下期开发）

        integralSum = cancelIntegral+sendAddressTimeoutIntegral+receiveAddressTimeoutIntegral;

        return integralSum;

    }

    /**
     * 平台运单取消
     */
    private int orderChildPlatformCancelIntegralSettlement(IntegralStatistics statistics, AppIntegralRuleVO ruleInfo, IntegralOrder integralOrder, OrderChild orderChild, OrderGoods orderGoods){

        int integralSum = 0;

        int cancelIntegral = 0;

        LocalDateTime lastArriveSendTime = orderGoods.getLastArriveSendTime();
        LocalDateTime lastArriveReceiveTime = orderGoods.getLastArriveReceiveTime();

        int sendAddressTimeoutIntegral = 0;
        if (null != lastArriveSendTime && null != orderChild.getArriveSendTime() && orderChild.getArriveSendTime().isAfter(lastArriveSendTime)){
            sendAddressTimeoutIntegral = -ruleInfo.getOrderChildTimeoutRule().getIntegralSendAddressTimeout();
        }
        else if (null != lastArriveSendTime && null == orderChild.getArriveSendTime() && orderChild.getFinishTime().isAfter(lastArriveSendTime)){
            sendAddressTimeoutIntegral = -ruleInfo.getOrderChildTimeoutRule().getIntegralSendAddressTimeout();
        }
        if (sendAddressTimeoutIntegral < 0){
            integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecordEnum.Type.ORDER_CHILD_CANCEL_TIMEOUT.getValue(), sendAddressTimeoutIntegral, "未按时到达货源地减分");
        }

        int receiveAddressTimeoutIntegral = 0;
        if (null != lastArriveReceiveTime && null != orderChild.getArriveReceiveTime() && orderChild.getArriveReceiveTime().isAfter(lastArriveReceiveTime)){
            receiveAddressTimeoutIntegral = -ruleInfo.getOrderChildTimeoutRule().getIntegralReceiveAddressTimeout();
        }
        else if (null != lastArriveReceiveTime && null != orderChild.getLoadTime() && orderChild.getFinishTime().isAfter(lastArriveReceiveTime)){
            receiveAddressTimeoutIntegral = -ruleInfo.getOrderChildTimeoutRule().getIntegralReceiveAddressTimeout();
        }

        if (receiveAddressTimeoutIntegral < 0){
            integralRecordService.save(statistics.getId(), statistics.getStatisticsDate(), statistics.getTruckNo(), IntegralRecordEnum.Type.ORDER_CHILD_CANCEL_TIMEOUT.getValue(), receiveAddressTimeoutIntegral, "未按时到达目的地减分");
        }

        // 装卸车超时（下期开发）


        integralSum = cancelIntegral+sendAddressTimeoutIntegral+receiveAddressTimeoutIntegral;


        return integralSum;

    }

    /**
     * 更新实时积分
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void integralRealTimeStatistics(Integer statisticsId){
        IntegralStatistics statistics = integralStatisticsDao.selectById(statisticsId).orNull();
        if (null == statistics) {return;}

        Integer integral = integralRecordDao.sumIntegral(statisticsId);
        if (null == integral){return;}

        statistics.setIntegral(integral);

        // 更新车辆总积分
        integralStatisticsDao.updateIntegral(statistics);

        // 更新车辆积分
        integralTruckDao.updateIntegralByTruckNo(statistics.getTruckNo(), statistics.getIntegral());

        // 更新本周实时排名
        integralRealTimeRankStatistics(statistics.getStatisticsDate());
    }

    /**
     * 更新实时排名
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void integralRealTimeRankStatistics(LocalDateTime statisticsDate){

        List<IntegralStatistics> list = integralStatisticsDao.selectByStatisticsDate(statisticsDate);

        list.get(0).setRank(1);
        for (int i=1; i<list.size(); i++){
            if (list.get(i).getIntegral().equals(list.get(i-1).getIntegral())){list.get(i).setRank(list.get(i-1).getRank());}
            else {list.get(i).setRank(i+1);};
        }

        integralStatisticsDao.batchUpdateRank(list);

    }

    /**
     * 时间差
     */
    private Integer diffSecond(LocalDateTime startTime, LocalDateTime endTime){
        Duration duration = Duration.between(startTime, endTime);
        return Long.valueOf(duration.getSeconds()).intValue();
    }

}
